/*
 * commands.c
 *
 *  Created on: Sep 5, 2018
 *      Author: root
 */

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "commands.h"
#include "platform.h"
#include "poe.h"
#include "inc/hw_memmap.h"
#include "adcUtil.h"
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include "driverlib/gpio.h"


//command level errors
const char *NAK_BadArgCount = "Incorrect number of arguments %d, expected %d.";
const char *NAK_BadArg = "Invalid argument '%s' in argument at index %d, expected %s.";
const char *NAK_BadArg2 = "Invalid argument '%s' at index %d.";
const char *NAK_BadArgRange = "Argument '%ld' out of range in argument at index %d, expected %ld - %ld.";
const char *NAK_BadArgLength = "Incorrect argument length %d in argument at index %d, expected %d.";
const char *NAK_BadArgChar = "Illegal character '%c' at index %d in argument at index %d, expected %s.";
const char *NAK_BadDigitalIo = "I/O channel at index %d not configured for output.";
const char *NAK_NotSupported = "I/O channel at index %d does not support %c setting.";

static long temp[18];
static char bigbuff[256];
static char intbuff[32];


int TryParseLong(const char *value, long *result)
{
	errno = 0;
	char * endptr = 0;

	*result = strtol(value, &endptr, 10);

	return ! (endptr == value || (*result == ULONG_MAX && errno == ERANGE));
}



int HandleIdAndVersion(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%s %s", GetPodId(), GetFirmwareVersion());
    return CMDSUCCESS;
}

int HandleReset(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    SetReset();

    send_buffer[0] = 0;

    return CMDSUCCESS;
}

int HandleSwitch(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%d", READ_MYADDR);

    return CMDSUCCESS;
}

int Handle24V(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }
    sprintf(send_buffer, "%4d %4d", adcResults[18], adcResults[19]);

    return CMDSUCCESS;
}


int HandleSetIoConfig(char** tokens, int nTokens, char* send_buffer)
{

	int nArgs = nTokens - 1;

	if (nArgs != 1)
	{
		sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
		return CMDFAILURE;
	}

	int ioStrLen = strlen(tokens[1]);

	if (NUM_IO_PORTS != ioStrLen)
	{
		sprintf(send_buffer, NAK_BadArgLength, ioStrLen, 0, NUM_IO_PORTS);
		return CMDFAILURE;
	}

	int i=0; for (i=0; i < NUM_IO_PORTS; i++)
	{
		char c = tokens[1][i];

		if (c != 'i' && c != 'o'  && c != '-')
		{
			sprintf(send_buffer, NAK_BadArgChar, c, i, 0, "[i|o|-]");
			return CMDFAILURE;
		}
	}

	for (i=0; i < NUM_IO_PORTS; i++)
	{
		SetIoConfig(i, tokens[1][i]);
	}

	send_buffer[0] = 0;

	return CMDSUCCESS;
}

int HandleGetIoConfig(char** tokens, int nTokens, char* send_buffer)
{
	int nArgs = nTokens - 1;

	if (nArgs != 0)
	{
		sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
		return CMDFAILURE;
	}

	sprintf(send_buffer, "%s", GetIoConfigAll());
	return CMDSUCCESS;
}

int HandleSetOutputMode(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;
    int mode[4];

    if (nArgs != 4)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 4);
        return CMDFAILURE;
    }

    int i=1; for (i = 1; i <= NUM_IO_GROUPS; i++)
    {
        char *c = &tokens[i][0];

        if (strcmp(c, "solenoid") == 0)
            mode[i - 1] = c[0];
        else if (strcmp(c, "led") == 0)
            mode[i - 1] = c[0];
        else
        {
            sprintf(send_buffer, NAK_BadArg2, c, i);
            return CMDFAILURE;
        }
    }

    for (i=0; i < NUM_IO_GROUPS; i++)
    {
        SetOutputMode(i, mode[i]);
    }

    send_buffer[0] = 0;

    return CMDSUCCESS;
}

int HandleGetOutputMode(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    int i;
    char mode;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    send_buffer[0] = 0;
    for (i = 0; i < NUM_IO_GROUPS; i++)
    {
        mode = GetOutputMode(i);
        if (mode == 'l')
            sprintf(send_buffer, "%s led", send_buffer);
        else
            sprintf(send_buffer, "%s solenoid", send_buffer);
    }

    return CMDSUCCESS;
}

int HandleSetInputImpedance(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    int ioStrLen = strlen(tokens[1]);

    if (NUM_IO_PORTS != ioStrLen)
    {
        sprintf(send_buffer, NAK_BadArgLength, ioStrLen, 0, NUM_IO_PORTS);
        return CMDFAILURE;
    }

    // check for parameter error
    int i=0; for (i=0; i < NUM_IO_PORTS; i++)
    {
        char c = tokens[1][i];

        if (c != 'u' && c != 'd'  && c != 'z'  && c != '-')
        {
            sprintf(send_buffer, NAK_BadArgChar, c, i, 0, "[u|d|z|-]");
            return CMDFAILURE;
        }

        if (i >= 16 && tokens[1][i] == 'd')
        {
            sprintf(send_buffer, NAK_NotSupported, i, c);
            return CMDFAILURE;
        }
    }

    for (i=0; i < NUM_IO_PORTS; i++)
    {
        if (tokens[1][i] != '-')
            SetInputImpedance(i, tokens[1][i]);
    }

    send_buffer[0] = 0;

    return CMDSUCCESS;
}

int HandleGetInputImpedance(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%s", GetInputImpedanceAll());
    return CMDSUCCESS;
}

int HandleSetDigitalOutput(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 2)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 2);
        return CMDFAILURE;
    }

    long temp;

    char * value = args[0];

    if (!TryParseLong(value, &temp))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (temp < DATA_DIGITAL_CHAN_MIN || temp > DATA_DIGITAL_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, temp, 0, DATA_DIGITAL_CHAN_MIN, DATA_DIGITAL_CHAN_MAX);
        return CMDFAILURE;
    }

    // verify channel is set to outout
    if (!ChannelIsOutput(temp))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, temp);
        return CMDFAILURE;
    }
    char c = tokens[2][0];

    if (c != '1' && c != '0')
    {
        sprintf(send_buffer, NAK_BadArgChar, c, 0, 2, "[1|0]");
        return CMDFAILURE;
    }

    SetDigitalOutput(temp, (c == '1'));

    UpdateDigitalOutput();

    send_buffer[0] = 0;
    return CMDSUCCESS;
}

int HandleGetDigitalOutput(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    long temp;

    char * value = args[0];

    if (!TryParseLong(value, &temp))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (temp < DATA_DIGITAL_CHAN_MIN || temp > DATA_DIGITAL_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, temp, 0, DATA_DIGITAL_CHAN_MIN, DATA_DIGITAL_CHAN_MAX);
        return CMDFAILURE;
    }


    sprintf(send_buffer, "%d", GetDigitalOutput(temp));
    return CMDSUCCESS;
}

int HandleGetDigitalAnalog(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    long temp;

    char * value = args[0];

    if (!TryParseLong(value, &temp))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (temp < DATA_DIGITAL_CHAN_MIN || temp > DATA_DIGITAL_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, temp, 0, DATA_DIGITAL_CHAN_MIN, DATA_DIGITAL_CHAN_MAX);
        return CMDFAILURE;
    }


    sprintf(send_buffer, "%d", GetAnalogInput(temp));
    return CMDSUCCESS;
}

int HandleSetDigitalOutputAll(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    int ioStrLen = strlen(tokens[1]);

    // check string is correct length
    if (NUM_DIGITAL_PORTS != ioStrLen)
    {
        sprintf(send_buffer, NAK_BadArgLength, ioStrLen, 0, NUM_DIGITAL_PORTS);
        return CMDFAILURE;
    }

    // check string contains only valid chars 1,0,-
    int i=0; for (i=0; i < NUM_DIGITAL_PORTS; i++)
    {
        char c = tokens[1][i];

        if (c != '1' && c != '0'  && c != '-')
        {
            sprintf(send_buffer, NAK_BadArgChar, c, i, 0, "[1|0|-]");
            return CMDFAILURE;
        }
    }

    // check all channels being set are configured for output
    for (i=0; i < NUM_DIGITAL_PORTS; i++)
    {
        char c = tokens[1][i];

        if (c != '-' && GetIoConfig(i) == '-')
        {
            sprintf(send_buffer, NAK_BadDigitalIo, i);
            return CMDFAILURE;
        }
    }

    //unsigned long

    //
    for (i=0; i < NUM_DIGITAL_PORTS; i++)
    {
        char c = tokens[1][i];

        if (c != '-')
            SetDigitalOutput(i, (c == '1'));
    }

    UpdateDigitalOutput();

    send_buffer[0] = 0;
    return CMDSUCCESS;
}

int HandleGetDigitalOutputAll(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%s", GetDigitalOutputAll());
    return CMDSUCCESS;
}

int HandleGetDigitalAnalogAll(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    int i;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    memset(bigbuff, 0, 256);

    for (i = 0; i < NUM_DIGITAL_PORTS; i++)
    {
        int temp = GetAnalogInput(i);
        sprintf(intbuff, (i == 0) ? "%4d" : " %4d", temp);
        strcat(bigbuff, intbuff);
    }

    sprintf(send_buffer, "%s", bigbuff);
    return CMDSUCCESS;
}

int HandleSetAnalogThresholds(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != NUM_ADC_PORTS)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, NUM_ADC_PORTS);
        return CMDFAILURE;
    }

    int i=0; for (i=0; i<NUM_ADC_PORTS; i++)
    {
        char * value = args[i];

        if (0 == strcmp("-", value))
        {
            temp[i] = -1;
        }
        else
        {
            if (!TryParseLong(value, &temp[i]))
            {
                sprintf(send_buffer, NAK_BadArg, value, i, "decimal integer or '-'");
                return CMDFAILURE;
            }
            if (temp[i] < ANALOG_THRESHOLD_MIN || temp[i] > ANALOG_THRESHOLD_MAX)
            {
                sprintf(send_buffer, NAK_BadArgRange, temp[i], i, ANALOG_THRESHOLD_MIN, ANALOG_THRESHOLD_MAX);
                return CMDFAILURE;
            }
        }
    }
    for (i=0; i<NUM_ADC_PORTS; i++)
    {
        SetAnalogThreshold(i, temp[i]);
    }

    send_buffer[0] = 0;
    return CMDSUCCESS;
}

int HandleGetAnalogThresholds(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    int i;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    memset(bigbuff, 0, 256);

    for (i = 0; i < NUM_ADC_PORTS; i++)
    {
        int temp = GetAnalogThreshold(i);
        if (temp == -1)
        {
            strcat(bigbuff, (i == 0) ? "-" : " -");

        }
        else
        {
            sprintf(intbuff, (i == 0) ? "%4d" : " %4d", temp);
            strcat(bigbuff, intbuff);
        }
    }

    sprintf(send_buffer, "%s", bigbuff);
    return CMDSUCCESS;
}

int HandleSetAnalogOutput(char** tokens, int nTokens, char* send_buffer)
{

    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 2)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    long channel;

    char * value = args[0];

    if (value)
    if (!TryParseLong(value, &channel))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (channel < DATA_ANALOG_CHAN_MIN || channel > DATA_ANALOG_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, channel, 0, DATA_ANALOG_CHAN_MIN, DATA_ANALOG_CHAN_MAX);
        return CMDFAILURE;
    }

    // verify channel is set to outout
    if (!ChannelIsOutput(channel))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, channel);
        return CMDFAILURE;
    }

    long output;

    value = args[1];

    if (!TryParseLong(value, &output))
    {
        sprintf(send_buffer, NAK_BadArg, value, 1, "decimal integer or '-'");
        return CMDFAILURE;
    }
    if (output < ANALOG_THRESHOLD_MIN || output > ANALOG_THRESHOLD_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, output, 1, ANALOG_THRESHOLD_MIN, ANALOG_THRESHOLD_MAX);
        return CMDFAILURE;
    }

    SetAnalogOutput(channel, output);

    send_buffer[0] = 0;

    return CMDSUCCESS;
}

int HandleGetAnalogOutput(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    long channel;

    char * value = args[0];

    if (!TryParseLong(value, &channel))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (channel < DATA_ANALOG_CHAN_MIN || channel > DATA_ANALOG_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, channel, 0, DATA_ANALOG_CHAN_MIN, DATA_ANALOG_CHAN_MAX);
        return CMDFAILURE;
    }

    // verify channel is set to outout
    if (!ChannelIsOutput(channel))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, channel);
        return CMDFAILURE;
    }

    int temp = GetAnalogOutput(channel);

    sprintf(send_buffer, "%4d", temp);
    return CMDSUCCESS;
}

int HandleSetAnalogOutputAll(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 2)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    // verify channel 16 is set to outout
    if (!ChannelIsOutput(16))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, 16);
        return CMDFAILURE;
    }

    // verify channel 17 is set to outout
    if (!ChannelIsOutput(17))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, 17);
        return CMDFAILURE;
    }

    long output16;

    char *value = args[0];

    if (strcmp("-", value) != 0) {
        if (!TryParseLong(value, &output16))
        {
            sprintf(send_buffer, NAK_BadArg, value, 1, "decimal integer or '-'");
            return CMDFAILURE;
        }
        if (output16 < ANALOG_THRESHOLD_MIN || output16 > ANALOG_THRESHOLD_MAX)
        {
            sprintf(send_buffer, NAK_BadArgRange, output16, 1, ANALOG_THRESHOLD_MIN, ANALOG_THRESHOLD_MAX);
            return CMDFAILURE;
        }
        SetAnalogOutput(16, output16);
    }

    long output17;

    value = args[1];

    if (strcmp("-", value) != 0) {
        if (!TryParseLong(value, &output17))
        {
            sprintf(send_buffer, NAK_BadArg, value, 1, "decimal integer or '-'");
            return CMDFAILURE;
        }
        if (output17 < ANALOG_THRESHOLD_MIN || output17 > ANALOG_THRESHOLD_MAX)
        {
            sprintf(send_buffer, NAK_BadArgRange, output17, 1, ANALOG_THRESHOLD_MIN, ANALOG_THRESHOLD_MAX);
            return CMDFAILURE;
        }

        SetAnalogOutput(17, output17);
    }

    send_buffer[0] = 0;

    return CMDSUCCESS;
}

int HandleGetAnalogOutputAll(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    // verify channel 16 is set to outout
    if (!ChannelIsOutput(16))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, 16);
        return CMDFAILURE;
    }

    // verify channel 17 is set to outout
    if (!ChannelIsOutput(17))
    {
        sprintf(send_buffer, NAK_BadDigitalIo, 17);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%4d %4d", GetAnalogOutput(16), GetAnalogOutput(17));
    return CMDSUCCESS;
}

int HandleGetAnalogInput(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    long channel;

    char * value = args[0];

    if (!TryParseLong(value, &channel))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (channel < DATA_ANALOG_CHAN_MIN || channel > DATA_ANALOG_CHAN_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, channel, 0, DATA_ANALOG_CHAN_MIN, DATA_ANALOG_CHAN_MAX);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%4d", GetAnalogInput(channel));
    return CMDSUCCESS;
}

int HandleGetAnalogInputAll(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    sprintf(send_buffer, "%4d %4d", GetAnalogInput(16), GetAnalogInput(17));
    return CMDSUCCESS;
}

int HandleSetAnalogFilterCoef(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;
    char **args = tokens + 1;

    if (nArgs != 1)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
        return CMDFAILURE;
    }

    long temp;

    char * value = args[0];

    if (!TryParseLong(value, &temp))
    {
        sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
        return CMDFAILURE;
    }

    if (temp < DATA_STREAM_DELAY_MIN || temp > DATA_STREAM_DELAY_MAX)
    {
        sprintf(send_buffer, NAK_BadArgRange, temp, 0, DATA_STREAM_DELAY_MIN, DATA_STREAM_DELAY_MAX);
        return CMDFAILURE;
    }

    SetAnalogFilterCoef(temp);
    //sprintf(send_buffer, "ACK\n");
    send_buffer[0] = 0;
    return CMDSUCCESS;
}

int HandleGetAnalogFilterCoef(char** tokens, int nTokens, char* send_buffer)
{
    int nArgs = nTokens - 1;

    if (nArgs != 0)
    {
        sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
        return CMDFAILURE;
    }

    char intbuff[32];

    sprintf(intbuff, "%ld", GetAnalogFilterCoef());

    sprintf(send_buffer, "%s", intbuff);
    return CMDSUCCESS;
}


int HandleSetDataStreamDelay(char** tokens, int nTokens, char* send_buffer)
{
	int nArgs = nTokens - 1;
	char **args = tokens + 1;

	if (nArgs != 1)
	{
		sprintf(send_buffer, NAK_BadArgCount, nArgs, 1);
		return CMDFAILURE;
	}

	long temp;

	char * value = args[0];

	if (!TryParseLong(value, &temp))
	{
		sprintf(send_buffer, NAK_BadArg, value, 0, "decimal integer");
		return CMDFAILURE;
	}

	if (temp < DATA_STREAM_DELAY_MIN || temp > DATA_STREAM_DELAY_MAX)
	{
		sprintf(send_buffer, NAK_BadArgRange, temp, 0, DATA_STREAM_DELAY_MIN, DATA_STREAM_DELAY_MAX);
		return CMDFAILURE;
	}

	SetDataStreamDelay(temp);
	//sprintf(send_buffer, "ACK\n");
	send_buffer[0] = 0;
	return CMDSUCCESS;
}

int HandleGetDataStreamDelay(char** tokens, int nTokens, char* send_buffer)
{
	int nArgs = nTokens - 1;

	if (nArgs != 0)
	{
		sprintf(send_buffer, NAK_BadArgCount, nArgs, 0);
		return CMDFAILURE;
	}

	char intbuff[32];

	sprintf(intbuff, "%ld", GetDataStreamDelay());

	sprintf(send_buffer, "%s", intbuff);
	return CMDSUCCESS;
}
