/*
 *  Copyright (C) 2018-2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ti_board_open_close.h"
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include <drivers/bootloader.h>
#include <drivers/bootloader/bootloader_uniflash/bootloader_uniflash.h>
#include <drivers/bootloader/bootloader_xmodem.h>
#include <drivers/mmcsd.h>
#include <drivers/sciclient.h>
#include <stdlib.h>

#define BOOTLOADER_UNIFLASH_MAX_FILE_SIZE                                      \
  (0x90000) /* This has to match the size of MSRAM_2 section in linker.cmd */
uint8_t gUniflashFileBuf[BOOTLOADER_UNIFLASH_MAX_FILE_SIZE]
    __attribute__((aligned(128), section(".bss.filebuf")));

#define BOOTLOADER_UNIFLASH_VERIFY_BUF_MAX_SIZE (32 * 1024)
uint8_t gUniflashVerifyBuf[BOOTLOADER_UNIFLASH_VERIFY_BUF_MAX_SIZE]
    __attribute__((aligned(128), section(".bss")));
uint8_t gAppimage[0x40000] __attribute__((section(".app"), aligned(4096)));
/* read buffer MUST be cache line aligned when using DMA */
uint8_t gOspiRxBuf[3] __attribute__((aligned(32U)));

/* MMCSD is not available with AM243-lp. Dummy variable is added to enable build
 * for AM243-lp */
MMCSD_Config gMmcsdConfig[1];
uint32_t gMmcsdConfigNum;

uint8_t TICK_1ms = 0;

#define UPDATE_CHECK_OFFSET                                                    \
  0x200000 // flash index where byte start firmware update is located

enum {
  Reset,  // Load new fw with UART boot
  Load,   // Load fw with OSPI
  Failure // Control failed
};

/*
 * Checks if a new image is to be received and runs UART UNIFLASH
 */
int32_t UART_UNIFLASH_boot_update(uint32_t bootInstance) {
  int32_t status = SystemP_SUCCESS;
  uint32_t done = 0U;
  uint32_t fileSize;
  uint8_t filesCount = 0;
  uint32_t blk, page;
  Bootloader_UniflashConfig uniflashConfig;
  Bootloader_UniflashResponseHeader respHeader;
  gOspiRxBuf[0] = 1;
  gOspiRxBuf[1] = 1;
  gOspiRxBuf[2] = 0;

  // Read byte start firmware update
  switch (gOspiRxBuf[0]) {
  case 1: // Load new update
    DebugP_log("Update found %d\n\rfiles: %d\n\rpos: %d\n\r", gOspiRxBuf[0],
               gOspiRxBuf[1], gOspiRxBuf[2]);

    // Set update finished
    // In case of fail you can recover previous version
    gOspiRxBuf[0] = 0;
    Flash_offsetToBlkPage(gFlashHandle[CONFIG_FLASH0], UPDATE_CHECK_OFFSET,
                          &blk, &page);
    Flash_eraseBlk(gFlashHandle[CONFIG_FLASH0], blk);
    status = Flash_write(gFlashHandle[CONFIG_FLASH0], UPDATE_CHECK_OFFSET,
                         gOspiRxBuf, 3);

    while (!done) {
      /* Xmodem Receive */
      status = Bootloader_xmodemReceive(CONFIG_UART0, gUniflashFileBuf,
                                        BOOTLOADER_UNIFLASH_MAX_FILE_SIZE,
                                        &fileSize);
      DebugP_log("status =  %d \n\r", status);
      DebugP_log("fileSize =  %X \n\r", fileSize);

      /*
       * The `fileSize` wouldn't be the actual filesize, but (actual filesize +
       * size of the header + padding bytes) added by xmodem. This adds ~1KB. We
       * can't know exactly how many bytes will be padded without checking the
       * file header. But doing that will unnecessary complicate the logic, so
       * since the overhead is as small as ~1KB we could check for file size
       * exceed by checking * this `fileSize` returned by xmodem as well.
       */

      if (fileSize >= BOOTLOADER_UNIFLASH_MAX_FILE_SIZE) {
        /* Possible overflow, send error to host side */
        status = SystemP_FAILURE;

        respHeader.magicNumber = BOOTLOADER_UNIFLASH_RESP_HEADER_MAGIC_NUMBER;
        respHeader.statusCode = BOOTLOADER_UNIFLASH_STATUSCODE_FLASH_ERROR;

        Bootloader_xmodemTransmit(CONFIG_UART0, (uint8_t *)&respHeader,
                                  sizeof(Bootloader_UniflashResponseHeader));
      }

      if (status == SystemP_SUCCESS) {
        uniflashConfig.flashIndex = CONFIG_FLASH0;
        uniflashConfig.buf = gUniflashFileBuf;
        uniflashConfig.bufSize =
            0; /* Actual fileSize will be parsed from the header */
        uniflashConfig.verifyBuf = gUniflashVerifyBuf;
        uniflashConfig.verifyBufSize = BOOTLOADER_UNIFLASH_VERIFY_BUF_MAX_SIZE;

        /* Process the flash commands and return a response */
        Bootloader_uniflashProcessFlashCommands(&uniflashConfig, &respHeader);

        status = Bootloader_xmodemTransmit(
            CONFIG_UART0, (uint8_t *)&respHeader,
            sizeof(Bootloader_UniflashResponseHeader));

        if (respHeader.statusCode == BOOTLOADER_UNIFLASH_STATUSCODE_SUCCESS) {
          filesCount++;
          if (filesCount >= gOspiRxBuf[1]) // All files loaded
            done = 1;                      // Boot finished load
        } else
          filesCount = 0; // Operation error, reset count
      }
    }
    // Point to new booted image
    if (gOspiRxBuf[2] == 1 ||
        gOspiRxBuf[2] == 255) // 255 is default value of flash
      gOspiRxBuf[2] = 0;      // Go to pos 0 (0x80000)
    else
      gOspiRxBuf[2] = 1; // Go to pos 1 (0x280000)
    Flash_offsetToBlkPage(gFlashHandle[CONFIG_FLASH0], UPDATE_CHECK_OFFSET,
                          &blk, &page);
    Flash_eraseBlk(gFlashHandle[CONFIG_FLASH0], blk);
    status = Flash_write(gFlashHandle[CONFIG_FLASH0], UPDATE_CHECK_OFFSET,
                         gOspiRxBuf, 3);
    return Reset;

  default: // No update detected
    DebugP_log("No update %d\n\r", gOspiRxBuf[0]);
    return Load;
  }
}

/*
 * Loads the application with OSPI bootloader
 */
int32_t OSPI_boot_application(uint32_t bootInstance) {
  int32_t status = SystemP_SUCCESS;

  Bootloader_BootImageInfo bootImageInfo;
  Bootloader_Params bootParams;
  Bootloader_Handle bootHandle;
  Bootloader_Config *bootConfig;

  Bootloader_Params_init(&bootParams);
  Bootloader_BootImageInfo_init(&bootImageInfo);

  bootHandle = Bootloader_open(bootInstance, &bootParams);
  if (bootHandle != NULL) {

    /* Initialize PRU Cores if applicable */
    Bootloader_Config *cfg = (Bootloader_Config *)bootHandle;
    bootConfig = (Bootloader_Config *)bootHandle;
    bootConfig->scratchMemPtr = gAppimage;
    if (TRUE == cfg->initICSSCores) {
      status =
          Bootloader_socEnableICSSCores(BOOTLOADER_ICSS_CORE_DEFAULT_FREQUENCY);
      DebugP_assert(status == SystemP_SUCCESS);
    }

    status = Bootloader_parseMultiCoreAppImage(bootHandle, &bootImageInfo);
    /* Load CPUs */
    /* Do not load M4 when MCU domain is reset isolated */

    uint32_t coreVariant = Bootloader_socGetCoreVariant();
    /*Checks the core variant(Dual/Quad) */

    if (!Bootloader_socIsMCUResetIsoEnabled()) {
      if (status == SystemP_SUCCESS &&
          (TRUE ==
           Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_M4FSS0_0))) {
        bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0].clkHz =
            Bootloader_socCpuGetClkDefault(CSL_CORE_ID_M4FSS0_0);
        Bootloader_profileAddCore(CSL_CORE_ID_M4FSS0_0);
        status = Bootloader_loadCpu(
            bootHandle, &bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0]);
      }
    }
    if (status == SystemP_SUCCESS &&
        (TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_0))) {
      bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0].clkHz =
          Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS1_0);
      Bootloader_profileAddCore(CSL_CORE_ID_R5FSS1_0);
      status = Bootloader_loadCpu(bootHandle,
                                  &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0]);
    }
    /*Checks the core variant(Dual/Quad) */
    if ((coreVariant == BOOTLOADER_DEVICE_VARIANT_QUAD_CORE) &&
        status == SystemP_SUCCESS &&
        (TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_1))) {
      bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1].clkHz =
          Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS1_1);
      Bootloader_profileAddCore(CSL_CORE_ID_R5FSS1_1);
      status = Bootloader_loadCpu(bootHandle,
                                  &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1]);
    }

    /* Assume self boot for either of the cores of R50 cluster */
    uint32_t isSelfBoot = FALSE;
    if (TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS0_0)) {
      isSelfBoot = TRUE;
      Bootloader_profileAddCore(CSL_CORE_ID_R5FSS0_0);
    }
    /*Checks the core variant(Dual/Quad) */
    if ((coreVariant == BOOTLOADER_DEVICE_VARIANT_QUAD_CORE) &&
        TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS0_1)) {
      isSelfBoot = TRUE;
      Bootloader_profileAddCore(CSL_CORE_ID_R5FSS0_1);
    }

    /* Self cores has to be reset together, so check for both */
    if (status == SystemP_SUCCESS && (TRUE == isSelfBoot)) {
      /* Set clocks for self cluster */
      bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_0].clkHz =
          Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS0_0);
      if ((Bootloader_socIsR5FSSDual(BOOTLOADER_R5FSS0))) {
        bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_1].clkHz =
            Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS0_1);
      }

      /* Reset self cluster, both Core0 and Core 1. Init RAMs and load the app
       */
      status = Bootloader_loadSelfCpu(
          bootHandle, &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_0], FALSE);
      if ((status == SystemP_SUCCESS) &&
          (TRUE == Bootloader_socIsR5FSSDual(BOOTLOADER_R5FSS0))) {
        status = Bootloader_loadSelfCpu(
            bootHandle, &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_1], FALSE);
      }
    }
    Bootloader_profileAddProfilePoint("CPU load");
    Bootloader_profileUpdateAppimageSize(
        Bootloader_getMulticoreImageSize(bootHandle));
    Bootloader_profileUpdateMediaAndClk(
        BOOTLOADER_MEDIA_FLASH, OSPI_getInputClk(gOspiHandle[CONFIG_OSPI0]));

#if 1
    if (status == SystemP_SUCCESS) {
      /* enable Phy and Phy pipeline for XIP execution */
      if (OSPI_isPhyEnable(gOspiHandle[CONFIG_OSPI0])) {
        status = OSPI_enablePhy(gOspiHandle[CONFIG_OSPI0]);
        DebugP_assert(status == SystemP_SUCCESS);

        status = OSPI_enablePhyPipeline(gOspiHandle[CONFIG_OSPI0]);
        DebugP_assert(status == SystemP_SUCCESS);
      }
    }
#endif

    if (status == SystemP_SUCCESS) {
      /* Print SBL Profiling logs to UART as other cores may use the UART for
       * logging */
      Bootloader_profileAddProfilePoint("SBL End");
      Bootloader_profilePrintProfileLog();
      DebugP_log("Image loading done, switching to application ...\r\n");
      UART_flushTxFifo(gUartHandle[CONFIG_UART0]);
    }

    /* Run CPUs */
    /* Do not run M4 when MCU domain is reset isolated */
    if (!Bootloader_socIsMCUResetIsoEnabled()) {
      if (status == SystemP_SUCCESS &&
          (TRUE ==
           Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_M4FSS0_0))) {
        status = Bootloader_runCpu(
            bootHandle, &bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0]);
      }
    }
    if (status == SystemP_SUCCESS &&
        (TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_0))) {
      status = Bootloader_runCpu(bootHandle,
                                 &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0]);
    }
    /*Checks the core variant(Dual/Quad) */
    if ((Bootloader_socIsR5FSSDual(BOOTLOADER_R5FSS1)) &&
        status == SystemP_SUCCESS &&
        (TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_1))) {
      status = Bootloader_runCpu(bootHandle,
                                 &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1]);
    }
    if (status == SystemP_SUCCESS) {
      /* Reset self cluster, both Core0 and Core 1. Init RAMs and run the app */
      status = Bootloader_runSelfCpu(bootHandle, &bootImageInfo);
    }
    /* it should not return here, if it does, then there was some error */
    Bootloader_close(bootHandle);
  }

  if (status != SystemP_SUCCESS) {
    DebugP_log("Some tests have failed!!\r\n");
  }
  Drivers_close();
  System_deinit();

  return 0;
  //     status = Bootloader_parseMultiCoreAppImage(bootHandle, &bootImageInfo);
  //     /* Load CPUs */
  //     /* Do not load M4 when MCU domain is reset isolated */
  //     if (!Bootloader_socIsMCUResetIsoEnabled())
  //     {
  //         if(status == SystemP_SUCCESS && (TRUE ==
  //         Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_M4FSS0_0)))
  //         {
  //             bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0].clkHz =
  //             Bootloader_socCpuGetClkDefault(CSL_CORE_ID_M4FSS0_0);
  //             Bootloader_profileAddCore(CSL_CORE_ID_M4FSS0_0);
  //             status = Bootloader_loadCpu(bootHandle,
  //             &bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0]);
  //         }
  //     }
  //     if(status == SystemP_SUCCESS && (TRUE ==
  //     Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_0)))
  //     {
  //         bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0].clkHz =
  //         Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS1_0);
  //         Bootloader_profileAddCore(CSL_CORE_ID_R5FSS1_0);
  //         status = Bootloader_loadCpu(bootHandle,
  //         &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0]);
  //     }
  //     if(status == SystemP_SUCCESS && (TRUE ==
  //     Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_1)))
  //     {
  //         bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1].clkHz =
  //         Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS1_1);
  //         Bootloader_profileAddCore(CSL_CORE_ID_R5FSS1_1);
  //         status = Bootloader_loadCpu(bootHandle,
  //         &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1]);
  //     }

  //     /* Assume self boot for either of the cores of R50 cluster */
  //     uint32_t isSelfBoot = FALSE;
  //     if(TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS0_0))
  //     {
  //         isSelfBoot = TRUE;
  //         Bootloader_profileAddCore(CSL_CORE_ID_R5FSS0_0);
  //     }

  //     if(TRUE == Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS0_1))
  //     {
  //         isSelfBoot = TRUE;
  //         Bootloader_profileAddCore(CSL_CORE_ID_R5FSS0_1);
  //     }

  //     /* Self cores has to be reset together, so check for both */
  //     if(status == SystemP_SUCCESS && (TRUE == isSelfBoot))
  //     {
  //         /* Set clocks for self cluster */
  //         bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_0].clkHz =
  //         Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS0_0);
  //         bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_1].clkHz =
  //         Bootloader_socCpuGetClkDefault(CSL_CORE_ID_R5FSS0_1);

  //         /* Reset self cluster, both Core0 and Core 1. Init RAMs and load
  //         the app  */ status = Bootloader_loadSelfCpu(bootHandle,
  //         &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_0], FALSE); if((status ==
  //         SystemP_SUCCESS) && (TRUE ==
  //         Bootloader_socIsR5FSSDual(BOOTLOADER_R5FSS0)))
  //         {
  //             status = Bootloader_loadSelfCpu(bootHandle,
  //             &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS0_1], FALSE);
  //         }
  //     }
  //     Bootloader_profileAddProfilePoint("CPU load");
  //     Bootloader_profileUpdateAppimageSize(Bootloader_getMulticoreImageSize(bootHandle));
  //     Bootloader_profileUpdateMediaAndClk(BOOTLOADER_MEDIA_FLASH,
  //     OSPI_getInputClk(gOspiHandle[CONFIG_OSPI0]));

  //     #if 1
  //     if( status == SystemP_SUCCESS)
  //     {
  //         /* enable Phy and Phy pipeline for XIP execution */
  //         if( OSPI_isPhyEnable(gOspiHandle[CONFIG_OSPI0]) )
  //         {
  //             status = OSPI_enablePhy(gOspiHandle[CONFIG_OSPI0]);
  //             DebugP_assert(status == SystemP_SUCCESS);

  //             status = OSPI_enablePhyPipeline(gOspiHandle[CONFIG_OSPI0]);
  //             DebugP_assert(status == SystemP_SUCCESS);
  //         }
  //     }
  //     #endif

  //     if(status == SystemP_SUCCESS)
  //     {
  //         /* Print SBL Profiling logs to UART as other cores may use the UART
  //         for logging */ Bootloader_profileAddProfilePoint("SBL End");
  //         Bootloader_profilePrintProfileLog();
  //         DebugP_log("Image loading done, switching to application ...\r\n");
  //         UART_flushTxFifo(gUartHandle[CONFIG_UART0]);
  //     }

  //     /* Run CPUs */
  //     /* Do not run M4 when MCU domain is reset isolated */
  //     if (!Bootloader_socIsMCUResetIsoEnabled())
  //     {
  //         if(status == SystemP_SUCCESS && (TRUE ==
  //         Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_M4FSS0_0)))
  //         {
  //             status = Bootloader_runCpu(bootHandle,
  //             &bootImageInfo.cpuInfo[CSL_CORE_ID_M4FSS0_0]);
  //         }
  //     }
  //     if(status == SystemP_SUCCESS && (TRUE ==
  //     Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_0)))
  //     {
  //         status = Bootloader_runCpu(bootHandle,
  //         &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_0]);
  //     }
  //     if(status == SystemP_SUCCESS && (TRUE ==
  //     Bootloader_isCorePresent(bootHandle, CSL_CORE_ID_R5FSS1_1)))
  //     {
  //         status = Bootloader_runCpu(bootHandle,
  //         &bootImageInfo.cpuInfo[CSL_CORE_ID_R5FSS1_1]);
  //     }
  //     if(status == SystemP_SUCCESS && (TRUE == isSelfBoot))
  //     {
  //         /* Reset self cluster, both Core0 and Core 1. Init RAMs and run the
  //         app  */ status = Bootloader_runSelfCpu(bootHandle, &bootImageInfo);
  //     }
  //     /* it should not return here, if it does, then there was some error */
  //     Bootloader_close(bootHandle);
  // }
  // else
  //     DebugP_log("Failed to open boot\n\r");
  // return status;
}

void boot_flow() {
  int32_t status;

  Bootloader_profileReset();

  Bootloader_socWaitForFWBoot();

#ifndef DISABLE_WARM_REST_WA
  /* Warm Reset Workaround to prevent CPSW register lockup */
  if (!Bootloader_socIsMCUResetIsoEnabled()) {
    Bootloader_socResetWorkaround();
  }
#endif

  Bootloader_profileAddProfilePoint("SYSFW init");

  if (!Bootloader_socIsMCUResetIsoEnabled()) {
    /* Update devGrp to ALL to initialize MCU domain when reset isolation is
    not enabled */
    Sciclient_BoardCfgPrms_t boardCfgPrms_pm = {
        .boardConfigLow = (uint32_t)0,
        .boardConfigHigh = 0,
        .boardConfigSize = 0,
        .devGrp = DEVGRP_ALL,
    };

    status = Sciclient_boardCfgPm(&boardCfgPrms_pm);

    Sciclient_BoardCfgPrms_t boardCfgPrms_rm = {
        .boardConfigLow = (uint32_t)0,
        .boardConfigHigh = 0,
        .boardConfigSize = 0,
        .devGrp = DEVGRP_ALL,
    };

    status = Sciclient_boardCfgRm(&boardCfgPrms_rm);

    /* Enable MCU PLL. MCU PLL will not be enabled by DMSC when devGrp is set
    to Main in boardCfg */
    Bootloader_enableMCUPLL();
  }

  Bootloader_socOpenFirewalls();

  Bootloader_socNotifyFirewallOpen();

  System_init();
  Bootloader_profileAddProfilePoint("System_init");

  Drivers_open();
  Bootloader_profileAddProfilePoint("Drivers_open");

#if 0
    DebugP_log("\r\n");
    DebugP_log("Starting OSPI Bootloader ... \r\n");
#endif

  status = Board_driversOpen();
  DebugP_assert(status == SystemP_SUCCESS);
  Bootloader_profileAddProfilePoint("Board_driversOpen");

  status = Sciclient_getVersionCheck(0);
  Bootloader_profileAddProfilePoint("Sciclient Get Version");
  if (SystemP_SUCCESS == status) {
    status = Flash_read(gFlashHandle[CONFIG_FLASH0], UPDATE_CHECK_OFFSET,
                        gOspiRxBuf, 3);
    if (status == SystemP_SUCCESS) {
      // Check for updates
      DebugP_log("Checking for updates\n\r");
      status = UART_UNIFLASH_boot_update(BOOTLOADER_FLASH_UPDATE);

      if (status == Reset) {
        DebugP_log("Reset ... \n\r");
        SOC_generateSwWarmResetMainDomain(); // Reset program
      }

      // Boot program
      DebugP_log("Starting OSPI boot %d\n\r",
                 BOOTLOADER_FLASH_APPLICATION_POS0);
      if (gOspiRxBuf[2] == 1) // Pos 1
        status = OSPI_boot_application(BOOTLOADER_FLASH_APPLICATION_POS1);
      else // Pos 0
        status = OSPI_boot_application(BOOTLOADER_FLASH_APPLICATION_POS0);
    } else
      DebugP_log("Failed to read flash\n\r");
  }
  if (status != SystemP_SUCCESS) {
    DebugP_log("Some tests have failed!!\r\n");
  }

  Drivers_close();
  System_deinit();
}
