//---------------------------------------------------------------------------------------------------------------
//  can_test.cc
//
//  Minimum code to debug CAN bus working in Loopback mode but not in normal mode.
//
//  Copyright (c) 2025-2026 Doug Broadwell, all rights reserved.
 //---------------------------------------------------------------------------------------------------------------

#include "common.h"

#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>

#include "tm4c123ae6pm.h"

#include "sysctl.h"
#include "pin_map.h"
#include "hw_memmap.h"
#include "hw_gpio.h"
#include "gpio.h"
#include "uart.h"

#include "can.h"
#include "hw_memmap.h"
#include "interrupt.h"
#include "hw_can.h"
#include "hw_types.h"


char sBuff[200];


//-----------------------------
//  TRANSMIT STATUS MESSAGES  |
//---------------------------------------------------------------------------------------------------------------
void TxStr(char* pStr) {

    while(*pStr) {
        UART4_DR_R = *pStr;                     // Tx character.
        while(UART4_FR_R & UART_FR_BUSY) {}     // Wait till it's done.
        pStr++;                                 // Next character.
    }
}
//---------------------------------------------------------------------------------------------------------------
inline void TxStr(const char* pChar) {
    TxStr(const_cast<char*>(pChar));
}
//---------------------------------------------------------------------------------------------------------------




//------------------
//  CONFIGURE MPU  |
//---------------------------------------------------------------------------------------------------------------
void ConfigMPU(void) {

    SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_XTAL_20MHZ | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN);  // Should be 80MHz
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);                // Enable Port B, CAN Bus.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB)) {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);                // Enable Port C, UART.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOC)) {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);                // Enable Port D, DIP Switch.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOD)) {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);                // Enable Port G, Heartbeat LEDs.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOG)) {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);                 // Enable CAN 0.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_CAN0))  {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART4);                // Enable UART4 Debug Port.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_UART4)) {}

    GPIOPinTypeCAN  (GPIO_PORTB_BASE, PB_Can2Rx_BIT | PB_Can2Tx_BIT );      // CAN I/O
    GPIOPinConfigure(GPIO_PB4_CAN0RX);
    GPIOPinConfigure(GPIO_PB5_CAN0TX);

    GPIOPinTypeUART(GPIO_PORTC_BASE, PC_DebugRx_BIT | PC_DebugTx_BIT);      // UART I/O
    GPIOPinConfigure(GPIO_PC4_U4RX);
    GPIOPinConfigure(GPIO_PC5_U4TX);
    GPIO_PORTC_PDR_R = 0b11000000;                                          // Weak pull-down.

    GPIOPinTypeGPIOInput(GPIO_PORTD_BASE, PD_DipSw1_BIT | PD_DipSw2_BIT);   // DIP Switch.
    GPIO_PORTD_PDR_R = PD_DipSw1_BIT | PD_DipSw2_BIT;                       // Weak pull-down.

    GPIOPinTypeGPIOOutput(GPIO_PORTG_BASE, PG_HB_LED_BIT | PG_ER_LED_BIT);  // LEDs.

}
//---------------------------------------------------------------------------------------------------------------




//--------------------------
//  DISPLAY CAN REGISTERS  |
//---------------------------------------------------------------------------------------------------------------
void PrtStatus(void) {

    CAN0_IF1MCTL_R = 0x1;        // Point to Message Object #1
    sprintf(sBuff, "\r\n    CAN0TXRQ1 = 0x%X, CANIF1CTL = 0x%X\r\n", CAN0_TXRQ1_R, CAN0_IF1MCTL_R);
    TxStr(sBuff);
    sprintf(sBuff, "\r\n    CAN0ERR   = 0x%X, CAN0STS   = 0x%X\r\n\n", CAN0_ERR_R, CAN0_STS_R);
    TxStr(sBuff);
}
//---------------------------------------------------------------------------------------------------------------




//-------------
//  CAN TEST  |
//---------------------------------------------------------------------------------------------------------------
//  The problem I have is that transmissions and receptions work fine in Loopback Mode (see Can2LoopbackTest()
//  in can_local.cc), and the Tx pin can be driven high and low in Test mode, but transmission never starts
//  when in Normal Mode.  The on-board DIP switch selects what test to run:
//
//  DIP-1  DIP-2
//  -----  -----
//   Off    Off    Normal Mode
//   On     Off    Loopback Test
//   Off    On     Set Tx Low
//   On     On     Set Tx High
//---------------------------------------------------------------------------------------------------------------
void CanTest(void) {

    volatile u32 i;
    uint Which;
    enum eWhich {Normal=0, Loopback=PD_DipSw1_BIT, SetLow=PD_DipSw2_BIT, SetHigh=PD_DipSw1_BIT | PD_DipSw2_BIT};

    static u8  Can0TxData1[8];    // Tx Message Data
    Can0TxData1[0] = 0x12;
    Can0TxData1[1] = 0x34;
    Can0TxData1[2] = 0x56;
    Can0TxData1[3] = 0x78;
    Can0TxData1[4] = 0x9A;
    Can0TxData1[5] = 0xBC;
    Can0TxData1[6] = 0xDE;
    Can0TxData1[7] = 0xF0;

    static u8 Can0RxData2[8];     // Rx Message Data

    tCANMsgObject Can0MsgObj1, Can0MsgObj2;

    //-- INITIALIZE UART AND CAN-0 --//

    UARTConfigSetExpClk(UART4_BASE, 80000000, 19200,                    // Init UART
         ( UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE ));
    CANInit(CAN0_BASE);                                                 // Init CAN 0.
    if(CANBitRateSet(CAN0_BASE, 80000000, 500000) == 0) {
        TxStr("\r\n CANBitRateSet Failed\r\n");
        while(true) {}
    }
    CANEnable(CAN0_BASE);

    TxStr("\r\n\n --");

    //-- SEE WHICH TEST TO RUN --//

    Which = GPIO_PORTD_DATA_R & (PD_DipSw1_BIT | PD_DipSw2_BIT);

    //------------------------------//
    //-- SET TX LOW OR HIGH TESTS --//
    //------------------------------//

    if(Which == SetLow or Which == SetHigh) {
        if(Which == SetLow) {
            HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_TX_DOMINANT;         // Drive Low.
            strcpy(sBuff, " DRIVE TX LOW ");
        } else {
            HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_TX_RECESSIVE;        // Drive high.
            strcpy(sBuff, " DRIVE TX HIGH ");
        }
        HWREG(CAN0_BASE + CAN_O_CTL) |= CAN_CTL_TEST;
        TxStr(sBuff);
        TxStr("--\r\n\n");
        while(true) {                               // Heartbeat LED.
            for(i=4000000; i != 0; i--) {}          // Die Here.
            HB_LED_Toggle();
        }
    }

    //------------------------------//
    //-- NORMAL OR LOOPBACK TESTS --//
    //------------------------------//

    if(Which == Loopback) {
        HWREG(CAN0_BASE + CAN_O_CTL) |= CAN_CTL_TEST;   // Turn On Local Loopback.
        HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_LBACK;
        TxStr(" LOOPBACK TEST --\r\n");
    } else {
        TxStr(" NORMAL TEST --\r\n");
    }

    //-- INITIALIZE MESSAGE OBJECTS --//

    Can0MsgObj2.ui32MsgID     = 0;                      // Receive Message Object #2.
    Can0MsgObj2.ui32MsgIDMask = 0;
    Can0MsgObj2.ui32Flags     = MSG_OBJ_USE_ID_FILTER;
    Can0MsgObj2.ui32MsgLen    = 8;
    Can0MsgObj2.pui8MsgData   = Can0RxData2;
    CANMessageSet(CAN0_BASE, 2, &Can0MsgObj2, MSG_OBJ_TYPE_RX);

    Can0MsgObj1.ui32MsgID     = 0x100;                  // Transmit Message Object #1.
    Can0MsgObj1.ui32MsgIDMask = 0;
    Can0MsgObj1.ui32Flags     = MSG_OBJ_TX_INT_ENABLE;  // Set this so we can read the INTPND flag.
    Can0MsgObj1.ui32MsgLen    = 8;
    Can0MsgObj1.pui8MsgData   = Can0TxData1;

    TxStr("\r\n\n- Message not yet set\r\n");
    PrtStatus();

    CANMessageSet(CAN0_BASE, 1, &Can0MsgObj1, MSG_OBJ_TYPE_TX); // Start Transmission.

    TxStr("- Transmit CANMessageSet() called\r\n");
    PrtStatus();

    for(i=2000000; i != 0; i--) {}              // Delay a bit.

    TxStr("- After a Delay\r\n");
    PrtStatus();

    CANMessageGet(CAN0_BASE, 2, &Can0MsgObj2, true);
    if(Can0MsgObj2.ui32Flags & MSG_OBJ_NEW_DATA) {
        sprintf(sBuff, "    Message Received: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\r\n\n",
            Can0RxData2[0], Can0RxData2[1], Can0RxData2[2], Can0RxData2[3],
            Can0RxData2[4], Can0RxData2[5], Can0RxData2[6], Can0RxData2[7]);
        TxStr(sBuff);
    } else {
        TxStr("    No Message Received\r\n\n");
    }

    while(true) {                               // Heartbeat LED.
        for(i=4000000; i != 0; i--) {}
        HB_LED_Toggle();
    }
}
//---------------------------------------------------------------------------------------------------------------





                                    ////////////////////////
                                    //                    //
                                    //  SIMPLE_TX.C TEST  //
                                    //                    //
                                    ////////////////////////


// u32 MsgCnt = 0;

//------------------------------
//  SIMPLE CAN TEST INTERRUPT  |
//---------------------------------------------------------------------------------------------------------------
void SCTT_ISR(void) {

    u32 Status;

    Status = CANIntStatus(CAN0_BASE, CAN_INT_STS_CAUSE);
    if(Status == CAN_INT_INTID_STATUS) {
        Status = CANStatusGet(CAN0_BASE, CAN_STS_CONTROL);
//      sprintf(sBuff, "\r\n Error, Status = 0x%X\r\n", Status);
//      TxStr(sBuff);
        while(true) {}                                              // Die Here.
    } else if(Status == 1) {
        CANIntClear(CAN0_BASE, 1);
        LED_1_Toggle();
//      MsgCnt++;
    } else {
//      TxStr("\r\n Spurious Interrupt\r\n");
        while(true) {}                                              // Die Here.
    }
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------
//  SIMPLE CAN TRANSMIT TEST  |
//---------------------------------------------------------------------------------------------------------------
void SimpleCanTxTest(void) {

    volatile u32 i;

    u32  Data = 0x12345678;
    u8* pData = (u8*)&Data;

    tCANMsgObject Message;

    //-- SET UP THE MPU --//

    SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_XTAL_20MHZ | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN);  // Should be 80MHz

    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);                            // Enable Port B, CAN Bus.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB)) {}
/*  SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);                            // Enable Port C, UART.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOC)) {}
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);                            // Enable Port G, LEDs.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOG)) {}
*/
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);                             // Enable CAN 0.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_CAN0))  {}
/*
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART4);                            // Enable UART4 Debug Port.
    while( ! SysCtlPeripheralReady(SYSCTL_PERIPH_UART4)) {}
*/
    GPIOPinTypeCAN  (GPIO_PORTB_BASE, PB_Can2Rx_BIT | PB_Can2Tx_BIT );      // CAN I/O
    GPIOPinConfigure(GPIO_PB4_CAN0RX);
    GPIOPinConfigure(GPIO_PB5_CAN0TX);
/*
    GPIOPinTypeUART(GPIO_PORTC_BASE, PC_DebugRx_BIT | PC_DebugTx_BIT);      // UART I/O
    GPIOPinConfigure(GPIO_PC4_U4RX);
    GPIOPinConfigure(GPIO_PC5_U4TX);

    GPIOPinTypeGPIOOutput(GPIO_PORTG_BASE, PG_HB_LED_BIT | PG_ER_LED_BIT);  // LEDs.
*/
    //-- INITIALIZE UART AND CAN-0 --//

//  UARTConfigSetExpClk(UART4_BASE, 80000000, 19200,                    // Init UART
//       ( UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE ));
    CANInit(CAN0_BASE);                                                 // Init CAN 0.
    if(CANBitRateSet(CAN0_BASE, 80000000, 500000) == 0) {
//      TxStr("\r\n CANBitRateSet Failed\r\n");
        while(true) {}
    }
    CANIntRegister(CAN0_BASE, SCTT_ISR);
    CANIntEnable(CAN0_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);
    IntEnable(INT_CAN0);
    CANEnable(CAN0_BASE);

//  TxStr("\r\n -- TEST START --\r\n");

    Message.ui32MsgID     = 1;
    Message.ui32MsgIDMask = 0;
    Message.ui32Flags     = MSG_OBJ_TX_INT_ENABLE;
    Message.ui32MsgLen    = 8;
    Message.pui8MsgData   = pData;

    while(true) {
        CANMessageSet(CAN0_BASE, 1, &Message, MSG_OBJ_TYPE_TX);
//      TxStr(" - Set\r\n");

        for(i=0; i<8000000; i++) {}
    }
}
//---------------------------------------------------------------------------------------------------------------


