#include "i2cfunc.h"

static bool __i2cinit = false;
uint8_t no_of_bytes_to_read = 1;
uint8_t no_of_bytes_to_write = 1;

static const unsigned long g_uli2cBase[4] =
{
    GPIO_PORTB_BASE, GPIO_PORTA_BASE, GPIO_PORTE_BASE, GPIO_PORTD_BASE
};

static const unsigned long g_uli2cSDAPins[4] =
{
    GPIO_PIN_3, GPIO_PIN_7, GPIO_PIN_5, GPIO_PIN_1
};
static const unsigned long g_uli2cSCLPins[4] =
{
    GPIO_PIN_2, GPIO_PIN_6, GPIO_PIN_4, GPIO_PIN_0
};

static const unsigned long g_uli2cConfig[4][2] =
{
    {GPIO_PB2_I2C0SCL, GPIO_PB3_I2C0SDA},
    {GPIO_PA6_I2C1SCL, GPIO_PA7_I2C1SDA},
    {GPIO_PE4_I2C2SCL, GPIO_PE5_I2C2SDA},
    {GPIO_PD0_I2C3SCL, GPIO_PD1_I2C3SDA}
};

static const unsigned long g_uli2cPeriph[4] =
{
    SYSCTL_PERIPH_I2C0, SYSCTL_PERIPH_I2C1,
    SYSCTL_PERIPH_I2C2, SYSCTL_PERIPH_I2C3
};

static const unsigned long g_uli2cMasterBase[4] =
{
    I2C0_MASTER_BASE, I2C1_MASTER_BASE,
	I2C2_MASTER_BASE, I2C3_MASTER_BASE
};

#define MASTER_BASE g_uli2cMasterBase[i2cModule]
#define i2cModule 1

// buffer used to write elements on i2c
#define BUFF_LEN 10

volatile static uint8_t buff_index = 0;
volatile static uint8_t buff[BUFF_LEN] = {};

// init I2C module 1
bool InitI2C1(void);

// begin transmission by setting slave address
void I2CBeginTransmission( uint8_t slaveAddr );

// end transmission
uint8_t I2CEndTransmission();

//sends an I2C command to the specified slave
//void I2CSend(uint8_t slave_addr, uint8_t num_of_args, ...);
void I2CSend( uint8_t data );

void I2CWriteRegister( uint8_t reg, uint8_t data );

//sends an array of data via I2C to the specified slave
//void I2CSendString(uint32_t slave_addr, char array[]);

// request a no. of bytes from slave
void I2CRequestFrom( uint8_t i2cAddr, uint8_t noOfBytes );

// receive a given no. of bytes from slave
unsigned long I2CReceiveBytes ( void );

// get value stored in a register on slave
unsigned long I2CReceiveRegContent( uint8_t reg );

// check device availability
bool available( uint8_t slave_addr );


i2cAPI Wire = {
		.InitI2C = &InitI2C1,

		.beginTransmission = &I2CBeginTransmission,
		.endTransmission = &I2CEndTransmission,

		.requestFrom = &I2CRequestFrom,
		.read = &I2CReceiveBytes,
		.readReg = &I2CReceiveRegContent,

		.write = &I2CSend,
		.writeReg = &I2CWriteRegister,

		.isAvailable = &available
};


bool InitI2C1(void)
{
	if (__i2cinit){
		return true;
	}
	// set i2c
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C1);

	//reset I2C module
	ROM_SysCtlPeripheralReset(SYSCTL_PERIPH_I2C1);

	//enable GPIO peripheral that contains I2C
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

	// Configure the pin muxing for I2C0 functions on port B2 and B3.
	ROM_GPIOPinConfigure(GPIO_PA6_I2C1SCL);
	ROM_GPIOPinConfigure(GPIO_PA7_I2C1SDA);

	// Select the I2C function for these pins.
	ROM_GPIOPinTypeI2CSCL(GPIO_PORTA_BASE, GPIO_PIN_6);
	ROM_GPIOPinTypeI2C(GPIO_PORTA_BASE, GPIO_PIN_7);

	// The last parameter sets the I2C data transfer rate.
	// If false the data rate is set to 100kbps and if true the data rate will
	// be set to 400kbps.
	ROM_I2CMasterInitExpClk(I2C1_MASTER_BASE, ROM_SysCtlClockGet(), false);
	__i2cinit = true ;
	return __i2cinit;
}

// begin transmission by setting address for slave
void I2CBeginTransmission( uint8_t slaveAddr ){
	ROM_I2CMasterSlaveAddrSet(I2C1_MASTER_BASE, slaveAddr, false);
}


void _send_data(){
	// send data, given elements present in buff
	ROM_I2CMasterDataPut(I2C1_MASTER_BASE, buff[0]);

	//if there is only one argument, we only need to use the
	//single send I2C function
	if(buff_index == 1) // case where we have only one element in buffer; buff_index is always no. of elements. + 1
	{
		//Initiate send of data from the MCU
		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_SINGLE_SEND);

		// Wait until MCU is done transferring.
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));

	}

	//otherwise, we start transmission of multiple bytes on the
	//I2C bus
	else
	{
		//Initiate send of data from the MCU
		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_START);

		// Wait until MCU is done transferring.
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));

		//send num_of_args-2 pieces of data, using the
		//BURST_SEND_CONT command of the I2C module

		uint8_t i;
		for (i=1; i < (buff_index -1); i++)
		{
			//put next piece of data into I2C FIFO
			ROM_I2CMasterDataPut(I2C1_MASTER_BASE, buff[i]);
			//send next data that was just placed into FIFO
			ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);

			// Wait until MCU is done transferring.
			while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
		}

		//put last piece of data into I2C FIFO
		ROM_I2CMasterDataPut(I2C1_MASTER_BASE, buff[buff_index - 1]);
		//send next data that was just placed into FIFO
		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);
		// Wait until MCU is done transferring.
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));

	}
}


//sends an I2C command to the specified slave
void I2CSend( uint8_t data )
{
	buff[buff_index] = data;
	buff_index = buff_index + 1;

//
//
//	ROM_I2CMasterDataPut(I2C1_MASTER_BASE, data);
//
//	//if there is only one argument, we only need to use the
//	//single send I2C function
//	if(no_of_bytes_to_write == 1)
//	{
//		//Initiate send of data from the MCU
//		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_SINGLE_SEND);
//
//		// Wait until MCU is done transferring.
//		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//
//		//"close" variable argument list
//		va_end(vargs);
//	}
//
//	//otherwise, we start transmission of multiple bytes on the
//	//I2C bus
//	else
//	{
//		//Initiate send of data from the MCU
//		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_START);
//
//		// Wait until MCU is done transferring.
//		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//
//		//send num_of_args-2 pieces of data, using the
//		//BURST_SEND_CONT command of the I2C module
//
//		uint8_t i;
//		for (i=1; i < (no_of_bytes_to_write -1); i++)
//		{
//			//put next piece of data into I2C FIFO
//			ROM_I2CMasterDataPut(I2C1_MASTER_BASE, va_arg(vargs, uint32_t));
//			//send next data that was just placed into FIFO
//			ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);
//
//			// Wait until MCU is done transferring.
//			while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//		}
//
//		//put last piece of data into I2C FIFO
//		ROM_I2CMasterDataPut(I2C1_MASTER_BASE, va_arg(vargs, uint32_t));
//		//send next data that was just placed into FIFO
//		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);
//		// Wait until MCU is done transferring.
//		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//
//		//"close" variable args list
//		va_end(vargs);
//	}
//
//    //stores list of variable number of arguments
//    va_list vargs;

    //specifies the va_list to "open" and the last fixed argument
    //so vargs knows where to start looking
//    va_start(vargs, num_of_args);

    //put data to be sent into FIFO
//    ROM_I2CMasterDataPut(I2C1_MASTER_BASE, va_arg(vargs, uint32_t));

}

// end transmission
uint8_t I2CEndTransmission( void ){
	_send_data();
	// reset buff_index to 0 allowing for next transmission
	buff_index = 0;
	return 0;
}

// write given data value to register
void I2CWriteRegister( uint8_t reg, uint8_t data ){
	I2CSend( data );
	I2CSend( data );
}

// begin transmission and specify how many bytes should be read from slave
void I2CRequestFrom( uint8_t i2cAddr, uint8_t noOfBytes ){
	I2CBeginTransmission( i2cAddr );
	no_of_bytes_to_read = noOfBytes;
}

// read a specified no. of bytes from slave
unsigned long I2CReceiveBytes ( void ){

	// check how many bytes we need to read
	if (no_of_bytes_to_read == 1){
		//send control byte and read from the register we
		//specified
		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);

		//wait for MCU to finish transaction
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));

		//return data pulled from the specified register
		return ROM_I2CMasterDataGet(I2C1_MASTER_BASE);
	}
	else{
		uint32_t received = 0;
		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_RECEIVE_START);

		//wait for MCU to finish transaction
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));

		// get first byte and place it accordingly in the final result
		received = ROM_I2CMasterDataGet(I2C1_MASTER_BASE);

		uint8_t i;
		for (i=1; i<(no_of_bytes_to_read -1); i++){
			ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_RECEIVE_CONT);

			//wait for MCU to finish transaction
			while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
			received <<= 8;
			received |= ROM_I2CMasterDataGet(I2C1_MASTER_BASE);

		}

		ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_RECEIVE_FINISH);

		//wait for MCU to finish transaction
		while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
		received <<= 8;
		received |= ROM_I2CMasterDataGet(I2C1_MASTER_BASE);
		no_of_bytes_to_read = 1; // default value. Except the case when requestFrom is called to ask for more than one byte of data from slave, all other cases assume one byte only
		return received;
	}
}

// receive content of a register from slave
unsigned long I2CReceiveRegContent( uint8_t reg ){
	I2CSend( reg );
	return I2CReceiveBytes();
}

bool available ( uint8_t slave_addr ){
	// send no data. Just wait for ACK after sending address. This confirms slave
	// is online and answers to master
	ROM_I2CMasterSlaveAddrSet(I2C1_MASTER_BASE, slave_addr, false);
	ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_SINGLE_SEND);

	while(I2CMasterBusy(I2C1_MASTER_BASE));

	volatile unsigned long err = ROM_I2CMasterErr(I2C1_MASTER_BASE);

	if (err == I2C_MASTER_ERR_NONE) {
		return true;
	} else {
		return false;
	}
}



////sends an array of data via I2C to the specified slave
//void I2CSendString(uint32_t slave_addr, char array[])
//{
//    // Tell the master module what address it will place on the bus when
//    // communicating with the slave.
//	ROM_I2CMasterSlaveAddrSet(I2C1_MASTER_BASE, slave_addr, false);
//
//    //put data to be sent into FIFO
//	ROM_I2CMasterDataPut(I2C1_MASTER_BASE, array[0]);
//
//    //if there is only one argument, we only need to use the
//    //single send I2C function
//    if(array[1] == '\0')
//    {
//        //Initiate send of data from the MCU
//    	ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_SINGLE_SEND);
//
//        // Wait until MCU is done transferring.
//        while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//    }
//
//    //otherwise, we start transmission of multiple bytes on the
//    //I2C bus
//    else
//    {
//        //Initiate send of data from the MCU
//    	ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_START);
//
//        // Wait until MCU is done transferring.
//        while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//
//        //initialize index into array
//        uint8_t i = 1;
//
//        //send num_of_args-2 pieces of data, using the
//        //BURST_SEND_CONT command of the I2C module
//        while(array[i + 1] != '\0')
//        {
//            //put next piece of data into I2C FIFO
//        	ROM_I2CMasterDataPut(I2C1_MASTER_BASE, array[i++]);
//
//            //send next data that was just placed into FIFO
//        	ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_CONT);
//
//            // Wait until MCU is done transferring.
//            while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//        }
//
//        //put last piece of data into I2C FIFO
//        ROM_I2CMasterDataPut(I2C1_MASTER_BASE, array[i]);
//
//        //send next data that was just placed into FIFO
//        ROM_I2CMasterControl(I2C1_MASTER_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH);
//
//        // Wait until MCU is done transferring.
//        while(ROM_I2CMasterBusy(I2C1_MASTER_BASE));
//    }
//}
