/*
 * RDGxxxx.c
 *
 *  Created on: 6 lug 2020
 *      Author: mfantina
 */

#include <ti/drivers/net/wifi/sl_socket.h>

#include <cmd_parser.h>
#include <pthread.h>
#include <stdint.h>
#include <uart_term.h>

#include "str.h"
#include "RDGxxxx.h"

void *Us_Task(void *arg);
void  Us_Send(int16_t sock);
void *Msg_Task(void *arg);
void  Msg_RecvSend(int16_t sock);
int16_t CreateTcpSock_Bind_Listen(int tcpPort);
int16_t Accept(int16_t tcpSocket);

RdgStruct Rdg; /* Istanza struttura Rdg */

/* Configura il Processore di Rete */

int32_t Configure_NWP(void)
{
    /* Avvio Network Processor */

    int32_t ret = sl_Start(0, 0, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Avvio Network Processor OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Avvio Network Processor fallito, codice %d", ret);
        return (FALSE);
    }

    /* Impostazione modalit P2P-CLIENT */

    ret = sl_WlanPolicySet(SL_WLAN_POLICY_P2P, SL_WLAN_P2P_POLICY(SL_WLAN_P2P_ROLE_CLIENT, NULL), NULL, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione modalit P2P-CLIENT OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione P2P-CLIENT fallita, codice %d", ret);
        return (FALSE);
    }

    /* Impostazione POLICY_CONNECTION */

    ret = sl_WlanPolicySet(SL_WLAN_POLICY_CONNECTION, SL_WLAN_CONNECTION_POLICY(0,0,0,0), NULL, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione POLICY_CONNECTION OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione POLICY_CONNECTION fallita, codice %d", ret);
        return (FALSE);
    }

    /* Impostazione DEVICE-TYPE (Default is 1-0050F204-1) */

    const char devType[] = "1-0050F204-1";

    ret = sl_WlanSet(SL_WLAN_CFG_P2P_PARAM_ID, SL_WLAN_P2P_OPT_DEV_TYPE, strlen(devType), (uint8_t*)devType);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione DEVICE-TYPE : %s OK!", devType);
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione DEVICE-TYPE fallita, codice %d", ret);
        return (FALSE);
    }

    /* Impostazione LISTEN_CHANNEL */

    uint8_t channels[] =
    {
        11,/*LISTEN_CHANNEL (Canale di attesa delle richieste "Wi-Fi probes", valori validi 1,6,11) */
        81,/*LISTEN_REGULATORY_CLASS;*/
         6,/*OPRA_CHANNEL (usato in realt dal GROUP-OWNER)*/
        81 /*OPRA_REGULATORY_CLASS (idem)*/
    };

    ret = sl_WlanSet(SL_WLAN_CFG_P2P_PARAM_ID, SL_WLAN_P2P_OPT_CHANNEL_N_REGS, sizeof(channels), channels);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione LISTEN_CHANNEL OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione LISTEN_CHANNEL fallita, codice %d", ret);
        return (FALSE);
    }

    /* Imposto URN (nome della rete Wi-Fi, non pi di 32 caratteri) */

    const char devURN[] = "RDGxxxx";

    ret = sl_NetAppSet(SL_NETAPP_DEVICE_ID, SL_NETAPP_DEVICE_URN, strlen(devURN), (uint8_t*)devURN);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione URN : %s OK!", devURN);
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione URN fallita, codice %d", ret);
        return (FALSE);
    }

    /* Impostazione ruolo Wi-Fi Direct */

    ret = sl_WlanSetMode(ROLE_P2P);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione ruolo Wi-Fi Direct OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione ruolo Wi-Fi Direct fallita, codice %d", ret);
        return (FALSE);
    }

    /* Abilitazione DHCP */

    ret = sl_NetCfgSet(SL_NETCFG_IPV4_STA_ADDR_MODE, SL_NETCFG_ADDR_DHCP, 0, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Abilitazione DHCP OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Abilitazione DHCP fallita, codice %d", ret);
        return (FALSE);
    }

    /* Disabilito i servizi mDNS */

    ret = sl_NetAppMDNSUnRegisterService(0, 0, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Disabilitazione servizi mDNS OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Impostazione servizi mDNS fallita, codice %d", ret);
        return (FALSE);
    }

    /* Fermo Network Processor */

    ret = sl_Stop(SL_STOP_TIMEOUT);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[RDG CONFIG] Fermo Network Processor OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Fermo di Network Processor fallito, codice %d", ret);
        return (FALSE);
    }

    /* Riavvio Network Processor */

    ret = sl_Start(0, 0, 0);

    if(ret == ROLE_P2P)
    {
        UART_PRINT("\r\n[RDG CONFIG] Riavvio Network Processor OK!");
    }
    else
    {
        UART_PRINT("\r\n[RDG CONFIG] Riavvio Network Processor fallito, codice %d", ret);
        return (FALSE);
    }

    return (TRUE);
}

/* Ciclo Connessione/Dis Wi-Fi */

void *Ciclo_ConnettiDisconnetti(void *arg)
{
    for(;;)
    {
        UART_PRINT("\n\r----------------------------------------------------------");

        /* Inizializzo semaforo disconnessione */

        sem_init(&Rdg.semDisconnect, 0, 0);

        /* Recupero e stampo indirizzo MAC */

        uint8_t     macAddress[6];
        uint16_t    macAddressLen = 6;

        int32_t ret = sl_NetCfgGet(SL_NETCFG_MAC_ADDRESS_GET, 0, &macAddressLen, &macAddress[0]);

        UART_PRINT("\n\r[RDG  CICLO] MAC NWP : %02x:%02x:%02x:%02x:%02x:%02x", macAddress[0], macAddress[1], macAddress[2], macAddress[3], macAddress[4], macAddress[5]);

        /* Connetto NWP alla rete Wi-Fi */

        UART_PRINT("\n\r[RDG  CICLO] Connetto NWP alla rete Wi-Fi (sl_WlanConnect)");

        SlWlanSecParams_t secParams = { .Key = "", .KeyLen = 0, .Type = SL_WLAN_SEC_TYPE_P2P_PBC };
        const char dummyRemote[] = "StartScan";
        ret = sl_WlanConnect((int8_t*)dummyRemote, strlen(dummyRemote), 0, &secParams, 0);

        if(ret < 0)
        {
            SHOW_WARNING(ret, WLAN_ERROR);
            return (NULL);
        }

        /* Attendo la negoziazione ... */

        UART_PRINT("\n\r[RDG  CICLO] Attendo la negoziazione ...");

        ret = sem_wait(&app_CB.P2P_CB.RcvNegReq);

        if(ret < 0)
        {
            SHOW_WARNING(ret, OS_ERROR);
            return (NULL);
        }

        /* Connetto a ??? */

        UART_PRINT("\n\r[RDG  CICLO] Connetto a '%s'", app_CB.P2P_CB.p2pPeerDeviceName);
        ret = sl_WlanConnect((signed char*)&app_CB.P2P_CB.p2pPeerDeviceName[0], strlen((char*)&app_CB.P2P_CB.p2pPeerDeviceName[0]), 0, &secParams, 0);

        if(ret < 0)
        {
            /* Connessione fallita */
            UART_PRINT("\n\r[RDG  CICLO] sl_WlanConnect('%s') fallita, returns %d", app_CB.P2P_CB.p2pPeerDeviceName, ret);

            /* Riavvio il Processo */
            UART_PRINT("\n\r[RDG CICLO] Riavvio");
            continue;
        }

        /* Attendo la conferma di connessione ... */

        long int dsec = 30;
        UART_PRINT("\n\r[RDG  CICLO] Attendo la conferma di connesione per %d secondi ...", dsec);

        struct timespec time;
        ret = clock_gettime(CLOCK_REALTIME, &time);

        if(ret < 0)
        {
            /* Get time fallito */
            UART_PRINT("\r\n[RDG  CICLO] clock_gettime fallito, returns %d", ret);

            /* Impossibile Continuare */
            UART_PRINT("\r\n[RDG  CICLO] IMPOSSIBILE CONTINUARE");
            for(;;) {}
        }

        time.tv_sec += dsec;
        ret = sem_timedwait(&app_CB.P2P_CB.RcvConReq, &time);

        if(ret < 0)
        {
            /* Connessione fallita */
            UART_PRINT("\n\r[RDG  CICLO] Connessione non confermata");

            /* Riavvio il Processo */
            UART_PRINT("\n\r[RDG  CICLO] Riavvio");
            continue;
        }

        /* Connesso! */

        UART_PRINT("\n\r[RDG  CICLO] Connesso!");

        /* Attendo rilascio IP */

        UART_PRINT("\n\r[RDG  CICLO] Attendo rilascio IP");

        ret = sem_wait(&app_CB.CON_CB.ip4acquireEventSyncObj);

        if(ret < 0)
        {
            SHOW_WARNING(ret, OS_ERROR);
            return (NULL);
        }

        /* Inizializzo la struttura RDGxxxx */

        Rdg.cntReadenUsPacket = 0;
        Rdg.cntWrittenUsPacket = 0;
        Rdg.isPcReqReady = false;

        int semInitResult = sem_init(&Rdg.fpgaAnswReady, 0, 0);

        if(semInitResult != 0)
        {
            /* Inizializzazione Semaforo Fallita */
            UART_PRINT("\r\n[RDG CICLO] sem_init(fpgaAnswReady) fallito, returns %d", semInitResult);

            /* Impossibile Continuare */
            UART_PRINT("\r\n[RDG CICLO] IMPOSSIBILE CONTINUARE");
            for(;;) {}
        }

        /* Avvio thread TrasmettiFlussoUltrasuoni */

        UART_PRINT("\n\r[RDG  CICLO] Avvio thread Tx_WiFi_UsPacket");

        pthread_t           pThread_TrasmettiFlussoUltrasuoni = (pthread_t)NULL;
        pthread_attr_t      pAttrs_myThread;
        struct sched_param  pParams_myThread;

        pthread_attr_init(&pAttrs_myThread);
        pParams_myThread.sched_priority = 2; /* PRIORITY */
        ret = pthread_attr_setschedparam(&pAttrs_myThread, &pParams_myThread);
        ret |= pthread_attr_setstacksize(&pAttrs_myThread, 2048 /* STACK-SIZE */);

        //ASSERT_ON_ERROR(ret, OS_ERROR);

        ret = pthread_create(&pThread_TrasmettiFlussoUltrasuoni, &pAttrs_myThread, Us_Task, NULL);
        //ASSERT_ON_ERROR(ret, OS_ERROR);

        /* Avvio thread RiceviRichiestePcElaboraTrasmettiRisposte */

        pthread_t           pThread_RiceviRichiestePcElaboraTrasmettiRisposte = (pthread_t)NULL;
        pthread_attr_t      pAttrs_myThread2;
        struct sched_param  pParams_myThread2;

        pthread_attr_init(&pAttrs_myThread2);
        pParams_myThread2.sched_priority = 2; /* PRIORITY */
        ret = pthread_attr_setschedparam(&pAttrs_myThread2, &pParams_myThread2);
        ret |= pthread_attr_setstacksize(&pAttrs_myThread2, 2048 /* STACK-SIZE */);
        ret = pthread_create(&pThread_RiceviRichiestePcElaboraTrasmettiRisposte, &pAttrs_myThread2, Msg_Task, NULL);

        /* Attendo evento disconnesione ... */

        sem_wait(&Rdg.semDisconnect);

        /* Cancello threads */

        ret = pthread_cancel(pThread_TrasmettiFlussoUltrasuoni);
        ret = pthread_cancel(pThread_RiceviRichiestePcElaboraTrasmettiRisposte);

        /* Chiudo sockets */

        ret = sl_Close(Rdg.tcpSmpSock);
        ret = sl_Close(Rdg.newSmpSock);
        ret = sl_Close(Rdg.tcpMsgSock);
        ret = sl_Close(Rdg.newMsgSock);
    }
}

/* Gestisce la comunicazione FPGA tramite SPI */

void *GestioneSpi(void *arg)
{
    /* Inizializzo Variabili */

    Rdg.cntReadenUsPacket = 0;    /* Puntatore Lettura   Pacchetto US = 0     */
    Rdg.cntWrittenUsPacket = 0;   /* Puntatore Scrittura Pacchetto US = 0     */
    Rdg.isPcReqReady = false;     /* Richiesta PC pronta              = FALSO */

    /* Inizializzo Semafori */

    int semInitResult = sem_init(&Rdg.fpgaAnswReady, 0, 0);

    if(semInitResult != 0)
    {
        UART_PRINT("\n\r[RDG] sem_init(fpgaAnswReady) failed, %d\n\r", semInitResult);
        return(NULL);
    }

    /* Impostazioni SPI */

    SPI_Params spi_Params;
    SPI_Params_init(&spi_Params);
    spi_Params.dataSize = 32; /* Massima dimensione WORD per massimizzare il throughput */
    spi_Params.frameFormat = SPI_POL0_PHA1;
    spi_Params.bitRate = 20e6;

    /* Apro la porta SPI */

    SPI_Handle spi_Handle = SPI_open(CONFIG_SPI_0, &spi_Params);

    if(spi_Handle==NULL)
    {
        /* Handle Error */
        UART_PRINT("\n\r'RDGxxxx_SpiThread' esecuzione di 'SPI_open' non riuscita\n\r");
        return(NULL);
    }

    /* Configuro direzione GPIO */

    GPIO_setConfig(CONFIG_GPIO_BUTTON_0, GPIO_CFG_INPUT);
    GPIO_setConfig(CONFIG_GPIO_LED_0, GPIO_CFG_OUTPUT);
    GPIO_setConfig(CONFIG_GPIO_LED_1, GPIO_CFG_OUTPUT);

    /* Gestione SPI */

    for(;;)
    {
        /* (L'FPGA ha prodotto pacchetti US) && (C' spazio per un nuovo pacchetto US da FPGA) ? */
        if((GPIO_read(CONFIG_GPIO_BUTTON_0) == 1) && ((uint8_t)(Rdg.cntWrittenUsPacket - Rdg.cntReadenUsPacket) < NUM_BUFFERS))
        {
            /* Chiedo pacchetto US all'FPGA */
            GPIO_write(CONFIG_GPIO_LED_0, 0);

            /* Preparo la transazione FPGA => SPI => fpgaUsPackets */
            SPI_Transaction spi_TransactionUsPacket;
            spi_TransactionUsPacket.count = SIZE_BUFFERS/4; /* SPI_Params::dataSize = 32-bit */
            spi_TransactionUsPacket.txBuf = NULL;
            spi_TransactionUsPacket.rxBuf = (void*)Rdg.fpgaUsPackets[Rdg.cntWrittenUsPacket % (NUM_BUFFERS - 1)];

            /* Eseguo la transazione */
            bool isTransferUsPacketOk = SPI_transfer(spi_Handle, &spi_TransactionUsPacket);

            if(isTransferUsPacketOk == false)
            {
                /* Handle Error */
                UART_PRINT("\n\r'RDGxxxx_SpiThread' esecuzione di 'SPI_transfer' non riuscita\n\r");
                for(;;) {};
            }

            /* Il pacchetto US da FPGA  stato scritto */
            Rdg.cntWrittenUsPacket++;

            /* Annullo richiesta pacchetto US all'FPGA */
            GPIO_write(CONFIG_GPIO_LED_0, 1);
        }
        else
        {
            /* Arrivata richiesta da PC ? */
            if(Rdg.isPcReqReady == true)
            {
                /* Segnalo trasmissione richiesta PC all'FPGA */
                GPIO_write(CONFIG_GPIO_LED_1, 0);

                /* Preparo la transazione pcReqPacket => SPI => FPGA */
                SPI_Transaction spi_TransactionReq;
                spi_TransactionReq.count = Rdg.pcReqSize/4; /* SPI_Params::dataSize = 32-bit */
                spi_TransactionReq.txBuf = (void*)Rdg.pcReqPacket;
                spi_TransactionReq.rxBuf = NULL;

                /* Eseguo la transazione */
                bool isTransferReqOk = SPI_transfer(spi_Handle, &spi_TransactionReq);

                if(isTransferReqOk == false)
                {
                    /* Handle Error */
                    UART_PRINT("\n\r'RDGxxxx_SpiThread' esecuzione di 'SPI_transfer' non riuscita\n\r");
                    for(;;) {};
                }

                /* La richiesta PC  stata consumata */
                Rdg.isPcReqReady = false;

                /* Preparo la transazione FPGA => SPI => fpgaAnswPacket */
                SPI_Transaction spi_TransactionAnsw;
                spi_TransactionAnsw.count = Rdg.fpgaAnswSize/4; /* SPI_Params::dataSize = 32-bit */
                spi_TransactionAnsw.txBuf = NULL;
                spi_TransactionAnsw.rxBuf = (void*)Rdg.fpgaAnswPacket;

                /* Eseguo la transazione */
                bool isTransferAnswOk = SPI_transfer(spi_Handle, &spi_TransactionAnsw);

                if(isTransferAnswOk == false)
                {
                    /* Handle Error */
                    UART_PRINT("\n\r'RDGxxxx_SpiThread' esecuzione di 'SPI_transfer' non riuscita\n\r");
                    for(;;) {};
                }

                /* La risposta FPGA  stata prodotta */
                int semPostResult = sem_post(&Rdg.fpgaAnswReady);

                if(semPostResult < 0)
                {
                    UART_PRINT("\r\nFile '%s' line '%d' semPostResult = '%d'\n\r", __FILE__, __LINE__, semPostResult);
                    for(;;) {};
                }

                /* Annullo seganlazione richiesta PC all'FPGA */
                GPIO_write(CONFIG_GPIO_LED_1, 1);
            }
            else
            {
                /* Lascio riposare il Thread */
                usleep(100);
            }
        }
    }
}

/* Gestione campioni Us */

void *Us_Task(void *arg)
{
    /* Creo socket TCP, lego, ascolto */

    Rdg.tcpSmpSock = CreateTcpSock_Bind_Listen(50001);

    if(Rdg.tcpSmpSock >= 0)
    {
        for(;;)
        {
            /* Accetto connessione TCP */
            Rdg.newSmpSock = Accept(Rdg.tcpSmpSock);

            if(Rdg.newSmpSock >= 0)
            {
                /* Trasmissione Us */
                Us_Send(Rdg.newSmpSock);
            }
            else
            {
                break;
            }
        }
    }
    else
    {
    }

    /* Chiudo sock */

    int32_t
    ret = sl_Close(Rdg.tcpSmpSock);

    return NULL;
}

/* Trasmissione campioni Us */

void Us_Send(int16_t sock)
{
    for(;;)
    {
        /* Il Thread SPI ha prodotto un pacchetto ? */
       if(true)//(uint8_t)(Rdg.cntWrittenUsPacket - Rdg.cntReadenUsPacket) > 0)
        {
            /* Lo spedisco al PC */
            int16_t ret = sl_Send(sock, Rdg.fpgaUsPackets[Rdg.cntReadenUsPacket % (NUM_BUFFERS - 1)], SIZE_BUFFERS, 0);//To(sock, RDGxxxxStructInstance.fpgaUsPackets[RDGxxxxStructInstance.cntReadenUsPacket % (NUM_BUFFERS - 1)], SIZE_BUFFERS, 0, (SlSockAddr_t*)&sAddr, (SlSocklen_t)sizeof(SlSockAddrIn_t));

            if(ret == 0)
            {
                UART_PRINT("\r\n[US    SEND] Connessione interrotta");
                break;
            }
            else if(ret < 0)
            {
                UART_PRINT("\r\n[US    SEND] sl_Recv failed, return %d", ret);
                for(;;)
                {
                    sleep(1); /* breakkare anche qui */
                };
            }

            /* Incremento il contatore di lettura */
            Rdg.cntReadenUsPacket ++;
        }
        else
        {
            /* Lascio riposare il Thread */
            usleep(100);
        }
    }

    /* Chiudo sock */

    int32_t
    ret = sl_Close(sock);
}

/* Gestione Messaggi */

void *Msg_Task(void *arg)
{
    /* Creo socket TCP, lego, ascolto */
    Rdg.tcpMsgSock = CreateTcpSock_Bind_Listen(50002);

    if(Rdg.tcpMsgSock >= 0)
    {
        for(;;)
        {
            /* Accetto connessione TCP */
            Rdg.newMsgSock = Accept(Rdg.tcpMsgSock);

            if(Rdg.newMsgSock >= 0)
            {
                /* Ricezione/Trasmissione Msg */
                Msg_RecvSend(Rdg.newMsgSock);
            }
            else
            {
                break;
            }
        }
    }
    else
    {

    }

    /* Chiudo sock */

    int32_t
    ret = sl_Close(Rdg.tcpMsgSock);

    return NULL;
}

/* Ricezione/Trasmissione Msg */

void Msg_RecvSend(int16_t sock)
{
    for(;;)
    {
        /* Attendo la richiesta da PC */

        /*SlSocklen_t sockRxLen = (SlSocklen_t)sizeof(SlSockAddrIn_t);*/
        /*sAddr.sin_addr.s_addr = sl_Htonl(app_CB.CON_CB.GatewayIP); Ricevo dal PC (GateWay) */
        int32_t ret = sl_Recv(sock, Rdg.pcReqPacket, sizeof(Rdg.pcReqPacket), 0);/*From(sockRx, RDGxxxxStructInstance.pcReqPacket, sizeof(RDGxxxxStructInstance.pcReqPacket), 0, (SlSockAddr_t*)&sAddr, &sockRxLen);*/

        if(ret == 0)
        {
            UART_PRINT("\r\n[MSG RCVSND] Connessione interrotta");
            break;
        }
        else if(ret < 0)
        {
            UART_PRINT("\r\n[MSG RCVSND] sl_Recv failed, return %d", ret);
            for(;;)
            {
                sleep(1);
            };
        }

        /* Eseguo azioni personalizzate in base al primo byte */
        switch(Rdg.pcReqPacket[0])
        {
            case 0: // Custom Ping
            {
                UART_PRINT("\r\n[RDG] Ricevuto 0x00, 0x%x", Rdg.pcReqPacket[1]);
                Rdg.fpgaAnswPacket[0] = 0;
                Rdg.fpgaAnswPacket[1] = Rdg.pcReqPacket[1];
                Rdg.fpgaAnswSize = 2;
                break;
            }

            default:
            {
                /* Imposto i campi "dimensione richiesta da PC" e "dimensione risposta FPGA" */

                Rdg.pcReqSize = (uint16_t)ret;
                Rdg.fpgaAnswSize = 16; /* TODO : calcolare in base al protocollo */

                /* Inizializzo il semaforo "Risposta FPGA pronta" */

                int semInitResult = sem_init(&Rdg.fpgaAnswReady, 0, 0);

                if(semInitResult != 0)
                {
                    UART_PRINT("\r\nFile '%s' line '%d' semInitResult = '%d'\n\r", __FILE__, __LINE__, semInitResult);
                    for(;;) {};
                }

                /* La richiesta PC  pronta */

                Rdg.isPcReqReady = true;

                /* Attendo la risposta FPGA */

                int semWaitResult = sem_wait(&Rdg.fpgaAnswReady);

                if(semWaitResult != 0)
                {
                    UART_PRINT("\r\nFile '%s' line '%d' semWaitResult = '%d'\n\r", __FILE__, __LINE__, semWaitResult);
                    for(;;) {};
                }

                break;
            }
        }

        /* Spedisco la risposta a PC */

        ret = sl_Send(sock, Rdg.fpgaAnswPacket, Rdg.fpgaAnswSize, 0);/*To(sockRx, RDGxxxxStructInstance.fpgaAnswPacket, RDGxxxxStructInstance.fpgaAnswSize, 0, (SlSockAddr_t*)&sAddr, (SlSocklen_t)sizeof(SlSockAddrIn_t));*/

        if(ret == 0)
        {
            UART_PRINT("\r\n[MSG RCVSND] Connessione interrotta");
            break;
        }
        else if(ret < 0)
        {
            UART_PRINT("\r\n[MSG RCVSND] sl_Send failed, return %d", ret);
            for(;;)
            {
                sleep(1);
            };
        }
    }

    /* Chiudo sock */

    int32_t
    ret = sl_Close(sock);
}

/* Creo socket TCP, lego, ascolto */

int16_t CreateTcpSock_Bind_Listen(int tcpPort)
{
    /* Creo socket TCP */

    int16_t tcpSocket = sl_Socket(SL_AF_INET, SL_SOCK_STREAM, 0);

    if(tcpSocket >= 0)
    {
        UART_PRINT("\r\n[TCP SOCKET] Socket creation OK!");
    }
    else
    {
        UART_PRINT("\r\n[TCP SOCKET] Socket creation failed, %d", tcpSocket);
        return -1;
    }

    /* Imposto porta */

    SlSockAddrIn_t sAddr;
    sAddr.sin_family = SL_AF_INET;
    sAddr.sin_port = sl_Htons(tcpPort);
    sAddr.sin_addr.s_addr = sl_Htonl(SL_INADDR_ANY);

    /* Lego*/

    int32_t ret = sl_Bind(tcpSocket, (SlSockAddr_t*)&sAddr, sizeof(SlSockAddrIn_t));

    if(ret >= 0)
    {
        UART_PRINT("\r\n[TCP SOCKET] Bind socket to port %d OK!", sl_Ntohs(sAddr.sin_port));
    }
    else
    {
        UART_PRINT("\r\n[TCP SOCKET] Bind socket failed, %d", ret);
        return -1;
    }

    /* Ascolto */

    ret = sl_Listen(tcpSocket, 0);

    if(ret >= 0)
    {
        UART_PRINT("\r\n[TCP SOCKET] Listening ...");
    }
    else
    {
        UART_PRINT("\r\n[TCP SOCKET] Listening failed, %d", ret);
        return -1;
    }

    return tcpSocket;
}

/* Accetto connessione TCP */

int16_t Accept(int16_t tcpSocket)
{
    /* Accetto connessione TCP */

    SlSocklen_t newSockLen = 0;
    SlSockAddrIn_t sAddr = {};
    int16_t newSock = sl_Accept(tcpSocket, (SlSockAddr_t*)&sAddr, (SlSocklen_t*)&newSockLen);

    if(newSock >= 0)
    {
        UART_PRINT("\r\n[TCP SOCKET] Accept OK!");
    }
    else
    {
        UART_PRINT("\r\n[TCP SOCKET] Accept failed, %d", newSock);
        return -1;
    }

    return newSock;
}
