
/*
 * Modified from TI Application "SPI Demo"
 *
 * Stream SPI-slave data to ping-pong buffers with DMA
 */

//*****************************************************************************
//
// Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/ 
// 
// 
//  Redistribution and use in source and binary forms, with or without 
//  modification, are permitted provided that the following conditions 
//  are met:
//
//    Redistributions of source code must retain the above copyright 
//    notice, this list of conditions and the following disclaimer.
//
//    Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the 
//    documentation and/or other materials provided with the   
//    distribution.
//
//    Neither the name of Texas Instruments Incorporated nor the names of
//    its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
//  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
//  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

// Standard includes
#include <string.h>

// Driverlib includes
#include "hw_types.h"
#include "hw_memmap.h"
#include "hw_common_reg.h"
#include "hw_ints.h"
#include "spi.h"
#include "hw_mcspi.h"
#include "rom.h"
#include "rom_map.h"
#include "utils.h"
#include "prcm.h"
#include "uart.h"
#include "interrupt.h"
#include "osi.h"
#include "udma.h"

// Common interface includes
#include "uart_if.h"
#include "gpio_if.h"
#include "udma_if.h"
//#include "pinmux.h"

#include "common.h"

// BIOS Events
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Event.h>
#include <xdc/runtime/Error.h>

Event_Handle spiEventHandle ;


#define SPI_IF_BIT_RATE  100000		// not meaningful in slave mode
#define TR_BUFF_SIZE     1024

#define	UDMA_ARB	UDMA_ARB_8		// number of transfers per DMA, which are 16 bits, so must be half of AFL
#define AFL	16		// almost full level for RX FIFO.  Must read this many bytes each INT
					// 64-byte FIFO, each 16-bit SPI word occupies 4 bytes (swru367b, table 8-5)
					// All FIFO dedicated to read, 32 words max

#define	MYDMACHANNEL	UDMA_CH6_GSPI_RX	// UDMA_CH30_GSPI_RX
#define TXCHANNEL		UDMA_CH7_GSPI_TX

//*****************************************************************************
//                 GLOBAL VARIABLES -- Start
//*****************************************************************************
//unsigned char g_ucABuff[TR_BUFF_SIZE], g_ucBBuff[TR_BUFF_SIZE];
uint16_t g_usRxBufA[TR_BUFF_SIZE], g_usRxBufB[TR_BUFF_SIZE];
uint16_t g_usTxBufA[TR_BUFF_SIZE], g_usTxBufB[TR_BUFF_SIZE];
uint32_t g_ulRxBufACount=0, g_ulRxBufBCount=0;
uint32_t g_ulTxBufACount=0, g_ulTxBufBCount=0;

unsigned char spiAB=1;
static unsigned int ucRxBuffNdx = 0;

#if defined(ccs)
extern void (* const g_pfnVectors[])(void);
#endif
#if defined(ewarm)
extern uVectorEntry __vector_table;
#endif
//*****************************************************************************
//                 GLOBAL VARIABLES -- End
//*****************************************************************************



//*****************************************************************************
//!
//! The interrupt handler for GSPI.  This interrupt will occur when a DMA
//! transfer is complete using the GSPI uDMA channel.This interrupt handler will
//! switch between receive ping-pong buffers A and B.
//!
//! \param None
//!
//! \return None
//*****************************************************************************
int overFlowCounter=0;
int syncErrorA=0, syncErrorB=0;
int slaveInts=0;
int slaveIntFalse=0;
uint32_t lastWrongStatus=0, lastWrongTXStatus = 0x1245cccc, lastOtherStatus=0xdeadbeef;
void
SlaveIntHandler(void)
{
    unsigned long ulStatus;
    unsigned long ulModeP, ulModeA, ulModeTXP, ulModeTXA;
    static int boo = 0;
    char	postEvent = 0;

    slaveInts++;

    //
    // Read the interrupt status of the GSPI.
    //
    ulStatus = MAP_SPIIntStatus(GSPI_BASE, true);	// ask for masked interrupts


    if(ulStatus & ( SPI_INT_RX_OVRFLOW))
    {
    	overFlowCounter++;
    }

    //
    // Clear any pending status
    //
    MAP_SPIIntClear(GSPI_BASE, ulStatus);

    //
    // Check the DMA control table to see if the ping-pong "A" transfer is
    // complete.  The "A" transfer uses receive buffer "A", and the primary
    // control structure.
    //
    ulModeP = MAP_uDMAChannelModeGet(MYDMACHANNEL | UDMA_PRI_SELECT);

    //
    // If the primary control structure indicates stop, that means the "A"
    // receive buffer is done.  The uDMA controller should still be receiving
    // data into the "B" buffer.
    //
    if(ulModeP == UDMA_MODE_STOP)
    {
        //
        // Increment a counter to indicate data was received into buffer A.
        //
        g_ulRxBufACount++;

        //
        // Set up the next transfer for the "A" buffer, using the primary
        // control structure.  When the ongoing receive into the "B" buffer is
        // done, the uDMA controller will switch back to this one.
        //

        MAP_uDMAChannelTransferSet( MYDMACHANNEL | UDMA_PRI_SELECT, UDMA_MODE_PINGPONG,
                (void *)(GSPI_BASE + MCSPI_O_RX0), g_usRxBufA,
                sizeof(g_usRxBufA)/sizeof(uint16_t) );

        if (spiAB == 1) syncErrorA++;
        spiAB = 1;	// buffer A is ready
        postEvent = 1;
    }

    //
    // Check the DMA control table to see if the ping-pong "B" transfer is
    // complete.  The "B" transfer uses receive buffer "B", and the alternate
    // control structure.
    //
    ulModeA = MAP_uDMAChannelModeGet(MYDMACHANNEL | UDMA_ALT_SELECT);

    //
    // If the alternate control structure indicates stop, that means the "B"
    // receive buffer is done.  The uDMA controller should still be receiving
    // data into the "A" buffer.
    //
    if(ulModeA == UDMA_MODE_STOP)
    {
        //
        // Increment a counter to indicate data was received into buffer A.
        //
        g_ulRxBufBCount++;

        //
        // Set up the next transfer for the "B" buffer, using the alternate
        // control structure.  When the ongoing receive into the "A" buffer is
        // done, the uDMA controller will switch back to this one.
        //

        MAP_uDMAChannelTransferSet( MYDMACHANNEL | UDMA_ALT_SELECT, UDMA_MODE_PINGPONG,
                (void *)(GSPI_BASE + MCSPI_O_RX0), g_usRxBufB,
                sizeof(g_usRxBufB)/sizeof(uint16_t) );

        if (spiAB == 0) syncErrorB++;
        spiAB = 0;	// buffer B is ready
        postEvent = 1;

    }

    /*
     * Quick hacks to get TX buffers for Master mode demonstration
     */

    ulModeTXP = MAP_uDMAChannelModeGet(TXCHANNEL | UDMA_PRI_SELECT);

    if(ulModeTXP == UDMA_MODE_STOP)
    {
        g_ulTxBufACount++;

        MAP_uDMAChannelTransferSet( TXCHANNEL | UDMA_PRI_SELECT, UDMA_MODE_PINGPONG,
                g_usTxBufA, (void *)(GSPI_BASE + MCSPI_O_TX0),
                sizeof(g_usTxBufA)/sizeof(uint16_t) );

//        postEvent = 1;
    }

    ulModeTXA = MAP_uDMAChannelModeGet(TXCHANNEL | UDMA_ALT_SELECT);

    if(ulModeTXA == UDMA_MODE_STOP)
    {
        g_ulTxBufBCount++;

        MAP_uDMAChannelTransferSet( TXCHANNEL | UDMA_ALT_SELECT, UDMA_MODE_PINGPONG,
                g_usTxBufB, (void *)(GSPI_BASE + MCSPI_O_TX0),
                sizeof(g_usTxBufB)/sizeof(uint16_t) );

//        postEvent = 1;

    }

    if ((ulModeA != UDMA_MODE_STOP) && (ulModeP != UDMA_MODE_STOP)
    		&& (ulModeTXA != UDMA_MODE_STOP) && (ulModeTXP != UDMA_MODE_STOP))
    	slaveIntFalse++;	// this was a false interrupt

    // post a semaphore
	if (postEvent) {
		Event_post(spiEventHandle, Event_Id_00);

		// toggle GPIO LED
		//GPIO_IF_LedToggle(LED1);
		if (boo)
			GPIO_IF_LedOn(LED1);
		else
			GPIO_IF_LedOff(LED1);
		boo ^= 1;
	} else if (SPI_INT_DMARX & ulStatus)
		lastWrongStatus = ulStatus; //MAP_SPIIntStatus(GSPI_BASE, true);	// remember this for debugging purposes
	else if (SPI_INT_DMATX & ulStatus)
		lastWrongTXStatus = ulStatus;
	else
		lastOtherStatus = ulStatus;

}


//*****************************************************************************
//
//! Initializes the GSPI peripheral and sets up the RX uDMA channels.
//! The uDMA RX channel
//! will receive any incoming data into a pair of buffers in ping-pong mode.
//!
//! \param None
//!
//! \return None
//!
//*****************************************************************************
static void
InitGSPITransfer(void)
{

	UDMAInit();

	//
	// Activate uDMA for GPSI
	//
	MAP_uDMAChannelAssign(MYDMACHANNEL);

	/*
	 * Make this channel high priority
	 */
	MAP_uDMAChannelAttributeEnable(MYDMACHANNEL, UDMA_ATTR_HIGH_PRIORITY );


    //
    // Configure the control parameters for the GSPI RX.  The uDMA GPSI RX
    // channel is used to transfer a block of data from GSPI to a buffer.
    // The data size is 8 bits.  The destination address increment is 8-bit bytes
    // since the data is going to a buffer.  The source increment is
    // none since the data is to be read from the GSPI data register.  The
    // arbitration size is set to AFL (nominally 16), which matches the GSPI Rx FIFO trigger
    // threshold.
    //
    UDMASetupTransfer(MYDMACHANNEL | UDMA_PRI_SELECT, UDMA_MODE_PINGPONG,
            sizeof(g_usRxBufA)/sizeof(uint16_t), UDMA_SIZE_16, UDMA_ARB,
            (void *)(GSPI_BASE + MCSPI_O_RX0), UDMA_SRC_INC_NONE,
                                            g_usRxBufA, UDMA_DST_INC_16);

    UDMASetupTransfer(MYDMACHANNEL | UDMA_ALT_SELECT, UDMA_MODE_PINGPONG,
            sizeof(g_usRxBufB)/sizeof(uint16_t),UDMA_SIZE_16, UDMA_ARB,
              (void *)(GSPI_BASE + MCSPI_O_RX0), UDMA_SRC_INC_NONE,
                                            g_usRxBufB, UDMA_DST_INC_16);


    MAP_SPIDmaEnable(GSPI_BASE, SPI_RX_DMA);
}

static void
InitGSPITransferTX(void)
{
	//
	// Activate uDMA for GPSI
	//
	MAP_uDMAChannelAssign(TXCHANNEL);

	/*
	 * Make this channel high priority
	 */
	MAP_uDMAChannelAttributeEnable(TXCHANNEL, UDMA_ATTR_HIGH_PRIORITY );


    //
    // Configure the control parameters for the GSPI RX.  The uDMA GPSI RX
    // channel is used to transfer a block of data from GSPI to a buffer.
    // The data size is 8 bits.  The destination address increment is 8-bit bytes
    // since the data is going to a buffer.  The source increment is
    // none since the data is to be read from the GSPI data register.  The
    // arbitration size is set to AFL (nominally 16), which matches the GSPI Rx FIFO trigger
    // threshold.
    //
    UDMASetupTransfer(TXCHANNEL | UDMA_PRI_SELECT, UDMA_MODE_PINGPONG,
            sizeof(g_usTxBufA)/sizeof(uint16_t), UDMA_SIZE_16, UDMA_ARB,
            g_usTxBufA, UDMA_SRC_INC_16,
				(void *)(GSPI_BASE + MCSPI_O_TX0), UDMA_DST_INC_NONE);

    UDMASetupTransfer(TXCHANNEL | UDMA_ALT_SELECT, UDMA_MODE_PINGPONG,
            sizeof(g_usTxBufB)/sizeof(uint16_t),UDMA_SIZE_16, UDMA_ARB,
            g_usTxBufB, UDMA_SRC_INC_16,
				(void *)(GSPI_BASE + MCSPI_O_TX0), UDMA_DST_INC_NONE);


    MAP_SPIDmaEnable(GSPI_BASE, SPI_TX_DMA);
}

//*****************************************************************************
//
//! SPI Slave mode main loop
//!
//! This function configures SPI module as slave and enables the channel for
//! communication
//!
//! \return None.
//
//*****************************************************************************
void SlaveMain()
{
	Error_Block eb;
	int i;

    //
    // Fill TX buffer, and clear RX buffer
    //
	memset(g_usRxBufA, 0, sizeof(g_usRxBufA));
	memset(g_usRxBufB, 0, sizeof(g_usRxBufA));

	for (i=0; i<TR_BUFF_SIZE; i++)
		g_usTxBufA[i] = i;
	for (i=0; i<TR_BUFF_SIZE; i++)
		g_usTxBufB[i] = i+TR_BUFF_SIZE;

    //
    // Reset SPI
    //
    MAP_SPIReset(GSPI_BASE);

    //
    // Configure SPI interface
    //
    MAP_SPIConfigSetExpClk(GSPI_BASE,MAP_PRCMPeripheralClockGet(PRCM_GSPI),
                     SPI_IF_BIT_RATE,SPI_MODE_MASTER,SPI_SUB_MODE_0,
                     (SPI_HW_CTRL_CS |
                     SPI_4PIN_MODE |
                     SPI_TURBO_OFF |
                     SPI_CS_ACTIVELOW |
                     SPI_WL_16));

    /*
     * Enable RX FIFO
     */
    MAP_SPIFIFOEnable(GSPI_BASE,SPI_RX_FIFO);
    MAP_SPIFIFOEnable(GSPI_BASE,SPI_TX_FIFO);
    MAP_SPIFIFOLevelSet(GSPI_BASE,AFL,AFL);

    InitGSPITransfer();
    InitGSPITransferTX();
    //
    // Register Interrupt Handler
    //
//    MAP_SPIIntRegister(GSPI_BASE,SlaveIntHandler);
    //
    // Enable the GSPI peripheral interrupts. uDMA controller will cause an
    // interrupt on the GSPI interrupt signal when a uDMA transfer is complete.
    //
    if (0 > osi_InterruptRegister(INT_GSPI, SlaveIntHandler, 0x80))
    	Message("SPI Slave setup failed\n\r");

    else {

		//
		// Enable Interrupts
		//
		MAP_SPIIntEnable(GSPI_BASE, SPI_INT_DMATX | SPI_INT_DMARX | SPI_INT_RX_OVRFLOW);

		//
		// Enable SPI for communication
		//
		MAP_SPIEnable(GSPI_BASE);

		//
		// Print mode on uart
		//
		Message("Enabled SPI Interface in Master Mode\n\r");
    }

    // Create event object to signal waiting tasks
    spiEventHandle = Event_create(NULL, &eb);
    if (spiEventHandle == NULL) {
    	Message("Event create failed.\n\r");

    }

#if 1
// Testing
    UInt 			events;
    int counter;
    for (counter=0; counter < 10; counter++) {
		events = Event_pend(spiEventHandle, Event_Id_NONE, ~Event_Id_NONE, BIOS_WAIT_FOREVER);
    }
		UART_PRINT("UDMA job complete %d, number of ints received %d, lastWrongStatus %08x, lastWrongTXStatus %08x, lastOtherStatus %08x\n\r",
				counter, slaveInts, lastWrongStatus, lastWrongTXStatus, lastOtherStatus);

		UART_PRINT("Rx count Primary: %d, Alternate %d\n\r", g_ulRxBufACount, g_ulRxBufBCount);
		UART_PRINT("Tx count Primary: %d, Alternate %d\n\r", g_ulTxBufACount, g_ulTxBufBCount);
//		char *ptr = spiAB ? g_ucBBuff : g_ucABuff;


    osi_Sleep(BIOS_WAIT_FOREVER);
#endif

}



