/*-----------------------------------------------------------------------------
 * Copyright(c)2016 Allegion PLC as an unpublished work.
 *
 * The information contained herein is confidential property of Allegion PLC.
 * All rights reserved.  Reproduction, adaptation, or translation without the
 * express written consent of Allegion PLC is prohibited, except as allowed
 * under the copyright laws.
 *---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * File Name:       wifiplatformnet.c
 * Purpose of File: All wifi net APIs are defined here.
 *---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Platform Module    :   WiFi                                <wifiPlatform>
 * Feature Abstraction:   Net                                 <Net>
 *---------------------------------------------------------------------------*/

/*-------------- HEADER INCLUDE FILES ---------------------------------------*/
#include "wifiplatform.h"
#include "stdio.h"

/*-------------- LOCAL DEFINITIONS ------------------------------------------*/
#define TIME1970              2208988800 /* See CC3100 SDK get_time example */
#define NTP_BUF_SIZE          48      /* NTP Version 3 Packet Header Size   */
#define NTP_CLIENT_FLAGS_V3   0x1B    /* NTP Version 3 Client Mode          */
#define NTP_SERVER_MODE_MASK  0x07    /* NTP Version 3 Server Mode Mask     */
#define NTP_SERVER_MODE_FLAG  4       /* NTP Version 3 Server Mode          */
#define NTP_CLIENT_FLAGS_V4   0x23    /* NTP Version 4 Client Mode, LI = 0, VN = 4, Mode = 3 */

/*-------------- LOCAL MACRO FUNCTION DEFINITIONS ---------------------------*/

/*-------------- LOCAL FUNCTION PROTOTYPES ----------------------------------*/
static void wifiPingReport(SlNetAppPingReport_t *pPingReport);
	
/*-------------- GLOBAL VARIABLES -------------------------------------------*/
extern uint32_t g_GatewayIP;
boolean_t g_pingStatus = FALSE;
uint32_t g_PingPacketsRecv = 0;

/*-------------- WIFI API FUNCTION DEFINITIONS ------------------------------*/

/******************************************************************************
* Function Name : wifiPlatformNetCreateSocket
* Description   : Creates socket
* Arguments     :
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetCreateSocket(int16_t *skt, int16_t domain,
                                                 int16_t type, int16_t protocol)
{
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if (NULL == skt)
    {
        returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;  
    }

    if (WIFI_PLATFORM_STATUS_SUCCESS == returnStatus)
    {
        *skt = sl_Socket(domain, type, protocol);
        if(*skt < 0)
        {
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CREATE_FAILURE;
        }
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetSetSocketOption
* Description   : sets socket option
* Arguments     :
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiPlatformNetCreateSocket()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetSetSocketOption(int16_t sd, int16_t level,
                                                    int16_t optname,
                                                    const void *optval,
                                                    uint16_t optlen)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if((level == WIFI_PLATFORM_SOL_SOCKET) && 
       (optname == WIFI_PLATFORM_SO_RCVTIMEO))
    {
        uint32_t *socketTimeout = (uint32_t * )(optval);
        if(*socketTimeout == 0)
        {
            returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;
        }
        else
        {
            SlTimeval_t tTimeout;
            tTimeout.tv_sec = (*socketTimeout)/1000;
            tTimeout.tv_usec = (*socketTimeout)%1000 * 1000;
            optval = &tTimeout;
            optlen = sizeof(SlTimeval_t);
        }
        
    }
    if(returnStatus == WIFI_PLATFORM_STATUS_SUCCESS)
    {
        status = sl_SetSockOpt(sd, level, optname, optval, optlen);
        if(status != 0)
        {
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_MODECONFIG_FAILURE;
        }
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetConnect
* Description   : Creates and connects socket
* Arguments     : char *destinationUrl,
*                 uint8_t ip,
*                 boolean_t secured,
*                 uint16_t port,
*                 int16_t *skt,
*                 uint32_t sktTimeout,
*                 char *rootCertPath,
*                 char *deviceCertPath,
*                 char *privateKeyPath
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiWlanConnect()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetConnect(char *destinationUrl,
                                            uint32_t ip,
                                            boolean_t secured,
                                            uint16_t port,
                                            int16_t *skt,
                                            uint32_t sktTimeout,
                                            char *rootCertPath,
                                            char *deviceCertPath,
                                            char *privateKeyPath,
                                            int16_t *driverStatus)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;
    uint32_t             tmpIp = 0;
    int                  tmpSkt = -1;
    boolean_t            closeSocket = FALSE;
    SlSockAddrIn_t       address;
    SlSockSecureMethod_t method;
    SlSockSecureMask_t   mask = {0};
    int16_t              socketOpt = secured ? SL_SEC_SOCKET : 0;

    address.sin_family = SL_AF_INET;
    address.sin_port = sl_Htons(port);

    if(destinationUrl != NULL)
    {
        status = sl_NetAppDnsGetHostByName((signed char *)destinationUrl,
                                strlen(destinationUrl), &tmpIp, SL_AF_INET);
        address.sin_addr.s_addr = sl_Htonl(tmpIp);

        if(status != 0)
        {
            returnStatus = WIFI_PLATFORM_STATUS_NW_DNS_FAILURE;
        }
    }
    else if(ip != NULL)
    {
        status = 0;
        address.sin_addr.s_addr = sl_Htonl(ip);
    }
    else
    {
        returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;
    }

    if (NULL == skt)
    {
        returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;
    }

    if ((status == 0) && (returnStatus == WIFI_PLATFORM_STATUS_SUCCESS))
    {
        tmpSkt = sl_Socket(SL_AF_INET, SL_SOCK_STREAM, socketOpt);
        if (tmpSkt >= 0)
        {
            *skt = tmpSkt;
            if(sktTimeout > 0)
            {
                SlTimeval_t tTimeout;
                tTimeout.tv_sec = sktTimeout/1000;
                tTimeout.tv_usec = 0;

                status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_RCVTIMEO, &tTimeout,
                        sizeof(SlTimeval_t));
                if(status != 0)
                {
                    returnStatus = WIFI_PLATFORM_STATUS_SOCKET_MODECONFIG_FAILURE;
                }
            }
            if(secured)
            {
                method.SecureMethod = WIFI_PLATFORM_TLS_VERSION;
                mask.SecureMask = WIFI_PLATFORM_TLS_CIPHER_SUITE;

                if (status == 0)
                {
                    status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECMETHOD,
                            &method.SecureMethod, sizeof(method.SecureMethod));
                }
                if (status == 0)
                {
                    status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECURE_MASK,
                            &mask.SecureMask, sizeof(mask.SecureMask));
                }
                if((status == 0) && (rootCertPath != NULL))
                {
                        status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECURE_FILES_CA_FILE_NAME,
                                (const char *)rootCertPath, strlen((const char *)rootCertPath));
                }
                if((status == 0) && (deviceCertPath != NULL))
                {
                        status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECURE_FILES_CERTIFICATE_FILE_NAME,
                                (const char *)deviceCertPath, strlen((const char *)deviceCertPath));
                }
                if((status == 0) && (privateKeyPath != NULL))
                {
                        status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECURE_FILES_PRIVATE_KEY_FILE_NAME,
                                (const char *)privateKeyPath, strlen((const char *)privateKeyPath));
                }
                if((status == 0) && (destinationUrl != NULL))
                {
                        status = sl_SetSockOpt(tmpSkt, SL_SOL_SOCKET, SL_SO_SECURE_DOMAIN_NAME_VERIFICATION,
                                (const char *)destinationUrl, strlen((const char *)destinationUrl));
                }
                if(status != 0)
                {
                    returnStatus = WIFI_PLATFORM_STATUS_SOCKET_SECURITY_FAILURE;
                }
            }
            if (status == 0)
            {
                status = sl_Connect(tmpSkt, (SlSockAddr_t *)&address, sizeof(address));

                if (status < 0)
                {
                    returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CONNECT_FAILURE;
                    if (tmpSkt >= 0)
                    {
                        if(status == SL_ERROR_BSD_ESEC_ASN_SIG_CONFIRM_E)
                        {
                            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CERTVALIDATION_FAILURE;
                            closeSocket = TRUE;
                        }
                        else if(status == SL_ERROR_BSD_ESECDATEERROR)
                        {
#if (WIFI_PLATFORM_TLS_ALLOW_DATE_VALIDATION_FAILURE == 0)
                            /* Allow application to handle this error based on the 
                             platform configuration WIFI_PLATFORM_TLS_ALLOW_DATE_VALIDATION_FAILURE 
                             Do not close socket if WIFI_PLATFORM_TLS_ALLOW_DATE_VALIDATION_FAILURE
                             is set to 1 */
                            closeSocket = TRUE;
#endif
                            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_DATE_VALIDATION_FAILURE; 
                        }
                        else
                        {
                            closeSocket = TRUE; // Close socket for all other errors
                        }
                    }
                    
                }
      
            }
        }
        else
        {
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CREATE_FAILURE;
        }
    }
    else
    {
        // Do nothing, just return the returnStatus
    }

    if(closeSocket)
    { 
        sl_Close(tmpSkt);
        *skt = -1;
    }
    if(driverStatus)
    {
        *driverStatus = status;
    }
    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetRead
* Description   : Reads the requested bytes from the socket
* Arguments     : int16_t skt,uint8_t *buffer,uint32_t numBytes,
                  uint32_t *bytesRead,int16_t *sktStatus
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiWlanConnect(),wifiIpResolveName(),wifiSocketConnect()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetRead(int16_t skt,uint8_t *buffer,uint32_t numBytes,
                                         uint32_t *bytesRead,int16_t *sktStatus)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if (NULL != sktStatus)
    {
        *sktStatus = 0;
    }

    if (NULL != bytesRead)
    {
        *bytesRead = 0;
    }

    if (skt < 0)
    {
       returnStatus = WIFI_PLATFORM_STATUS_SOCKET_INVALID;
    }
    else
    {
        status = sl_Recv(skt, buffer, numBytes, 0);
        if( status <= 0 )
        {
            sl_Close(skt);
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_READ_FAILURE;
        }
        else
        {
            if (NULL != bytesRead)
            {
                *bytesRead = status;
            }
        }

        if(SL_ERROR_BSD_EAGAIN == status)
        {
            if (NULL != sktStatus)
            {
                *sktStatus = WIFI_PLATFORM_RETRY_SOCKET_OPT;
            }
        }
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetWrite
* Description   : Writes the requested bytes to the socket
* Arguments     : int16_t skt,uint8_t *buffer,uint32_t numBytes,
                  uint32_t *bytesSent
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiWlanConnect(),wifiIpResolveName(),wifiSocketConnect()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetWrite(int16_t skt,uint8_t *buffer,int16_t numBytes,
                                          int16_t flags,uint32_t *bytesSent)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if(bytesSent != NULL)
    {
        *bytesSent = 0;
    }
    if (skt < 0)
    {
       returnStatus = WIFI_PLATFORM_STATUS_SOCKET_INVALID;
    }
    else
    {
        status = sl_Send(skt, buffer, numBytes, flags);
        if( status < 0 )
        {
            sl_Close(skt);
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_WRITE_FAILURE;
        }
        else
        {
            if(bytesSent != NULL)
            {
                *bytesSent = status;
            }
        }
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetsClose
* Description   : Close socket connection
* Arguments     : int16_t *skt
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiWlanConnect(),wifiIpResolveName(),wifiSocketConnect()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetClose(int16_t skt)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if (skt < 0)	
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_INVALID;
    }
    else
    {
        status = sl_Close(skt);
        if(status < 0)
        {
            returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CLOSE_FAILURE;
        }
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetGetIpFromName
* Description   : returns an IP from Domain Name
* Arguments     : char *dns, unsigned long ip
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetGetIpFromName(char *dns, uint32_t *ip)
{
    int16_t              status = -1;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    status = sl_NetAppDnsGetHostByName((signed char *)dns, strlen(dns), ip, SL_AF_INET);

    if(status < 0)
    {
        returnStatus = WIFI_PLATFORM_STATUS_IP_ACQUIRE_FAILURE;
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetInetNtoa
* Description   : returns an IP string from IP integer
* Arguments     : char *buffer, unsigned long ip
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetInetNtoa(char *buffer, uint32_t ip)
{
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if(buffer == NULL)
    {
        returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;
    }

    sprintf(buffer, "%d.%d.%d.%d", (ip >> 24) & 0xFF, (ip >> 16) & 0xFF,
                                   (ip >> 8) & 0xFF, ip & 0xFF);

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetInetAton
* Description   : returns an IP integer from IP string (Example "192.168.1.1")
* Arguments     : char *ip, uint32_t *ipVal
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetInetAton(const char *ip, uint32_t *ipVal)
{
    /* The return value. */
    uint32_t v = 0;
    /* The count of the number of bytes processed. */
    int i;
    /* A pointer to the next digit to process. */
    const char *start;
    char c;
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;

    if((ip == NULL) || (ipVal == NULL))
    {
        returnStatus = WIFI_PLATFORM_STATUS_INVALID_PARAM;
    }

    *ipVal = 0;
    start = ip;
    for (i = 0; i < 4; i++) 
    {
        /* The digit being processed. */
   
        /* The value of this byte. */
        int n = 0;
        while (1) 
        {
            c = *start;
            start++;
            if (c >= '0' && c <= '9') 
            {
                n *= 10;
                n += c - '0';
            }
            /* We insist on stopping at "." if we are still parsing
               the first, second, or third numbers. If we have reached
               the end of the numbers, we will allow any character. */
            else if ((i < 3 && c == '.') || i == 3) 
            {
                break;
            }
            else 
            {
                return WIFI_PLATFORM_STATUS_FAILURE;
            }
        }
        if (n >= 256) 
        {
            return WIFI_PLATFORM_STATUS_FAILURE;
        }
        v *= 256;
        v += n;
    }
    *ipVal = v;

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetBuildRawRfTxParams
* Description   : returns the flags param
* Arguments     : wifiPlatformRadioChannel_t channel,
                  wifiPlatformRadioRateIndex_t rate,
                  uint32_t powerLevel,
                  uint8_t preamble,
                  int16_t *flags
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetBuildRawRfTxParams(wifiPlatformRadioChannel_t channel,
                                                       wifiPlatformRadioRateIndex_t rate,
                                                       uint32_t powerLevel,
                                                       uint8_t preamble,
                                                       int16_t *flags)
{
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;
    *flags = SL_WLAN_RAW_RF_TX_PARAMS(channel, rate, powerLevel, preamble);

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformPingAddress
* Description   : Wait for timeout or IP acquire event from wifi chip
* Arguments     :
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiPlatformStart()
* Function type : Global (Platform API)
******************************************************************************/
wifiPlatformStatus_t wifiPlatformPingAddress(uint32_t *pkts,
                                             char *destinationUrl,
                                             uint32_t interval,
                                             uint16_t size,
                                             uint16_t timeout,
                                             uint32_t attempts,
                                             uint32_t flags)
{
    SlNetAppPingCommand_t pingParams = {0};
    SlNetAppPingReport_t pingReport = {0};
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;
    uint32_t ip = 0;
    int16_t retVal = -1;

    g_pingStatus = FALSE;
    g_PingPacketsRecv = 0;

    if(destinationUrl != NULL)
    {
        retVal = sl_NetAppDnsGetHostByName((signed char *)destinationUrl,
                                strlen(destinationUrl), &ip, SL_AF_INET);
        pingParams.Ip = ip;
    }
    else
    {
        pingParams.Ip = g_GatewayIP;
        retVal = 0;
    }

    if(retVal >= 0)
    {
        pingParams.PingIntervalTime = interval;
        pingParams.PingSize = size;
        pingParams.PingRequestTimeout = timeout;
        pingParams.TotalNumberOfAttempts = attempts;
        pingParams.Flags = flags;

        retVal = sl_NetAppPing( (SlNetAppPingCommand_t*)&pingParams, SL_AF_INET,
                (SlNetAppPingReport_t*)&pingReport, wifiPingReport);

        if(retVal >= 0)
        {
            while(g_pingStatus == FALSE)
            {
#if (WIFI_USE_MULTITHREADING == 0)
                _SlNonOsHandleSpawnTask(); // @TBD Timeout check to be added
#else
                vTaskDelay(10);
#endif
            }

            if(0 == g_PingPacketsRecv)
            {
                 returnStatus =  WIFI_PLATFORM_STATUS_FAILURE;
            }
            *pkts = g_PingPacketsRecv;
        }
        else
        {
            returnStatus =  WIFI_PLATFORM_STATUS_FAILURE;
        }
    }
    else
    {
        returnStatus = WIFI_PLATFORM_STATUS_FAILURE;
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformGetNetworkTime
* Description   : get NTP time from supplied server
* Arguments     : char *ntpServer - the ntp server we're contacting
*                 uint32_t *time - the utc epoch time
* Return        : wifiPlatformStatus_t
* Pre-requiste  : wifiPlatformStart(), wifiPlatformWlanConnect()
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformGetNetworkTime(char *ntpServer,
                                                uint32_t *utcTime,
                                                uint32_t sktTimeoutSec,
                                                int16_t *driverStatus)
{
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;
    uint8_t              dataBuf[NTP_BUF_SIZE] = {0};
    int16_t              status = -1;
    int16_t              sockId = -1;
    int16_t              addrSize = 0;
    uint32_t             destinationIP = 0;
    uint32_t             elapsedSec = 0;
    SlSockAddrIn_t       localAddr;
    SlSockAddr_t         addr;

    status = sl_NetAppDnsGetHostByName((signed char *)ntpServer, strlen(ntpServer),
            &destinationIP, SL_AF_INET);
    if(status != 0)
    {
        returnStatus = WIFI_PLATFORM_STATUS_IP_ACQUIRE_FAILURE;
        goto END;
    }

    status = sl_Socket( SL_AF_INET, SL_SOCK_DGRAM, SL_IPPROTO_UDP);
    if(status < 0)
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CREATE_FAILURE;
        goto END;
    }

    sockId = status;

    struct SlTimeval_t tv;
    tv.tv_sec = sktTimeoutSec;
    tv.tv_usec = 0;

    status = sl_SetSockOpt(sockId, SL_SOL_SOCKET, SL_SO_RCVTIMEO,
            (char *)&tv, sizeof(struct SlTimeval_t));
    if(status != 0)
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CREATE_FAILURE;
        goto END;
    }

    sl_Memset(dataBuf, 0, sizeof(dataBuf));
#ifdef WIFI_PLATFORM_NTP_VERSION4
    dataBuf[0] = NTP_CLIENT_FLAGS_V4;
#else  
    dataBuf[0] = NTP_CLIENT_FLAGS_V3;
#endif
    addr.sa_family = SL_AF_INET;
    addr.sa_data[0] = 0x00;
    addr.sa_data[1] = 0x7B;    /* 123 */
    addr.sa_data[2] = (_u8)((destinationIP >> 24) & 0xFF);
    addr.sa_data[3] = (_u8)((destinationIP >> 16) & 0xFF);
    addr.sa_data[4] = (_u8)((destinationIP >> 8) & 0xFF);
    addr.sa_data[5] = (_u8) (destinationIP & 0xFF);

    status = sl_SendTo(sockId, dataBuf, sizeof(dataBuf), 0,
            &addr, sizeof(addr));
    if (status != sizeof(dataBuf))
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_WRITE_FAILURE;
        goto END;
    }

    addrSize = sizeof(SlSockAddrIn_t);
    localAddr.sin_family = SL_AF_INET;
    localAddr.sin_port = 0;
    localAddr.sin_addr.s_addr = 0;

    status = sl_Bind(sockId,(SlSockAddr_t *)&localAddr, addrSize);
    if(status != 0)
    {
        //ignoring SL_EINVAL until TI replies
        //returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CONNECT_FAILURE;
        //goto END;
    }

    status = sl_RecvFrom(sockId, dataBuf, sizeof(dataBuf), 0,
            (SlSockAddr_t *)&localAddr,  (SlSocklen_t*)&addrSize);
    if(status != sizeof(dataBuf))
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_READ_FAILURE;
        goto END;
    }

    if ((dataBuf[0] & NTP_SERVER_MODE_MASK) != NTP_SERVER_MODE_FLAG)
    {
        returnStatus = WIFI_PLATFORM_STATUS_FAILURE;
        goto END;
    }
    else
    {
        elapsedSec = dataBuf[40];
        elapsedSec <<= 8;
        elapsedSec += dataBuf[41];
        elapsedSec <<= 8;
        elapsedSec += dataBuf[42];
        elapsedSec <<= 8;
        elapsedSec += dataBuf[43];
        elapsedSec -= TIME1970;

        *utcTime = elapsedSec;
    }

    status = sl_Close(sockId);
    if(status < 0)
    {
        returnStatus = WIFI_PLATFORM_STATUS_SOCKET_CLOSE_FAILURE;
    }
    sockId = -1;
END:
    if(driverStatus)
    {
        *driverStatus = status;
    }
    if(sockId >= 0)
    {
        sl_Close(sockId);//close opened socket on error condition
    }
    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPlatformNetGetIPConfig
* Description   : returns the flags param
* Arguments     : uint32_t  ipV4
*                 uint32_t  ipV4Mask
*                 uint32_t  ipV4Gateway
*                 uint32_t  ipV4DnsServer
* Return        : wifiPlatformStatus_t
* Pre-requiste  : none
* Function type : Global /WiFi platform API
******************************************************************************/
wifiPlatformStatus_t wifiPlatformNetGetIPConfig(uint8_t *dhcpIsOn, uint32_t *ipV4,
                                               uint32_t *ipV4Mask,uint32_t *ipV4Gateway,
                                               uint32_t *ipV4DnsServer)
{
    wifiPlatformStatus_t returnStatus = WIFI_PLATFORM_STATUS_SUCCESS;
    int16_t              status = -1;
    uint16_t len = sizeof(SlNetCfgIpV4Args_t);
    SlNetCfgIpV4Args_t ipV4Args = {0};
    *dhcpIsOn = 0;

    status  = sl_NetCfgGet(SL_NETCFG_IPV4_STA_ADDR_MODE,(uint16_t *)dhcpIsOn,&len,(_u8 *)&ipV4Args);
    if(status >= 0)
    {
        *ipV4 = ipV4Args.Ip;
        *ipV4Mask = ipV4Args.IpMask;
        *ipV4Gateway = ipV4Args.IpGateway;
        *ipV4DnsServer = ipV4Args.IpDnsServer;
    }
    else
    {
        returnStatus = WIFI_PLATFORM_STATUS_FAILURE;
    }

    return returnStatus;
}

/******************************************************************************
* Function Name : wifiPingReport
* Description   : callback function for sl_NetAppPingStart
* Arguments     : SlPingReport_t *pPingReport
* Return        : none
* Pre-requiste  : sl_NetAppPingStart
* Function type : Local
******************************************************************************/
static void wifiPingReport(SlNetAppPingReport_t *pPingReport)
{
    g_pingStatus = TRUE;

    if(pPingReport == NULL)
    {
        return;
    }

    g_PingPacketsRecv = pPingReport->PacketsReceived;
}
