/**************************************************************************************************
    Filename:       tvsa.c
    Revised:        $Date: 2012-02-23 08:05:55 -0800 (Thu, 23 Feb 2012) $
    Revision:       $Revision: 29468 $

    Description:

    This file implements the Temperature/Voltage Sample Application.

    The Dongle build must define these in the IAR compiler options:
      TVSA_DONGLE
      HAL_UART
      HAL_UART_DMA_HIGH=16
      HAL_UART_DMA_RX_MAX=32
      HAL_UART_DMA_TX_MAX=254


    Copyright 2009-2011 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License"). You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product. Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include "af.h"
#include "hal_key.h"
#include "hal_lcd.h"
#include "hal_led.h"
#include "hal_uart.h"
#include "OnBoard.h"
#include "OSAL.h"
#include "tvsa.h"
#include "ZComDef.h"
#include "ZDApp.h"

#if defined ZAP_DEVICETYPE
#if !TVSA_DONGLE
#include "tvsa_zap.c"
#endif
#else
#include "mac_rx.h"
#include "mac_tx.h"
#include "mac_main.h"

#if TVSA_LCD
#include "mac_data.h"
#if defined( HAL_BOARD_F2618 ) || defined( HAL_BOARD_F5438 ) || defined( HAL_BOARD_LM3S )
#include "mac_spi.h"
#include "mac_radio_defs.h"
#endif
#endif

#if LQI_ADJUST
#include "ZMAC.h"
#endif
#endif

/* ------------------------------------------------------------------------------------------------
 *                                          Target Defines
 * ------------------------------------------------------------------------------------------------
 */

#if defined ZAP_DEVICETYPE
#define HAL_BOARD_ZAP
#elif defined( HAL_BOARD_F2618 ) || defined( HAL_BOARD_F5438 )
#define HAL_BOARD_MSP
#elif defined( HAL_BOARD_CC2430DB) || defined( HAL_BOARD_CC2430EB )
#define HAL_BOARD_2430
#elif defined( HAL_BOARD_CC2530EB_REV13 ) || defined( HAL_BOARD_CC2530EB_REV17 )
#define HAL_BOARD_2530
#elif defined HAL_BOARD_CC2538
#define HAL_BOARD_2538
#endif

#if !TVSA_DONGLE
#if defined( HAL_BOARD_2430 )
#include "tvsa_cc2430.c"
#elif defined( HAL_BOARD_2530 ) || defined HAL_BOARD_ZAP
#include "tvsa_cc2530.c"
#elif defined( HAL_BOARD_MSP )
#include "tvsa_msp430.c"
#elif defined( HAL_BOARD_LM3S )
#include "tvsa_ARM_CM3.c"
#elif defined( HAL_BOARD_2538 )
#include "tvsa_cc2538.c"
#else
#error Unexpected HAL_BOARD - need specific Temp/Volt conversion from specific A2D channel.
#endif
#endif

/* ------------------------------------------------------------------------------------------------
 *                                           Constants
 * ------------------------------------------------------------------------------------------------
 */

static const cId_t TVSA_ClusterList[TVSA_CLUSTER_CNT] =
{
  TVSA_CLUSTER_ID
};

static const SimpleDescriptionFormat_t TVSA_SimpleDesc =
{
  TVSA_ENDPOINT,
  TVSA_PROFILE_ID,
  TVSA_DEVICE_ID,
  TVSA_DEVICE_VERSION,
  TVSA_FLAGS,
  TVSA_CLUSTER_CNT,
  (cId_t *)TVSA_ClusterList,
  TVSA_CLUSTER_CNT,
  (cId_t *)TVSA_ClusterList
};

static const endPointDesc_t TVSA_epDesc=
{
  TVSA_ENDPOINT,
  &tvsaTaskId,
  (SimpleDescriptionFormat_t *)&TVSA_SimpleDesc,
  noLatencyReqs,
};

/* ------------------------------------------------------------------------------------------------
 *                                           Typedefs
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           Macros
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           Global Variables
 * ------------------------------------------------------------------------------------------------
 */

#if TVSA_DATA_CNF
uint8 tvsaCnfErrCnt;
#endif
uint8 tvsaTaskId;

/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */

// Network address of the TVSA Dongle.
static uint16 tvsaAddr;
// Report counter.
#if TVSA_DONGLE
static uint32 tvsaCnt = 0;
#else
static uint32 tvsaAttemTxCnt = 0;
static uint32 tvsaAchTxCnt = 0;
static uint16 tvsaNoRouteCnt = 0;
#if TVSA_DATA_CNF
#if PERF_TEST
static uint32 tvsaApsAckCtr = 0;
#endif // PERF_TEST
#endif // TVSA_DATA_CNF
#endif // else TVSA_DONGLE

// ZigBee-required packet transaction sequence number in calls to AF_DataRequest().
static uint8 tvsaTSN;

#if TVSA_DONGLE
static uint8 tvsaBuf[TVSA_BUF_LEN];   // to send data to UART port
#if !PERF_TEST
static uint8 tvsaCmd;
#endif  //!PERF_TEST
static uint8 tvsaState;
#else // !TVSA_DONGLE
static uint8 tvsaDat[TVSA_DAT_LEN];   // to send data OTA
#endif

#if TVSA_DONGLE
  static uint16 annceCntr = 0;

#if PERF_TEST
  uint8 testParams[TEST_MAXMSG_LEN];     // this will hold all test values received from UART
  uint8 paramCntr;
  uint16 tvsaAnnInterval;
#endif

#else // if !TVSA_DONGLE
  static uint16 dataReqCntr = 0;    // Counter to control sending of data OTA
  static uint16 dataMsgCntr = 0;    // Counter to control data calculation/collection
  static uint8 messagingStarted = FALSE;

  static zdoConcentratorInd_t conInfo;
  static uint8 conExtAddr[Z_EXTADDR_LEN];

#if PERF_TEST
  uint8 perfTstConf[ANN_MAXCFG_LEN];  // holds all configuration parameters received in Announce msg
  uint16 tvsaTxInterval;
  uint16 tvsaJitterInterval;
  uint8  tvsaPadLen;
  uint8  tvsaApsAckFlag;
  uint32 tvsaTxApsTime;     // time stamp for TX APS message
  uint32 tvsaRxAckTime;     // time stamp for RX APS ACK message
  uint32 tvsaElapsedTime;   // delta between RX and TX time stamps
  uint32 tvsaMinTime = TVSA_32BIT_MAX_VALUE;   // minimum time in ms between APS TX and APS ACK RX
  uint32 tvsaMaxTime = 0;   // maximum time in ms between APS TX and APS ACK RX
  uint32 tvsaAvrTime = 0;   // average time in ms between APS TX and APS ACK RX
  uint32 tvsaTimeSamples[TVSA_MAX_TIME_SAMPLES];
  uint8  tvsaSampleCtr = 0;   // counts time samples to calculate average latency
  uint8  tvsaSampleIndex = 0; // index for saving samples in array
  uint8 *tvsaExtendedMsg;
#endif

#endif  // TVSA_DONGLE

/* ------------------------------------------------------------------------------------------------
 *                                           Local Functions
 * ------------------------------------------------------------------------------------------------
 */

static void tvsaAfMsgRx(afIncomingMSGPacket_t *msg);
static void tvsaSysEvtMsg(void);
#if TVSA_KEY
static void tvsaKeyEvt(byte keys);
#endif
static void* tvsaConcentratorInd( void *pStr );
#if !TVSA_DONGLE
static void tvsaDataCalc(void);
static void tvsaDataReq(void);
static void tvsaZdoStateChange(void);
#if PERF_TEST
static void tvsaSaveConfig(void);
static void tvsaResetTxPacketStats(void);
static void tvsaResetTimeStats(void);
static void tvsaResetNwkAndMacStats(void);
static void tvsaCalcTimeStats(void);
#endif
#else //if TVSA_DONGLE
static void tvsaAnnce(void);
static void tvsaDataRx(afIncomingMSGPacket_t *msg);
static void tvsaUartRx(uint8 port, uint8 event);
static void tvsaZdoStateChange(void);
#endif

/**************************************************************************************************
 * @fn          tvsaInit
 *
 * @brief       This function is the application's task initialization.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void tvsaInit(uint8 id)
{
#if TVSA_DONGLE
  halUARTCfg_t uartConfig;

  uartConfig.configured           = TRUE;              // 2x30 don't care - see uart driver.
  uartConfig.baudRate             = HAL_UART_BR_115200;
  uartConfig.flowControl          = FALSE;
  uartConfig.flowControlThreshold = 16;                // 2x30 don't care - see uart driver.
  uartConfig.rx.maxBufSize        = 32;                // 2x30 don't care - see uart driver.
  uartConfig.tx.maxBufSize        = 254;               // 2x30 don't care - see uart driver.
  uartConfig.idleTimeout          = 6;                 // 2x30 don't care - see uart driver.
  uartConfig.intEnable            = TRUE;              // 2x30 don't care - see uart driver.
  uartConfig.callBackFunc         = tvsaUartRx;
  HalUARTOpen(TVSA_PORT, &uartConfig);
#else
  tvsaDat[TVSA_TYP_IDX] = (uint8)TVSA_DEVICE_ID;
#if TVSA_SRC_RTG
  tvsaDat[TVSA_OPT_IDX] = TVSA_OPT_SRC_RTG;
#endif
#if PERF_TEST
  osal_memset(&perfTstConf, 0x00, ANN_MAXCFG_LEN);
#endif
#endif

#if TVSA_DONGLE
#else
  conInfo.nwkAddr = INVALID_NODE_ADDR;
  conInfo.extAddr = (uint8 *)NULL;
  conInfo.pktCost = MAX_LINK_COST;
#endif

  /* Register with ZDO for Concentrator indication callback */
  ZDO_RegisterForZdoCB( ZDO_CONCENTRATOR_IND_CBID, &tvsaConcentratorInd );

  tvsaTaskId = id;
  tvsaAddr = INVALID_NODE_ADDR;
  (void)afRegister((endPointDesc_t *)&TVSA_epDesc);
#if TVSA_KEY
  (void)RegisterForKeys(tvsaTaskId);
#endif
#if TVSA_LED
  HalLedBlink(HAL_LED_2, 0, 1, 1000);
#endif

#if LQI_ADJUST
  ZMacLqiAdjustMode(LQI_ADJ_MODE1);
#endif

  osal_start_reload_timer( tvsaTaskId, TVSA_EVT_MSG_TIMER, TVSA_MSG_TIMER_INTERVAL );
}

/**************************************************************************************************
 * @fn          tvsaEvt
 *
 * @brief       This function is called to process the OSAL events for the task.
 *
 * input parameters
 *
 * @param       id - OSAL task Id.
 * @param       evts - OSAL events bit mask of pending events.
 *
 * output parameters
 *
 * None.
 *
 * @return      evts - OSAL events bit mask of unprocessed events.
 **************************************************************************************************
 */
uint16 tvsaEvt(uint8 id, uint16 evts)
{
  uint16 mask = 0;
  (void)id;

  if (evts & SYS_EVENT_MSG)
  {
    mask = SYS_EVENT_MSG;
    tvsaSysEvtMsg();
  }
#if TVSA_DONGLE
  else if (evts & TVSA_EVT_ANN)
  {
    mask = TVSA_EVT_ANN;
    tvsaAnnce();
  }
#else
  else if (evts & TVSA_EVT_DAT)
  {
    mask = TVSA_EVT_DAT;
    tvsaDataCalc();
  }
  else if (evts & TVSA_EVT_REQ)
  {
    mask = TVSA_EVT_REQ;
    tvsaDataReq();
  }
#endif
  else if ( evts & TVSA_EVT_MSG_TIMER )
  {
#if TVSA_DONGLE
    if ( annceCntr )
    {
      annceCntr--;
      if ( annceCntr == 0 )
      {
        osal_set_event( tvsaTaskId, TVSA_EVT_ANN );
      }
    }
#else
    if ( dataMsgCntr )
    {
      dataMsgCntr--;
      if ( dataMsgCntr == 0 )
      {
        osal_set_event( tvsaTaskId, TVSA_EVT_DAT );
      }
    }
    if ( dataReqCntr )
    {
      dataReqCntr--;
      if ( dataReqCntr == 0 )
      {
        osal_set_event( tvsaTaskId, TVSA_EVT_REQ );
      }
    }
#endif
    mask = TVSA_EVT_MSG_TIMER;
  }
  else
  {
    mask = evts;  // Discard unknown events - should never happen.
  }

  return (evts ^ mask);  // Return unprocessed events.
}

/***************************************************************************************************
 * @fn      tvsaConcentratorInd
 *
 * @brief   Handle the concentrator indication.
 *
 * @param   pStr  - pointer to the data structure for the concentrator indication
 *
 * @return  void*
 */
static void* tvsaConcentratorInd( void *pStr )
{
#if TVSA_DONGLE
#else
  zdoConcentratorInd_t *conInd = (zdoConcentratorInd_t *)pStr;

  if ( (conInfo.nwkAddr == INVALID_NODE_ADDR)
      || (conInfo.pktCost > conInd->pktCost) )
  {
    conInfo.nwkAddr = conInd->nwkAddr;
    if ( conInd->extAddr )
    {
      osal_cpyExtAddr( conExtAddr, conInd->extAddr );
      conInfo.extAddr = conExtAddr;
    }
    conInfo.pktCost = conInd->pktCost;

    tvsaAddr = conInfo.nwkAddr;
  }

  if ( messagingStarted == FALSE )
  {
#if PERF_TEST
    // set the Tx Data message counter to its maximum value
    dataMsgCntr = tvsaTxInterval + JITTER_OF(tvsaJitterInterval);
#else
    dataMsgCntr = TVSA_DLY_MSG_CNTR + JITTER_60SEC;
#endif
    NLME_PermitJoiningRequest( 0xFF );
    messagingStarted = TRUE;
  }
#endif
  return NULL;
}

/**************************************************************************************************
 * @fn          tvsaAfMsgRx
 *
 * @brief       This function is called by tvsaSysEvtMsg() to process an incoming AF message.
 *
 * input parameters
 *
 * @param       msg - A pointer to the afIncomingMSGPacket_t packet.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaAfMsgRx(afIncomingMSGPacket_t *msg)
{
  uint8 *buf = msg->cmd.Data;

#if TVSA_DONGLE
  if (buf[TVSA_CMD_IDX] == TVSA_CMD_DAT)
  {
    tvsaDataRx(msg);
  }
#else

  if (buf[ANN_CMD] & (TVSA_CMD_BEG | TVSA_CMD_TST_PAR | TVSA_CMD_TST_OPT | TVSA_CMD_RES_STATS))
  {
#if PERF_TEST
    // do this only if something has changed in the test configuration
    if (!osal_memcmp(perfTstConf, buf, ANN_MAXCFG_LEN))
    {
      osal_memcpy(perfTstConf, buf, ANN_MAXCFG_LEN);

      tvsaSaveConfig();

#if TVSA_LCD
      HalLcdWriteString( "Rcv Config", HAL_LCD_LINE_3 );
#endif

      // to make sure next if statement will be TRUE and will post an event
      if ( messagingStarted == TRUE )
      {
        tvsaAddr = INVALID_NODE_ADDR;
      }
    }
#endif // PERF_TEST

    if (INVALID_NODE_ADDR == tvsaAddr)
    {
      NLME_SetPollRate(0);
      (void)osal_set_event(tvsaTaskId, TVSA_EVT_DAT);
    }
    tvsaAddr = BUILD_UINT16(buf[TVSA_ADR_LSB], buf[TVSA_ADR_MSB]);
  }
  else if (buf[ANN_CMD] & TVSA_CMD_END)
  {
    NLME_SetPollRate(POLL_RATE);
    tvsaAddr = INVALID_NODE_ADDR;
  }

#endif

}

/**************************************************************************************************
 * @fn          tvsaSysEvtMsg
 *
 * @brief       This function is called by tvsaAppEvt() to process all of the pending OSAL messages.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaSysEvtMsg(void)
{
  uint8 *msg;

  while ((msg = osal_msg_receive(tvsaTaskId)))
  {
    switch (*msg)
    {
    case AF_INCOMING_MSG_CMD:
      tvsaAfMsgRx((afIncomingMSGPacket_t *)msg);
      break;

#if TVSA_KEY
    case KEY_CHANGE:
      tvsaKeyEvt(((keyChange_t *)msg)->keys);
      break;
#endif

    case ZDO_STATE_CHANGE:
      tvsaZdoStateChange();
      break;

#if TVSA_DATA_CNF
    case AF_DATA_CONFIRM_CMD:
      if (ZSuccess != ((afDataConfirm_t *)msg)->hdr.status)
      {
        if (0 == ++tvsaCnfErrCnt)
        {
          tvsaCnfErrCnt = 255;
        }
      }
#if !TVSA_DONGLE
#if PERF_TEST
      else
      {
        // calculate latency independently of APS ACK on
        tvsaRxAckTime = osal_GetSystemClock();

        tvsaCalcTimeStats();

        if (tvsaApsAckFlag == AF_ACK_REQUEST)
        {
          // we are only sending 3 bytes OTA, reset counter when MAX VALUE has been reached
          if (tvsaApsAckCtr >= TVSA_24BIT_MAX_VALUE)
          {
            tvsaApsAckCtr = 0;
          }
          tvsaApsAckCtr++;
        }
      }
#endif // PERF_TEST
#endif  // !TVSA_DONGLE

#if TVSA_LED
      HalLedSet(HAL_LED_2, HAL_LED_MODE_OFF);
      HalLedBlink(HAL_LED_2, 0, 1, 1000);
#endif
      break;
#endif

    default:
      break;
    }

    (void)osal_msg_deallocate(msg);  // Receiving task is responsible for releasing the memory.
  }
}

#if TVSA_KEY
/**************************************************************************************************
 * @fn          tvsaKeyEvt
 *
 * @brief       This function is called by tvsaSysEvtMsg() to process an incoming AF message.
 *
 * input parameters
 *
 * @param       keys - Bit mask corresponding to key(s) pressed.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaKeyEvt(byte keys)
{
  if (keys & HAL_KEY_SW_1)
  {
    HalLedSet(HAL_LED_1, HAL_LED_MODE_TOGGLE);
  }
  if (keys & HAL_KEY_SW_2)
  {
    HalLedSet(HAL_LED_2, HAL_LED_MODE_TOGGLE);
  }
}
#endif

#if !TVSA_DONGLE
/**************************************************************************************************
 * @fn          tvsaDataCalc
 *
 * @brief       This function is called by tvsaAppEvt() to calculate the data for a TVSA report.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaDataCalc(void)
{
#if PERF_TEST
  uint8 i;
#endif

  if (INVALID_NODE_ADDR == tvsaAddr)
  {
    return;
  }

#if defined ZAP_DEVICETYPE
  tvsaZapMacDataDump();  // Populate the local, cache variables with the MAC debug data.
#endif

#if PERF_TEST
  // set the Tx Data message counter to its maximum value
  dataMsgCntr = tvsaTxInterval + JITTER_OF(tvsaJitterInterval);
#else
  dataMsgCntr = TVSA_DLY_MSG_CNTR + JITTER_60SEC;
#endif

#if PERF_TEST
  if (perfTstConf[ANN_OPT] & TVSA_TEST_OPT_LNSBLD_MASK)
  {
    // LNS Build Code
    tvsaDat[TVSA_LBC_IDX] = LO_UINT16(LNS_BUILD_ID);
    tvsaDat[TVSA_LBC_IDX+1] = HI_UINT16(LNS_BUILD_ID);
  }
  else
#endif
  {
    // Update temperature & Voltage info
    HalCalcTV(tvsaDat);
  }

#if TVSA_DATA_CNF
  tvsaDat[TVSA_RTG_IDX] = tvsaCnfErrCnt;
#else
  tvsaDat[TVSA_RTG_IDX] = 0;
#endif

#if PERF_TEST
  tvsaDat[TVSA_ATTEM_TX_IDX]   = BREAK_UINT32(tvsaAttemTxCnt, 2);
  tvsaDat[TVSA_ATTEM_TX_IDX+1] = BREAK_UINT32(tvsaAttemTxCnt, 1);
  tvsaDat[TVSA_ATTEM_TX_IDX+2] = BREAK_UINT32(tvsaAttemTxCnt, 0);

  tvsaDat[TVSA_ACHVD_TX_IDX]   = BREAK_UINT32(tvsaAchTxCnt, 2);
  tvsaDat[TVSA_ACHVD_TX_IDX+1] = BREAK_UINT32(tvsaAchTxCnt, 1);
  tvsaDat[TVSA_ACHVD_TX_IDX+2] = BREAK_UINT32(tvsaAchTxCnt, 0);

  tvsaDat[TVSA_APS_ACK_IDX]   = BREAK_UINT32(tvsaApsAckCtr, 2);
  tvsaDat[TVSA_APS_ACK_IDX+1] = BREAK_UINT32(tvsaApsAckCtr, 1);
  tvsaDat[TVSA_APS_ACK_IDX+2] = BREAK_UINT32(tvsaApsAckCtr, 0);

  if (tvsaMinTime == TVSA_32BIT_MAX_VALUE)
  {
    // this is the first message
    tvsaDat[TVSA_MIN_TX_IDX]   = 0;
    tvsaDat[TVSA_MIN_TX_IDX+1] = 0;
    tvsaDat[TVSA_MIN_TX_IDX+2] = 0;
  }
  else
  {
    tvsaDat[TVSA_MIN_TX_IDX]   = BREAK_UINT32(tvsaMinTime, 2);
    tvsaDat[TVSA_MIN_TX_IDX+1] = BREAK_UINT32(tvsaMinTime, 1);
    tvsaDat[TVSA_MIN_TX_IDX+2] = BREAK_UINT32(tvsaMinTime, 0);
  }

  tvsaDat[TVSA_MAX_TX_IDX]   = BREAK_UINT32(tvsaMaxTime, 2);
  tvsaDat[TVSA_MAX_TX_IDX+1] = BREAK_UINT32(tvsaMaxTime, 1);
  tvsaDat[TVSA_MAX_TX_IDX+2] = BREAK_UINT32(tvsaMaxTime, 0);

  // Calculate average of the time samples
  tvsaAvrTime = 0;
  for (i = 0; i < tvsaSampleCtr; i++)
  {
    tvsaAvrTime += tvsaTimeSamples[i];
  }

  if (tvsaSampleCtr > 0)
  {
    tvsaAvrTime /= tvsaSampleCtr;
  }

  tvsaDat[TVSA_AVR_TX_IDX]   = BREAK_UINT32(tvsaAvrTime, 2);
  tvsaDat[TVSA_AVR_TX_IDX+1] = BREAK_UINT32(tvsaAvrTime, 1);
  tvsaDat[TVSA_AVR_TX_IDX+2] = BREAK_UINT32(tvsaAvrTime, 0);

#if PACKET_FILTER_STATS
  tvsaDat[TVSA_INV_PKT_IDX]   = BREAK_UINT32(nwkInvalidPackets, 3);
  tvsaDat[TVSA_INV_PKT_IDX+1] = BREAK_UINT32(nwkInvalidPackets, 2);
  tvsaDat[TVSA_INV_PKT_IDX+2] = BREAK_UINT32(nwkInvalidPackets, 1);
  tvsaDat[TVSA_INV_PKT_IDX+3] = BREAK_UINT32(nwkInvalidPackets, 0);

  tvsaDat[TVSA_MAC_PKT_RX_IDX]   = BREAK_UINT32(rxCrcSuccess, 3);
  tvsaDat[TVSA_MAC_PKT_RX_IDX+1] = BREAK_UINT32(rxCrcSuccess, 2);
  tvsaDat[TVSA_MAC_PKT_RX_IDX+2] = BREAK_UINT32(rxCrcSuccess, 1);
  tvsaDat[TVSA_MAC_PKT_RX_IDX+3] = BREAK_UINT32(rxCrcSuccess, 0);

  tvsaDat[TVSA_MAC_PKT_CRC_IDX]   = BREAK_UINT32(rxCrcFailure, 3);
  tvsaDat[TVSA_MAC_PKT_CRC_IDX+1] = BREAK_UINT32(rxCrcFailure, 2);
  tvsaDat[TVSA_MAC_PKT_CRC_IDX+2] = BREAK_UINT32(rxCrcFailure, 1);
  tvsaDat[TVSA_MAC_PKT_CRC_IDX+3] = BREAK_UINT32(rxCrcFailure, 0);
#endif

#endif  // PERF_TEST

  osal_set_event(tvsaTaskId, TVSA_EVT_REQ);
}

/**************************************************************************************************
 * @fn          tvsaDataReq
 *
 * @brief       This function is called by tvsaAppEvt() to send a TVSA data report.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaDataReq(void)
{
  afAddrType_t addr;
  uint8 *sendData;
  uint16 sendDataLen;
  afStatus_t rtrn;
#if PERF_TEST
  uint16 resendInterval;
#endif
#if TVSA_LCD
  #define LCD_W  16
  uint8 lcd_buf[LCD_W+1];
  uint8 numBuf[5];
  uint8 tmp;
#endif

#if TVSA_LCD
  osal_memset( lcd_buf, ' ', LCD_W );
  lcd_buf[LCD_W] = '\0';

#if ZSTACK_END_DEVICE_BUILD
  lcd_buf[0] = 'E';
#else
  lcd_buf[0] = 'R';
#endif
  lcd_buf[1] = ':';

  lcd_buf[3] = '0';
  lcd_buf[4] = '0';
  lcd_buf[5] = '0';
  lcd_buf[6] = '0';
#if !defined HAL_BOARD_F5438
  lcd_buf[7] = ',';
#endif

  lcd_buf[9] = 'P';
  lcd_buf[10] = ':';

  lcd_buf[12] = '0';
  lcd_buf[13] = '0';
  lcd_buf[14] = '0';
  lcd_buf[15] = '0';

  osal_memset( numBuf, 0, 5 );
  _ltoa((unsigned long)NLME_GetShortAddr(), numBuf, 16 );
  tmp = osal_strlen( (char *)numBuf );
  osal_memcpy(&lcd_buf[3+(4-tmp)], numBuf,  tmp );

  osal_memset( numBuf, 0, 5 );
  _ltoa((unsigned long)NLME_GetCoordShortAddr(), numBuf, 16 );
  tmp = osal_strlen( (char *)numBuf );
  osal_memcpy(&lcd_buf[12+(4-tmp)], numBuf,  tmp );

  HalLcdWriteString( (char*)lcd_buf, HAL_LCD_LINE_1 );
#endif

  addr.addr.shortAddr = tvsaAddr;
  addr.addrMode = afAddr16Bit;
  addr.endPoint = TVSA_ENDPOINT;

#if PERF_TEST
  uint8 i;

  // add extra data to message if requested by test
  if (tvsaPadLen > 0)
  {
    sendDataLen = TVSA_DAT_LEN + tvsaPadLen;

    // create new buffer with new length
    if ((tvsaExtendedMsg = osal_mem_alloc(sendDataLen)) != NULL)
    {
      osal_memcpy(tvsaExtendedMsg, tvsaDat, TVSA_DAT_LEN);

      for (i = TVSA_DAT_LEN; i < sendDataLen; i++)
      {
        tvsaExtendedMsg[i] = 0xEE;    // pattern for padded data
      }

      sendData = tvsaExtendedMsg;
    }
  }
  else
#endif // this handles both PERF_DATA without padded bytes and regular TVSA tests
  {
    sendData = tvsaDat;
    sendDataLen = TVSA_DAT_LEN;
  }

#if PERF_TEST
  tvsaTxApsTime = osal_GetSystemClock();
#endif

  rtrn = AF_DataRequest(&addr, (endPointDesc_t *)&TVSA_epDesc, TVSA_CLUSTER_ID,
                                          sendDataLen, sendData, &tvsaTSN,
                                          AF_DISCV_ROUTE
                                          | AF_LIMIT_CONCENTRATOR
#if TVSA_DATA_CNF
#if PERF_TEST
                                          | tvsaApsAckFlag
#else
                                          | AF_ACK_REQUEST
#endif // PERF_TEST
#endif // TVSA_DATA_CNF

                                         ,AF_DEFAULT_RADIUS);

  if(afStatus_SUCCESS == rtrn)
  {
#if TVSA_LED && TVSA_DATA_CNF
    HalLedSet(HAL_LED_2, HAL_LED_MODE_ON);
#endif

    // total number of attempted APS tx messages, reset counter at 3 bytes MAX
    // VALUE, that's what is sent OTA
    if (tvsaAchTxCnt >= TVSA_24BIT_MAX_VALUE)
    {
      tvsaAchTxCnt = 0;
    }
    tvsaAchTxCnt++;

#if TVSA_LCD
    HalLcdWriteStringValue( "Tx succ ", tvsaNoRouteCnt, 16, HAL_LCD_LINE_3 );
#endif
  }
#if PERF_TEST
  else if(afStatus_NO_ROUTE == rtrn)
  {
    // maximum value in uint16
    resendInterval = TVSA_TEST_MAX_RESEND_INTRVL_MS;

    // use half the regular Tx Interval in ms only if it fits in uint16
    if ((tvsaTxInterval/TVSA_TEST_INTERVALS_PER_SECOND) < TVSA_TEST_MAX_RESEND_INTRVL_MS)
    {
      resendInterval = (tvsaTxInterval/TVSA_TEST_INTERVALS_PER_SECOND) * TVSA_MS_IN_SEC;
    }

    if (tvsaNoRouteCnt >= TVSA_16BIT_MAX_VALUE)
    {
      tvsaNoRouteCnt = 0;
    }
    tvsaNoRouteCnt++;

#if TVSA_LCD
//    HalLcdWriteStringValue( "NoRoute ", tvsaNoRouteCnt, 16, HAL_LCD_LINE_3 );
    {
      uint8 regTemp = 0;

      lcd_buf[LCD_W] = '\0';

      lcd_buf[0]  = '0'; // R
      lcd_buf[1]  = '0';
      lcd_buf[2]  = '0'; // T
      lcd_buf[3]  = '0';
      lcd_buf[4]  = '0'; // M
      lcd_buf[5]  = '0';
      lcd_buf[6]  = '*';
      lcd_buf[7]  = '0'; // C
      lcd_buf[8]  = '0';
      lcd_buf[9]  = '0'; // D
      lcd_buf[10] = '0';
      lcd_buf[11] = '*';
      lcd_buf[12] = '0'; // Stat0
      lcd_buf[13] = '0';
      lcd_buf[14] = '0'; // Stat1
      lcd_buf[15] = '0';

      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)macRxActive, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[0+(2-tmp)], numBuf,  tmp );

      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)macTxActive, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[2+(2-tmp)], numBuf,  tmp );

      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)macMain.state, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[4+(2-tmp)], numBuf,  tmp );

      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)macData.rxCount, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[7+(2-tmp)], numBuf,  tmp );

      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)macData.directCount, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[9+(2-tmp)], numBuf,  tmp );

#if defined HAL_BOARD_ZAP
      regTemp = fsmstat0;
#elif defined( HAL_BOARD_MSP ) || defined( HAL_BOARD_LM3S )
      regTemp = macSpiReadReg(FSMSTAT0);
#elif defined( HAL_BOARD_2530 )
      regTemp = FSMSTAT0;
#endif
      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)regTemp, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[12+(2-tmp)], numBuf,  tmp );

#if defined HAL_BOARD_ZAP
      regTemp = fsmstat1;
#elif defined( HAL_BOARD_MSP ) || defined( HAL_BOARD_LM3S )
      regTemp = macSpiReadReg(FSMSTAT1);
#elif defined( HAL_BOARD_2530 )
      regTemp = FSMSTAT1;
#endif
      osal_memset( numBuf, 0, 5 );
      _ltoa((unsigned long)regTemp, numBuf, 16 );
      tmp = osal_strlen( (char *)numBuf );
      osal_memcpy(&lcd_buf[14+(2-tmp)], numBuf,  tmp );

      HalLcdWriteString( (char*)lcd_buf, HAL_LCD_LINE_3 );
    }
#endif

    // set a timer to resend the data
    osal_start_timerEx(tvsaTaskId, TVSA_EVT_REQ, resendInterval);
  }
#endif // PERF_TEST
  else
  {
#if !PERF_TEST
    // retry to send data only in regular TVSA aplication and not for PER test
    osal_set_event(tvsaTaskId, TVSA_EVT_REQ);
#endif
  }

  // total number of attempted APS tx messages, reset counter at 3 bytes MAX
  // VALUE, that's what is sent OTA
  if (tvsaAttemTxCnt >= TVSA_24BIT_MAX_VALUE)
  {
    tvsaAttemTxCnt = 0;
  }
  tvsaAttemTxCnt++;

#if TVSA_LCD
  HalLcdWriteStringValue( "TxCnt:", tvsaAttemTxCnt, 10, HAL_LCD_LINE_2 );
#endif

#if PERF_TEST
  if (tvsaExtendedMsg != NULL)
  {
    osal_mem_free(tvsaExtendedMsg);

    // make sure the pointer and length are set to initial values
    tvsaExtendedMsg = NULL;
    sendDataLen = 0;
  }
#endif
}

/**************************************************************************************************
 * @fn          tvsaZdoStateChange
 *
 * @brief       This function is called by tvsaSysEvtMsg() for a ZDO_STATE_CHANGE message.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaZdoStateChange(void)
{
  dataMsgCntr = 0;

  if ((DEV_ZB_COORD == devState) || (DEV_ROUTER == devState) || (DEV_END_DEVICE == devState))
  {
    uint16 tmp = NLME_GetCoordShortAddr();

    tvsaDat[TVSA_PAR_IDX] = LO_UINT16(tmp);
    tvsaDat[TVSA_PAR_IDX+1] = HI_UINT16(tmp);
    if ((DEV_ROUTER == devState) || (DEV_ZB_COORD == devState))
    {
      tvsaDat[TVSA_TYP_IDX] |= 0x80;
    }
    else
    {
      tvsaDat[TVSA_TYP_IDX] &= (0xFF ^ 0x80);
    }

#if TVSA_DONGLE_IS_ZC
    if (INVALID_NODE_ADDR == tvsaAddr)
    {
      // Assume ZC is the TVSA Dongle until a TVSA_CMD_BEG gives a different address.
      tvsaAddr = NWK_PAN_COORD_ADDR;
    }
#endif

    if ( (DEV_END_DEVICE == devState) && (INVALID_NODE_ADDR != tvsaAddr) )
    {
#if PERF_TEST
      // set the Tx Data message counter to its maximum value
      dataMsgCntr = tvsaTxInterval + JITTER_OF(tvsaJitterInterval);
#else
      dataMsgCntr = TVSA_DLY_MSG_CNTR + JITTER_60SEC;
#endif

      messagingStarted = TRUE;
    }
    else
    {
      NLME_PermitJoiningRequest( 0 );
      messagingStarted = FALSE;
    }

#if !TVSA_DONGLE
#if !defined ( HAL_BOARD_2530 ) && !defined HAL_BOARD_ZAP
    HalInitTV();
#endif
    (void)osal_cpyExtAddr(tvsaDat+TVSA_IEE_IDX, &aExtendedAddress);
#endif
  }
#if TVSA_LCD
  HalLcdWriteValue(devState, 10, HAL_LCD_LINE_3);
#endif
}

#if PERF_TEST
/**************************************************************************************************
 * @fn          tvsaSaveConfig
 *
 * @brief       This function is called by tvsaAfMsgRx() to save received test configuration.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaSaveConfig(void)
{
  if (perfTstConf[ANN_CMD] & TVSA_CMD_TST_PAR)
  {
    // get test parameter values
    tvsaTxInterval = TVSA_TEST_SEC_TO_TIMER_PERIOD(
                                    BUILD_UINT16(perfTstConf[ANN_TXINTRVL],
                                                 perfTstConf[ANN_TXINTRVL+1]));

    tvsaJitterInterval = TVSA_TEST_SEC_TO_TIMER_PERIOD(
                                    BUILD_UINT16(perfTstConf[ANN_TXJITTER],
                                                 perfTstConf[ANN_TXJITTER+1]));
    tvsaPadLen = perfTstConf[ANN_MSGPAD];
  }

  if (perfTstConf[ANN_CMD] & TVSA_CMD_TST_OPT)
  {
    // get test options values
    if (perfTstConf[ANN_OPT] & TVSA_TEST_OPT_JITT_MASK)
    {
      tvsaJitterInterval = TVSA_TEST_SEC_TO_TIMER_PERIOD(
                                    BUILD_UINT16(perfTstConf[ANN_TXJITTER],
                                                 perfTstConf[ANN_TXJITTER+1]));
    }
    else
    {
      tvsaJitterInterval = 0;
    }

    // enable/disable APS ACK
    if (perfTstConf[ANN_OPT] & TVSA_TEST_OPT_APSACK_MASK)
    {
      tvsaApsAckFlag = AF_ACK_REQUEST;
    }
    else
    {
      // disable APS ACK and clear time stats and APS ACK counter
      tvsaApsAckFlag = 0x00;
      tvsaResetTimeStats();
      tvsaApsAckCtr = 0;
    }
  }

  if (perfTstConf[ANN_CMD] & TVSA_CMD_RES_STATS)
  {
    // reset all statistics
    tvsaResetTxPacketStats();
    tvsaResetTimeStats();
#ifdef PACKET_FILTER_STATS
    tvsaResetNwkAndMacStats();
#endif
  }
}

/**************************************************************************************************
 * @fn          tvsaResetTxPacketStats
 *
 * @brief       This function resets Tx counters variables to initial values.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaResetTxPacketStats(void)
{
  tvsaAttemTxCnt = 0;
  tvsaAchTxCnt = 0;

#if TVSA_DATA_CNF
  tvsaApsAckCtr = 0;
#endif
}

/**************************************************************************************************
 * @fn          tvsaResetTimeStats
 *
 * @brief       This function resets time stats variables to initial values.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaResetTimeStats(void)
{
  tvsaMinTime = TVSA_32BIT_MAX_VALUE;
  tvsaMaxTime = 0;
  tvsaAvrTime = 0;
  tvsaSampleCtr = 0;
  tvsaSampleIndex = 0;
}

#ifdef PACKET_FILTER_STATS
/**************************************************************************************************
 * @fn          tvsaResetNwkAndMacStats
 *
 * @brief       This function resets Invalid network packets and MAC stats variables to initial values.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaResetNwkAndMacStats(void)
{
  nwkInvalidPackets = 0;

  HAL_DISABLE_INTERRUPTS();
  rxCrcSuccess = 0;
  HAL_ENABLE_INTERRUPTS();

  HAL_DISABLE_INTERRUPTS();
  rxCrcFailure = 0;
  HAL_ENABLE_INTERRUPTS();
}
#endif // PACKET_FILTER_STATS

/**************************************************************************************************
 * @fn          tvsaCalcTimeStats
 *
 * @brief       This function calculates all statistic related to TX packet latency.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaCalcTimeStats(void)
{
  if (tvsaRxAckTime <= tvsaTxApsTime)
  {
    tvsaElapsedTime = (tvsaTxApsTime ^ TVSA_32BIT_MAX_VALUE) + tvsaRxAckTime;
  }
  else
  {
    tvsaElapsedTime = tvsaRxAckTime - tvsaTxApsTime;
  }

  if (tvsaElapsedTime < tvsaMinTime)
  {
    // save new minimum time
    tvsaMinTime = tvsaElapsedTime;
  }

  if (tvsaElapsedTime > tvsaMaxTime)
  {
    // save new maximum time
    tvsaMaxTime = tvsaElapsedTime;
  }

  if (tvsaSampleIndex >= TVSA_MAX_TIME_SAMPLES)
  {
    // start putting data at the beginning of the queue
    tvsaSampleIndex = 0;
  }

  if (tvsaSampleCtr < TVSA_MAX_TIME_SAMPLES)
  {
    // once it reaches the maximum number that means buffer is full
    // and it will be using TVSA_MAX_TIME_SAMPLES to calculate average
    tvsaSampleCtr++;
  }

  // store samples to calculate data later
  tvsaTimeSamples[tvsaSampleIndex++] = tvsaElapsedTime;
}
#endif // PERF_TEST

#else // if TVSA_DONGLE
/**************************************************************************************************
 * @fn          tvsaAnnce
 *
 * @brief       This function is called by tvsaAppEvt() to send a TVSA announce to start or stop.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaAnnce(void)
{
  uint8 msgAnnounce[ANN_MAXCFG_LEN];

  afAddrType_t addr;
  afStatus_t stat;

  addr.addr.shortAddr = NWK_BROADCAST_SHORTADDR_DEVALL;
  addr.addrMode = afAddrBroadcast;
  addr.endPoint = TVSA_ENDPOINT;

  if (INVALID_NODE_ADDR != tvsaAddr)
  {
#if PERF_TEST
    msgAnnounce[ANN_CMD] = testParams[TEST_CMD];

    annceCntr = tvsaAnnInterval;
#else
    msgAnnounce[ANN_CMD] = TVSA_CMD_BEG;

    annceCntr = TVSA_DLY_ANN_CNTR;
#endif
  }
  else
  {
    msgAnnounce[ANN_CMD] = TVSA_CMD_END;
  }

  msgAnnounce[ANN_DONGLE_ADDR_LSB] = LO_UINT16(tvsaAddr);
  msgAnnounce[ANN_DONGLE_ADDR_MSB] = HI_UINT16(tvsaAddr);

#if PERF_TEST
  // copy the rest of the values received from UART and put them in the msgAnnounce
  osal_memcpy(&msgAnnounce[ANN_OPT], &testParams[TEST_OPT], ANN_MAXCFG_LEN - ANN_OPT);
#endif

  stat = AF_DataRequest(&addr, (endPointDesc_t *)&TVSA_epDesc, TVSA_CLUSTER_ID,
                               ANN_MAXCFG_LEN, msgAnnounce, &tvsaTSN,
                               AF_TX_OPTIONS_NONE, AF_DEFAULT_RADIUS);
  if ( afStatus_SUCCESS != stat )
  {
    // retry to send data
    osal_set_event(tvsaTaskId, TVSA_EVT_REQ);
#if TVSA_LCD
    HalLcdWriteStringValue( "Annce Fail: ", (uint16)stat, 16, HAL_LCD_LINE_3 );
#endif
  }
  else
  {
    tvsaCnt++;
  }
}

/**************************************************************************************************
 * @fn          tvsaDataRx
 *
 * @brief       This function is called by tvsaAfMsgRx() to process incoming TVSA data.
 *
 * input parameters
 *
 * @param       msg - A pointer to the afIncomingMSGPacket_t packet.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaDataRx(afIncomingMSGPacket_t *msg)
{
  uint8 fcs = 0, idx;

  // Last announce broadcast to stop must have expired before a parent could forward to a ZED.
  if (INVALID_NODE_ADDR == tvsaAddr)
  {
    (void)osal_set_event(tvsaTaskId, TVSA_EVT_ANN);
  }

  tvsaBuf[TVSA_SOP_IDX] = TVSA_SOP_VAL;
  tvsaBuf[TVSA_ADR_LSB] = LO_UINT16(msg->srcAddr.addr.shortAddr);
  tvsaBuf[TVSA_ADR_MSB] = HI_UINT16(msg->srcAddr.addr.shortAddr);

  // 1st byte of message is skipped - CMD is always 0 for data.
  (void)osal_memcpy(tvsaBuf+TVSA_DAT_OFF, msg->cmd.Data+1, TVSA_DAT_LEN-1);

  for (idx = TVSA_ADR_LSB; idx < TVSA_FCS_IDX; idx++)
  {
    fcs ^= tvsaBuf[idx];
  }
  tvsaBuf[idx] = fcs;

  HalUARTWrite(TVSA_PORT, tvsaBuf, TVSA_BUF_LEN);
}

/**************************************************************************************************
 * @fn          tvsaUartRx
 *
 * @brief       This function is the Uart callback for Rx data.
 *
 * input parameters
 *
 * @param       port - Don't care.
 * @param       event - Don't care.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaUartRx(uint8 port, uint8 event)
{
  uint8 ch;

#if PERF_TEST
  paramCntr = 0;
#endif

  while (HalUARTRead(TVSA_PORT, &ch, 1))
  {
    switch (tvsaState)
    {
    case SOP_STATE:
      if (TVSA_SOP_VAL == ch)
      {
        tvsaState = CMD_STATE;
      }
      break;

    case CMD_STATE:
#if PERF_TEST
      // Identify what commands are set

      testParams[TEST_CMD] = ch;

      paramCntr++;

      tvsaState = GET_CFG_STATE;
#else
      tvsaCmd = ch;
      tvsaState = FCS_STATE;
#endif  // PERF_TEST

      break;

#if PERF_TEST
    case GET_CFG_STATE:
      // save test configuration parameters one at the time
      testParams[paramCntr] = ch;
      paramCntr++;
      // the maximum number of parameters has been reached
      if (paramCntr >= ANN_MAXCFG_LEN)
      {
        if (testParams[TEST_CMD] & (TVSA_CMD_BEG | TVSA_CMD_TST_PAR | TVSA_CMD_TST_OPT | TVSA_CMD_RES_STATS))
        {
          // first configure DONGLE if Announce msg should be sent and the period
          if (testParams[TEST_OPT] & TVSA_TEST_OPT_ANN_MASK)
          {
            tvsaAnnInterval = TVSA_TEST_SEC_TO_TIMER_PERIOD(
                                                            BUILD_UINT16(testParams[TEST_ANNINTRVL],
                                                                         testParams[TEST_ANNINTRVL+1]));

            // clear announce bit before sending commands to devices, this parameter is only for DONGLE
            testParams[TEST_OPT] &= ~TVSA_TEST_OPT_ANN_MASK;

            tvsaAddr = NLME_GetShortAddr();

            osal_set_event(tvsaTaskId, TVSA_EVT_ANN);
          }
          else
          {
            // do not send Announce message to the network
            tvsaAnnInterval = 0;
          }
        }
        else if (testParams[TEST_CMD] & TVSA_CMD_END)
        {
          // first configure Announce interval
          if (testParams[TEST_OPT] & TVSA_TEST_OPT_ANN_MASK)
          {
            tvsaAnnInterval = TVSA_TEST_SEC_TO_TIMER_PERIOD(
                                                            BUILD_UINT16(testParams[TEST_ANNINTRVL],
                                                                         testParams[TEST_ANNINTRVL+1]));
          }
          else
          {
            // set default, to make sure the Announce msg is sent to the NWK
            annceCntr = TVSA_DLY_ANN_CNTR;
          }

          // Send message to all devices to terminate the test
          tvsaAddr = INVALID_NODE_ADDR;

          osal_set_event(tvsaTaskId, TVSA_EVT_ANN);

        }
        tvsaState = SOP_STATE;
      }

      break;

#endif  // PERF_TEST

    case FCS_STATE:
#if !PERF_TEST
      if (tvsaCmd == ch)
      {
        if (tvsaCmd == TVSA_CMD_BEG)
        {
          tvsaAddr = NLME_GetShortAddr();
        }
        else if (tvsaCmd == TVSA_CMD_END)
        {
          tvsaAddr = INVALID_NODE_ADDR;
        }
        (void)osal_set_event(tvsaTaskId, TVSA_EVT_ANN);

      }
#endif

      tvsaState = SOP_STATE;
      break;

    default:
     break;
    }
  }
}

/**************************************************************************************************
 * @fn          tvsaZdoStateChange
 *
 * @brief       This function is called by tvsaSysEvtMsg() for a ZDO_STATE_CHANGE message.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void tvsaZdoStateChange(void)
{
  annceCntr = 0; // Stop timer

  if ((DEV_ZB_COORD == devState) || (DEV_ROUTER == devState) || (DEV_END_DEVICE == devState))
  {
#if TVSA_DONGLE_IS_ZC
    if (INVALID_NODE_ADDR == tvsaAddr)
    {
      // Assume ZC is the TVSA Dongle until a TVSA_CMD_BEG gives a different address.
      tvsaAddr = NWK_PAN_COORD_ADDR;
    }
#endif

    if ( INVALID_NODE_ADDR != tvsaAddr )
    {
      annceCntr = TVSA_DLY_ANN_CNTR;
    }
  }
}
#endif // if TVSA_DONGLE

/**************************************************************************************************
 */
