/**************************************************************************************************
    Filename:       tvsa.h
    Revised:        $Date: 2012-02-23 08:05:55 -0800 (Thu, 23 Feb 2012) $
    Revision:       $Revision: 29468 $

    Description:

    This file implements the Temperature/Voltage Sample Application.


    Copyright 2009-2011 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License"). You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product. Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/
#ifndef TVSA_H
#define TVSA_H

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include "hal_board_cfg.h"
#include "ZComDef.h"

/* ------------------------------------------------------------------------------------------------
 *                                          Constants
 * ------------------------------------------------------------------------------------------------
 */

// Expanding on a TI proprietary Profile Id used for Z-Accel / ZASA / CC2480.
#define TVSA_PROFILE_ID            0x0F10
#define TVSA_CLUSTER_ID            0x0002
#define TVSA_CLUSTER_CNT           1
#define TVSA_DEVICE_VERSION        0
#define TVSA_FLAGS                 0

#define TVSA_ENDPOINT              3

#if !defined TVSA_DEVICE_ID
// Master for CC2480 ZASA reports sets to zero.
#if defined HAL_BOARD_CC2430BB
#define TVSA_DEVICE_ID             0x0001
#elif defined HAL_BOARD_CC2430DB
#define TVSA_DEVICE_ID             0x0002
#elif defined HAL_BOARD_CC2430EB
#define TVSA_DEVICE_ID             0x0003
#elif defined HAL_BOARD_CC2530EB_REV13
#define TVSA_DEVICE_ID             0x0004
#elif defined HAL_BOARD_CC2530EB_REV17
#if defined HAL_BOARD_CC2530USB
#define TVSA_DEVICE_ID             0x0008
#else
#define TVSA_DEVICE_ID             0x0005
#endif
#elif defined HAL_BOARD_F5438
#define TVSA_DEVICE_ID             0x0006
#elif defined HAL_BOARD_F2618
#define TVSA_DEVICE_ID             0x0007
#elif defined HAL_BOARD_LM3S8962
#define TVSA_DEVICE_ID             0x0009
#elif defined HAL_BOARD_LM3S9B96
#define TVSA_DEVICE_ID             0x0009
#elif defined HAL_BOARD_CC2538
#define TVSA_DEVICE_ID             0x000A
#else

//efine TVSA_DEVICE_ID             0x0016  // ZAP on a MSP5438-to-CC2530EM-ZNP as ZED.
//efine TVSA_DEVICE_ID             0x0096  // ZAP on a MSP5438-to-CC2530EM-ZNP as ZR.
//efine TVSA_DEVICE_ID             0x0017  // ZAP on a MSP2618-to-CC2530EM-ZNP as ZED.
//efine TVSA_DEVICE_ID             0x0097  // ZAP on a MSP2618-to-CC2530EM-ZNP as ZR.
//efine TVSA_DEVICE_ID             0x0018  // ZAP by Z-Tool script to CC2530EM as ZED.
//efine TVSA_DEVICE_ID             0x0098  // ZAP by Z-Tool script to CC2530EM as ZR.
//efine TVSA_DEVICE_ID             0x0019  // ZAP by Z-Tool script to CC2531EM as ZED.
//efine TVSA_DEVICE_ID             0x0099  // ZAP by Z-Tool script to CC2531EM as ZR.

#error Unexpected HAL_BOARD - need specific Temp/Volt conversion from specific A2D channel.
#endif
#endif

#if !defined TVSA_DONGLE
#define TVSA_DONGLE                FALSE
#endif

#if !defined TVSA_DONGLE_IS_ZC
#define TVSA_DONGLE_IS_ZC          TRUE
#endif

// Bit values for the TVSA_OPT_IDX.
#define TVSA_OPT_SRC_RTG           0x01

#if PERF_TEST
#define TVSA_DAT_LEN               46   // length of calculated data sent OTA
#else
#define TVSA_DAT_LEN               16   // length of calculated data sent OTA
#endif

// Indices of message received OTA
#define TVSA_CMD_IDX               0    // Command
#define TVSA_IEE_IDX               1    // IEEE address
#define TVSA_PAR_IDX               9    // Parent Short Address
#define TVSA_LBC_IDX               11   // LNS Build Code (replaces temp+volt)
#define TVSA_TEM_IDX               11   // Temperature
#define TVSA_BUS_IDX               12   // Voltage
#define TVSA_TYP_IDX               13   // Node Type
#define TVSA_OPT_IDX               14
#define TVSA_RTG_IDX               15
#define TVSA_ATTEM_TX_IDX          16   // Attempted TX APS messages counter
#define TVSA_ACHVD_TX_IDX          19   // Achieved TX APS messages counter
#define TVSA_APS_ACK_IDX           22   // Received APS ACK messages counter
#define TVSA_MIN_TX_IDX            25   // Minimum time between TX messages
#define TVSA_MAX_TX_IDX            28   // Maximum time between TX messages
#define TVSA_AVR_TX_IDX            31   // Average time between TX messages
#define TVSA_INV_PKT_IDX           34   // Invalid Network packets counter
#define TVSA_MAC_PKT_RX_IDX        38   // MAC packets RX counter
#define TVSA_MAC_PKT_CRC_IDX       42   // MAC packets RX with bad CRC counter


#define TVSA_ADR_LSB               1
#define TVSA_ADR_MSB               2

#if TVSA_DONGLE
#define TVSA_SOP_VAL               0xFE
#define TVSA_PORT                  0
#define SOP_STATE                  0
#define CMD_STATE                  1
#define GET_CFG_STATE              2
#define FCS_STATE                  3

#define TVSA_SOP_IDX               0
#define TVSA_DAT_OFF               TVSA_ADR_MSB+1
// Not adding +1 for FCS because 1st byte of message is skipped - CMD is always 0 for data.
#define TVSA_BUF_LEN               TVSA_DAT_LEN+TVSA_DAT_OFF
#define TVSA_FCS_IDX               TVSA_BUF_LEN-1
#endif

// TVSA Command set. This is a Bitmap for the commands of the test
#define TVSA_CMD_DAT               0x00  // TVSA data message.
#define TVSA_CMD_BEG               0x01  // Start reporting TVSA data.
#define TVSA_CMD_END               0x02  // Stop reporting TVSA data.
#define TVSA_CMD_TST_PAR           0x04  // Performance test parameters included.
#define TVSA_CMD_TST_OPT           0x08  // Performance test options included.
#define TVSA_CMD_RES_STATS         0x10  // Reset Statistics.
#define TVSA_CMD_RESERVED1         0x20  // Reserved command for future use.
#define TVSA_CMD_RESERVED2         0x40  // Reserved command for future use.
#define TVSA_CMD_RESERVED3         0x80  // Reserved command for future use.

#define TVSA_EVT_ANN               0x4000
#define TVSA_EVT_DAT               TVSA_EVT_ANN  /* Mutually exclusive use by TVSA_DONGLE. */
#define TVSA_EVT_REQ               0x2000

#define JITTER_15SEC               (osal_rand() & 0x001F)
#define JITTER_30SEC               (osal_rand() & 0x003F)
#define JITTER_60SEC               (osal_rand() & 0x007F)
// if passed interval for jitter is 0 the do not apply jitter
#define JITTER_OF(x)               (((x) > 0)?(osal_rand() % (x)):0)

#define TVSA_DLY_ANN_CNTR          240  // interval - 500 ms, so 240 is 2 minutes
#define TVSA_DLY_MSG_CNTR          240  // interval - 500 ms, so 240 is 2 minutes
#define TVSA_MSG_TIMER_INTERVAL    500
#define TVSA_EVT_MSG_TIMER         0x0010

#define TVSA_DLY_ANN               60000
#define TVSA_DLY_DAT               TVSA_DLY_ANN

#if !defined TVSA_SRC_RTG
#define TVSA_SRC_RTG               FALSE
#endif

#if TVSA_SRC_RTG
#if !defined ZIGBEE_SOURCE_ROUTING
  #error ZIGBEE_SOURCE_ROUTING must be globally defined to TRUE.
#endif
#if !defined ZIGBEE_MANY_TO_ONE
  #error ZIGBEE_MANY_TO_ONE must be globally defined to TRUE.
#endif
#if TVSA_DONGLE
#if !defined CONCENTRATOR_ENABLE || !CONCENTRATOR_ENABLE
  #error CONCENTRATOR_ENABLE must be globally defined to TRUE.
#endif
#if !defined CONCENTRATOR_DISCOVERY_TIME || ((CONCENTRATOR_DISCOVERY_TIME * 1000) > TVSA_DLY_DAT)
  #error CONCENTRATOR_DISCOVERY_TIME must be globally defined to be less than or equal to TVSA_DLY_DAT
#endif
#endif
#endif

#if !defined TVSA_DATA_CNF
#define TVSA_DATA_CNF              TRUE
#endif

#if !defined TVSA_KEY
#define TVSA_KEY                   FALSE
#endif

#if !defined TVSA_LCD
#if defined LCD_SUPPORTED
#define TVSA_LCD                   TRUE
#else
#define TVSA_LCD                   FALSE
#endif
#endif

#if !defined TVSA_LED
#define TVSA_LED                   FALSE
#endif

#define TVSA_16BIT_MAX_VALUE    0x0000FFFF
#define TVSA_24BIT_MAX_VALUE    0x00FFFFFF
#define TVSA_32BIT_MAX_VALUE    0xFFFFFFFF

#if PERF_TEST
// values used for Performance Test
#define TVSA_MS_IN_SEC              1000
#define TVSA_TEST_TICK_INTERVAL     500     // ms
#define TVSA_TEST_INTERVALS_PER_SECOND      (TVSA_MS_IN_SEC / TVSA_TEST_TICK_INTERVAL)
#define TVSA_TEST_SEC_TO_TIMER_PERIOD(a)    ((a) * TVSA_TEST_INTERVALS_PER_SECOND) // this is because the interval in 500 ms
#define TVSA_TEST_TIMER_PERIOD_TO_SEC(a)    ((a) / TVSA_TEST_INTERVALS_PER_SECOND)
#define TVSA_TEST_MAX_RESEND_INTRVL_MS 60000   // maximum value for uint16 in ms
#define TVSA_MAX_TIME_SAMPLES       10      // number of samples to calculate Average

#define TVSA_TEST_OPT_ANN_MASK      0x0001  // Announcement on/off
#define TVSA_TEST_OPT_JITT_MASK     0x0002  // Jitter on/off
#define TVSA_TEST_OPT_APSACK_MASK   0x0004  // Aps ACK on/off
#define TVSA_TEST_OPT_LNSBLD_MASK   0x0008  // Return LNS Build ID instead of Temp/Voltage.
// reserved for future use 0x0010 to 0x8000

// Definition of Test Parameters message received from UART PORT
typedef enum
{
  TEST_CMD       = 0,  // 1 byte long, Bitmap for command
  TEST_OPT       = 1,  // 2 bytes long, Bitmap for test options
  TEST_TXINTRVL  = 3,  // 2 bytes long, Tx Interval in seconds
  TEST_TXJITTER  = 5,  // 2 bytes long, Jitter in ms to apply for Tx
  TEST_MSGPAD    = 7,  // 1 byte long, Number of bytes to add to each Tx message
  TEST_ANNINTRVL = 8,  // 2 bytes long, Interval to broadcast announcement

  TEST_MAXMSG_LEN = 10  // Do not add fields after this value, this field value is
                       // the maximum number of bytes in the message recieved from
                       // UART. Adjust properly if fields are added/removed.
} tvsaPerfTstCfg_t;

#endif // PERF_TEST

// Definition of Configuration parameters sent to Devices in Announcement message OTA
typedef enum
{
  ANN_CMD               = 0,  // 1 byte long, Bitmap for command
  ANN_DONGLE_ADDR_LSB   = 1,  // 1 byte long, short address LSB of dongle
  ANN_DONGLE_ADDR_MSB   = 2,  // 1 byte long, short address MSB of dongle
#if PERF_TEST
  ANN_OPT               = 3,  // 2 bytes long, Bitmap for test options
  ANN_TXINTRVL          = 5,  // 2 bytes long, Interval in secods to Tx values
  ANN_TXJITTER          = 7,  // 2 bytes long, Jitter to apply to Tx values
  ANN_MSGPAD            = 9,  // 1 byte long, Number of bytes to add to each Tx message

  ANN_MAXCFG_LEN        = 10  // Do not add fields after this value, this field value is
                              // the maximum number of bytes in the message recieved
                              // send/received OTA from/to dongle/devices.
                              // Adjust properly if fields are added/removed.
#else
  ANN_MAXCFG_LEN        =  3   // for backwards compatibility with regular TVSA test
#endif // PERF_TEST
} tvsaPerfTstAnnMsg_t;

/* ------------------------------------------------------------------------------------------------
 *                                          LNS Build ID
 * ------------------------------------------------------------------------------------------------
 */

// Build ID code: used for 'over-the-air' identification of a device's TVSA image,
// This ID should be incremented when a new build is used on nodes for the LNS,
// You can over-ride this ID by using LNS_BUILD_ID=0xNNNN in the IDE options
#if !defined( LNS_BUILD_ID )
  #define LNS_BUILD_ID 0x0018   // 04/15/2011  SVN: 25722
#endif

/* ------------------------------------------------------------------------------------------------
 *                                          Macros
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           Global Variables
 * ------------------------------------------------------------------------------------------------
 */

#if TVSA_DATA_CNF
extern uint8 tvsaCnfErrCnt;
#endif
extern uint8 tvsaTaskId;

/* ------------------------------------------------------------------------------------------------
 *                                          Functions
 * ------------------------------------------------------------------------------------------------
 */

/**************************************************************************************************
 * @fn          tvsaInit
 *
 * @brief       This function is the application's task initialization.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void tvsaInit(uint8 id);

/**************************************************************************************************
 * @fn          tvsaEvt
 *
 * @brief       This function is called to process the OSAL events for the task.
 *
 * input parameters
 *
 * @param       id - OSAL task Id.
 * @param       evts - OSAL events bit mask of pending events.
 *
 * output parameters
 *
 * None.
 *
 * @return      evts - OSAL events bit mask of unprocessed events.
 **************************************************************************************************
 */
uint16 tvsaEvt(uint8 id, uint16 evts);

#endif
/**************************************************************************************************
*/
