/**************************************************************************************************
    Filename:       tvsa_cc2530.c
    Revised:        $Date: 2011-05-20 18:49:45 -0700 (Fri, 20 May 2011) $
    Revision:       $Revision: 26058 $

    Description:

    This file implements the CC253x board-specific Temperature/Voltage calculations.

    Copyright 2009-2011 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License").  You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product.  Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include "comdef.h"
#include "tvsa.h"

/* ------------------------------------------------------------------------------------------------
 *                                           Constants
 * ------------------------------------------------------------------------------------------------
 */

//#define VOLTAGE_AT_TEMP_25        1480  // 14-bit ADC = 1480 at 25C.
#define TEMP_COEFFICIENT            4     // 4 ADC counts per degC.
// Adjust this according to room temperature where devices are powered up.
#define TEMP_AT_CALBR               22    // degC.

#define HAL_ADC_REF_125V    0x00    /* Internal 1.25V Reference */
#define HAL_ADC_DEC_512     0x30    /* Decimate by 512 : 14-bit resolution */
#define HAL_ADC_CHN_TEMP    0x0e    /* Temperature sensor */

/* ------------------------------------------------------------------------------------------------
 *                                           Macros
 * ------------------------------------------------------------------------------------------------
 */

#define degC_0       (voltageAtCalbr - (TEMP_AT_CALBR * TEMP_COEFFICIENT))
#define degC_100     (voltageAtCalbr + ((100 - TEMP_AT_CALBR) * TEMP_COEFFICIENT))

/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */

static uint16 voltageAtCalbr;

/* ------------------------------------------------------------------------------------------------
 *                                           Local Functions
 * ------------------------------------------------------------------------------------------------
 */

static void HalCalcTV(uint8 *tvDat);

/**************************************************************************************************
 * @fn          HalCalcTV
 *
 * @brief       This function is called by tvsaDataCalc() to calculate the board-specific data
 *              for a TVSA report.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void HalCalcTV(uint8 *tvDat)
{
  uint16 tmp;

  // Make the SOC temperature sensor measurement.
#if defined ZAP_DEVICETYPE
  uint8 args[2] = { HAL_ADC_CHANNEL_TEMP, HAL_ADC_RESOLUTION_14 };
  zapSysReq(MT_SYS_ADC_READ, (uint8 *)&tmp, args);
#else
  ADCCON3 = 0x3E;
  while (!(ADCCON1 & 0x80));
  // Use all 14 MSB of adcValue
  tmp = ADCL >> 2;
  tmp |= ADCH << 6;
#endif

  if (0 == voltageAtCalbr)
  {
    voltageAtCalbr = tmp;
  }

  if (tmp < degC_0)
  {
    tmp = degC_0;
  }
  else if (tmp > degC_100)
  {
    tmp = degC_100;
  }
  tvDat[TVSA_TEM_IDX] = (uint8)((tmp - degC_0 + TEMP_COEFFICIENT-1) / TEMP_COEFFICIENT);

  // Measure CC2530 SoC "Vdd/3" with minimum decimate by 64 for 7 ENOB using internal reference.
#if defined ZAP_DEVICETYPE
  args[0] = HAL_ADC_CHANNEL_VDD;
  zapSysReq(MT_SYS_ADC_READ, (uint8 *)&tmp, args);
#else
  ADCCON3 = 0x0F;
  while (!(ADCCON1 & 0x80));
  tmp = ADCH;
#endif

  // Vdd = ADC X 3 X Internal Reference / ENOB
  // But 3.45 X 20 = 69 and "Vdd X 10" gives 10ths of a V, so use ADC x 69 / 254
  tmp = (tmp * 69) / 254;
  tvDat[TVSA_BUS_IDX] = (uint8)tmp;
}

/**************************************************************************************************
*/
