/******************************************************************************
 *
 *          MSP430 core firmware debugger/emulator
 *
******************************************************************************
**
** MODULE NAME:     emulator.c
** MODULE DATE:     10-12-2009
** AUTHOR:          Jef van de Molengraft
**                  jef@jvdm.nl
**
** MODULE FUNCTION:
**
** I had some time to kill, so I wrote this little MSP430 emulator just for
** fun. It's up to the point where it can do basic emulation of firmware. And
** break the code at a given breakpoint. The idea is to implement a debugger
** on wireless sensor nodes in cobinaion with a wireless bootloader to enable
** wireless debugging.
**
** This emulator enables debugging when the JTAG is not accessable or if the
** JTAG fuse bit is blown.
**
** Limitations:
**
** MSP430X core is not supported, but who knows...if I have some more time to
** kill or if I would get a pile of money to do so :)
**
** Because the emulator is causing a lot of overhead on the MSP430, it can
** become practically impossible to debug realtime applications.
**
** Once the interupt vectors are redirected the program can't run in normal mode
** anymore when interrupts are enabled.
**
**
** Please send me an email if you find this code useful.
**
** Cheers,
** Jef
*/
#include "msp430.h"
#include "emulator.h"
#define true  1
#define false 0

// MSP430 double operand instruction bit fields
#define OPCODE      0xF000
#define S_REG       0x0F00
#define AD          0x0080
#define WB          0x0040
#define AS          0x0030
#define D_REG       0x000F

// MSP430 single operand instructions
#define CALL        0x1280
#define RETI        0x1300

// MSP430 double operand instructions
#define JNZ         0x2000
#define JZ          0x2400
#define JNC         0x2800
#define JC          0x2C00
#define JN          0x3000
#define JGE         0x3400
#define JL          0x3800
#define JMP         0x3C00

#define BIS_SR      0xD032                                                      // BIS #src, SR

// MSP430 emulated instructions
#define RET         0x4130                                                      // MOV @SP+,PC
#define BR          0x4000                                                      // MOV dst, PC (See below)
#define NOP         0x4303                                                      // MOV #0, R3

// Instruction to set stack pointer
#define SET_SP      0x4031                                                      // MOV @PC+, SP

// Branch addressing modes (Branch is a MSP430 emulated instruction)
#define BR_DIRECT   0x030                                                       // MOV @PC+,PC
#define BR_XPC      0x010                                                       // MOV X(PC),PC
#define BR_X0       0x210                                                       // MOV X(0),PC
#define BR_Rn       0x000                                                       // MOV Rn,PC
#define BR_atRn     0x020                                                       // MOV @Rn,PC
#define BR_atRnP    0x030                                                       // MOV @Rn+,PC
#define BR_XRn      0x010                                                       // MOV X(Rn),PC

#define SEGMENT0_     0xFE00
#define RESETVECTOR_  0xFFFE
#define RESETVECTOR   *(unsigned int *)RESETVECTOR_
#define SEGMENT0      *(unsigned int *)SEGMENT0_

#pragma data_alignment=2
__no_init regs reg;                                                             // Emulated CPU registers

// These variables need to be stored in ram because the working registers are lost after a ram execute or if they have to be referred to in inline-assembly
#pragma data_alignment=2
__no_init unsigned int ram_instruction[5];
__no_init unsigned int stackpointer;
__no_init unsigned int stackcontent;
__no_init unsigned int *address;
__no_init unsigned int *oldaddress;
__no_init unsigned int inst_size;
__no_init unsigned int *gbreakpoint;
__no_init unsigned int temp;

__no_init istate_t _state;

#pragma location=0xFEE0
__no_init unsigned int redirect_vectors[16*8];                                  // Catch interrupt table

//******************************************************************************
// Dummy function. (Just to get the right branch address after compilation)
void branchback()
{
  asm("br #instruction_end");
}

//******************************************************************************
// Dummy function.
// These instructions are used to redirect the interrupt vectors
void catch_interrupt_instructions()
{
  asm("mov &address, &oldaddress");                                             // Store address
  asm("mov #0x1234, &address");                                                 // #0x1234 is replaced by the actual interrupt handler address
  asm("br #emulator_start");                                                    // Branch to emulator
}

//******************************************************************************
// The interrupt vectors and reset vector are located in the FLASH memory.
// To redirect the vectors they are temporary saved.
// Then the FLASH segment is erased and the reset vector is restored.
// New interrupt vectors are generated and point to catch_interrupt_instructions.
// Finaly the catch_interrupt_instructions are generated and stored in FLASH.
void redirect_interrupt_vectors()
{
  unsigned int interrupt_vectors_temp[16];
  unsigned int temp = 0;

  // Copy the current interrupt vectors to a temporary storage
  for(unsigned int *vector=(unsigned int *)0xFFE0; vector>=(unsigned int *)0xFFE0; vector++)
  {
    if(*vector >= SEGMENT0_ && *vector !=0xFFFF)                                // Check if interrupts vectors are already redirected
    {
      temp=0;
      break;                                                                    // Interrupts already have been redirected,  break
    }
    else
    {
      interrupt_vectors_temp[temp++] = *vector;
    }
  }

  if(temp)
  {
    FCTL2 = FWKEY | FSSEL0 | 15;                                                // MCLK/16 (15+1) for Flash Timing Generator (must bein the range from ~ 257 kHz to ~ 476 kHz) (so MCLK should be within ~ 4.1MHz to ~ 7.6MHz)

    FCTL3 = FWKEY;                                                              // Clear Lock bit
    FCTL1 = FWKEY | ERASE;
    SEGMENT0 = 0;                                                               // Dummy write to erase Flash segment
    FCTL1 = FWKEY | WRT;                                                        // Set WRT bit for write operation.

    RESETVECTOR = interrupt_vectors_temp[15];                                   // Restore reset vector

    temp = 0;
    // Copy vectors to the catch interrupt instructions table
    for(unsigned int *vector = (unsigned int *)0xFFE0; vector < (unsigned int *)0xFFFE; vector++)
    {
       *(unsigned int *)vector = (unsigned int)&redirect_vectors[temp];
       temp += 8;
    }

    // Copy catch interrupt instructions to the table
    temp = 0;
    for(unsigned int vector=0; vector<15; vector++)
    {
      redirect_vectors[temp]   = *( unsigned int *)(&catch_interrupt_instructions);
      redirect_vectors[temp+1] = *((unsigned int *)(&catch_interrupt_instructions)+1);
      redirect_vectors[temp+2] = *((unsigned int *)(&catch_interrupt_instructions)+2);
      redirect_vectors[temp+3] = *((unsigned int *)(&catch_interrupt_instructions)+3);
      redirect_vectors[temp+4] = interrupt_vectors_temp[vector];
      redirect_vectors[temp+5] = *((unsigned int *)(&catch_interrupt_instructions)+5);
      redirect_vectors[temp+6] = *((unsigned int *)(&catch_interrupt_instructions)+6);
      redirect_vectors[temp+7] = *((unsigned int *)(&catch_interrupt_instructions)+7);
      temp += 8;
    }
    FCTL1 = FWKEY;                                                              // Clear WRT bit for write operation.
    FCTL3 = FWKEY | LOCK;                                                       // set Lock bit
  }
}

//******************************************************************************
regs *emulator(void (*start_address)(), unsigned int *break_point)
{
  unsigned int  instruction;
  unsigned char do_jump;

  __disable_interrupt();

  asm("mov SP, &stackpointer");                                                 // Save stack pointer
  asm("mov 0(SP), &stackcontent");

  redirect_interrupt_vectors();

  // Initialize emulated registers
  asm("mov SP, &reg+2");
  asm("mov SR, &reg+4");
  asm("mov #0, &reg+6");                                                        // R3 = constant generator register
  asm("mov R4, &reg+8");
  asm("mov R5, &reg+10");
  asm("mov R6, &reg+12");
  asm("mov R7, &reg+14");
  asm("mov R8, &reg+16");
  asm("mov R9, &reg+18");
  asm("mov R10, &reg+20");
  asm("mov R11, &reg+22");
  asm("mov R12, &reg+24");
  asm("mov R13, &reg+26");
  asm("mov R14, &reg+28");
  asm("mov R15, &reg+30");

  address     = (unsigned int *)start_address;                                  // Set address to start address
  gbreakpoint = break_point;

  while(1)
  {
    asm("emulator_start:");

    if(reg.sr.gie)                                                              // If the application has interrupt enabled
    {
      __enable_interrupt();
    }

    //**********************************************************
    // Break point
    //**********************************************************

    if(gbreakpoint == address)                                                  // Check if breakpoint is reached
    {
      __no_operation();                                                         // Set a breakpoint here to stop executing when gbreakpoint is reached
      break;
    }

    //**********************************************************
    // Catch all PC manipulation instructions
    //**********************************************************

    // Catch jump instructions
    do_jump     = false;
    instruction = (*address & 0xFC00);                                          // Mask for Jump instructions

    switch(instruction)
    {
      case JNZ:                                                                 // JNE Jump if non-zero / Jump if not equal
        if(reg.sr.z)
        {
          address++;
          asm("br #emulator_start");                                            // Hello C teacher, See... I'm not using goto's :)
        }
        do_jump = true;
        break;

      case JZ:                                                                  // JEQ  Jump if zero / Jump if equal
        if(!(reg.sr.z))
        {
          address++;
          asm("br #emulator_start");
        } do_jump = true;
        break;

      case JNC:                                                                 // Jump carry unset
        if(reg.sr.c)
        {
          address++;
          asm("br #emulator_start");
        } do_jump = true;
        break;

      case JC:                                                                  // Jump carry set
        if(!(reg.sr.c))
        {
          address++;
          asm("br #emulator_start");
        }
        do_jump = true;
        break;

      case JN:                                                                  // Jump negative
        if(!(reg.sr.n))
        {
          address++;
          asm("br #emulator_start");
        }
        do_jump = true;
        break;

      case JGE:                                                                 // Jump if greater or equal
        if((reg.sr.v) ^ (reg.sr.n))
        {
          address++;
          asm("br #emulator_start");
          break;
        }
        do_jump = true;
        break;

      case JL:                                                                  // Jump if less
        if(!((reg.sr.v) ^ (reg.sr.n)))
        {
          address++;
          asm("br #emulator_start");
          break;
        }
        do_jump = true;
        break;

      case JMP:
        do_jump = true;
    }
    if(do_jump)                                                                 // Jump uses a 10 bit offset address, jump range is -512 to + 512 words
    {
      if((*address & 0x0200))                                                   // Check sign of offset address (+ or -)
      {
        address = (unsigned int *)(address - (0x200 - (*address & 0x01FF))) + 1;
      }
      else
      {
        address = (unsigned int *)(address + (*address & 0x01FF)) + 1;
      }
      asm("br #emulator_start");
    }

    if(*address == SET_SP)                                                      // If setting the stack pointer, don't execute the instruction
    {
      address += 2;
      asm("br #emulator_start");
    }

    // Catch call, ret, reti and branch instructions
    instruction = (*address & 0xFF80);                                          // Mask for Single operand instruction

    __disable_interrupt();                                                      // Disable interrupts

    if(instruction == CALL)                                                     // Call function
    {
      address++;                                                                // Set address to point to the call address
      asm("dec.w SP");                                                          // Add entry on the stack pointer
      asm("mov &address, 0(SP)");                                               // Put call address on the stack
      address = (unsigned int *)*address;                                       // Set address to the call address
      asm("br #emulator_start");
    }

    if(instruction == RETI)                                                     // Return from interrupt
    {
      asm("mov &oldaddress, &address");                                         // Restore the address from before the interrupt
      asm("mov #emulator_start, 2(SP)");                                        // After interrupt return to the emulator
      asm("reti");                                                              // Return from interrupt
    }

    instruction = *address;                                                     // No mask for MSP430 emulated instructions
    if(instruction == RET)                                                      // Return from function
    {
      asm("mov SP, &temp");
      if(temp == stackpointer) break;                                           // If this is the final return then break the emulator
      asm("mov 0(SP), &address");                                               // Get address from the stack
      asm("add  #2, SP");                                                       // Remove entry from the stack
      address++;                                                                // Set address to next instruction
      asm("br #emulator_start");
    }

    if((instruction & 0xF0CF) == BR)                                            // Brach to address (mov dst, PC)
    {
      unsigned int branchmode;
      unsigned int branchregmode;
      unsigned int branchreg;

      address++;                                                                // Set address to point to the branch address or index
      branchmode = instruction & 0x0FF0;                                        // Branch mode mask

      // Branch direct, inderect, absolute
      switch(branchmode)
      {
        case BR_DIRECT:                                                         // Direct branch (br #address)
          address = (unsigned int *)*address;                                   // Set address to the branch address
          asm("br #emulator_start");

        case BR_XPC:                                                            // Branch to the address contained in address (br address)
          address = address + (((unsigned int)*address) >>1);
          address = (unsigned int *)*address;                                   // Set address to the branch address
          asm("br #emulator_start");

        case BR_X0:                                                             // Branch to the address contained in absolute addres (br &address)
          address = (unsigned int *)*(unsigned int *)*address;                  // Set address to the branch address
          asm("br #emulator_start");
      }

      branchregmode = instruction & 0x00FF;                                     // Branch register mode mask
      branchreg     = ((instruction & 0x0F00) >> 8);                            // Get the register number from the instruction

      // Branch register mode
      switch(branchregmode)
      {
        case BR_Rn:                                                             // Branch to the address contained in Rn (br Rn)
          address = (unsigned int *)*((unsigned int *)(&reg) + branchreg);
          asm("br #emulator_start");

        case BR_atRn:                                                           // Branch to the address contained in the word pointed to by Rn (br @Rn)
          address = (unsigned int *)*((unsigned int *)(&reg) + branchreg);
          address = (unsigned int *)*address;                                   // Set address to the branch address
          asm("br #emulator_start");

        case BR_atRnP:                                                          // Branch to the address contained in the word pointed to by Rn and increment pointer in Rn afterwards. (br @Rn+)
          address = (unsigned int *)*((unsigned int *)(&reg) + branchreg);
          address = (unsigned int *)*address;                                   // Set address to the branch address
          *((unsigned int *)(&reg) + branchreg) += 2;                           // Increment pointer in Rn
          asm("br #emulator_start");

        case BR_XRn:                                                            // Branch to the address contained in the address pointed to by Rn + X (br X(Rn))
          temp = *((unsigned int *)(&reg) + branchreg);                         // Get Rn register value
          temp = temp + *address;                                               // Add X
          address = (unsigned int *)*(unsigned int *)temp;
          asm("br #emulator_start");
      }
    }

    //**********************************************************
    // Catch bis #src, SR to handle low power modes correctly
    //**********************************************************

    if(*address  == BIS_SR)                                                     // bis #src, SR
    {
      address++;                                                                // Set address to bis source bitfield
      temp = *(unsigned int *)address;                                          // Temporary save the bitfield
      address++;                                                                // Set address to next instruction
      asm("mov &reg+4, SR");                                                    // Restore SR (gie)
      asm("bis &temp, SR");                                                     // Bit set SR (And if CPU_OFF/OSC_OFF/SCG0/SCG1 is set goto low power mode)
      asm("br #emulator_start");
    }

    //**********************************************************
    // Execute the instruction (if it doesn't manipulate the PC)
    //**********************************************************

    if(reg.sr.gie)                                                              // If application has interrupts enabled
    {
      __enable_interrupt();
    }

    inst_size = 1;                                                              // The instruction size is 1 word

    // Get size of instruction (1, 2 or 3 words)
    if((instruction & OPCODE) >= 0x2000)                                        // If double operand  instruction
    {
      // If source addressing mode is immediate
      if((instruction & AS) == 0x0030 &&                                        // AS = Indirect auto increment / Immediate mode
         (instruction & S_REG)  == 0)                                           // S_REG = PC (source address is the next word)
      {
        inst_size++;                                                            // Instruction has a source address operand
      }
      // If source addressing mode is absolute, indexed mode or symbolic
      else if((instruction & AS)     == 0x0010 &&                               // AS = Indexed / Symbolic / Absolute mode
             ((instruction & S_REG)  >= 0x0400 ||                               // S_REG = R4 to R15 (indexed mode)
              (instruction & S_REG)  == 0x0200 ||                               // S_REG = R2 (absolute mode)
              (instruction & S_REG)  == 0x0000))                                // S_REG = PC (symbolic mode)
      {
        inst_size++;                                                            // Instruction has a source address or offset operand
      }
    }
    // If destination mode is not register
    if((instruction & AD) == 0x0080)                                            // If AD mode != register mode
    {
      inst_size++;                                                              // Instruction has a destination address operand
    }

    // Put the instruction into RAM
    ram_instruction[0] = address[0];
    ram_instruction[1] = address[1];
    ram_instruction[2] = address[2];

    // Add branch instruction after the instruction in RAM
    ram_instruction[inst_size]   = *(unsigned int *)&branchback;
    ram_instruction[inst_size+1] = *((unsigned int *)(&branchback)+1);

    __disable_interrupt();                                                      // Disable interrupts

    asm("mov #reg+2, R15");
    asm("mov @R15+, SP");                                                       // Restore the stack pointer
    asm("mov @R15+, SR");                                                       // Restore the status register
//    asm("add #2, R15"); // Causes an unwanted changes of the SR
    asm("mov @R15+, R4"); // Do this instead to increase R15
    asm("mov @R15+, R4");                                                       // Restore registers R4 to R15
    asm("mov @R15+, R5");
    asm("mov @R15+, R6");
    asm("mov @R15+, R7");
    asm("mov @R15+, R8");
    asm("mov @R15+, R9");
    asm("mov @R15+, R10");
    asm("mov @R15+, R11");
    asm("mov @R15+, R12");
    asm("mov @R15+, R13");
    asm("mov @R15+, R14");
    asm("mov @R15,  R15");

    asm("br #ram_instruction");                                                 // Branch to ram function with inserted instruction
    asm("instruction_end:");                                                    // Put return label here

    asm("mov SP, &reg+2");                                                      // Save the stack pointer
    asm("mov SR, &reg+4");                                                      // Save the status register
    asm("mov R4, &reg+8");                                                      // Save registers R4 to R15
    asm("mov R5, &reg+10");
    asm("mov R6, &reg+12");
    asm("mov R7, &reg+14");
    asm("mov R8, &reg+16");
    asm("mov R9, &reg+18");
    asm("mov R10, &reg+20");
    asm("mov R11, &reg+22");
    asm("mov R12, &reg+24");
    asm("mov R13, &reg+26");
    asm("mov R14, &reg+28");
    asm("mov R15, &reg+30");

    address += inst_size;                                                       // Increase address to next instruction
  }

  __disable_interrupt();                                                        // Disable interrupts
  reg.PC = (unsigned int)address;                                               // Save the program counter

  asm("mov &stackpointer, SP");                                                 // Restore stack pointer
  asm("mov &stackcontent, 0(SP)");
  return &reg;                                                                  // Return pointer to emulated CPU registers
}

