#include "msp430.h"
#include "emulator.h"

// Function prototypes
void breakpoint();
void test_call_ret_break();
void test_interrupt();
void test_addr_modes();
void test_branch();
void test_jump();
void test_stackpointer();
void flashingled();

__no_init unsigned char testnumber;
__no_init regs *emul_cpu_regs;                                                  // Emulated CPU registers

//******************************************************************************
// main is run in normal mode. Use 'step over' with the IAR emulator to step
// over the functions in main and watch the variables 'testnumber' and
// emul_cpu_regs.
void main()
{
  WDTCTL  = WDTPW + WDTHOLD;                                                    // Stop watchdog timer
  BCSCTL1 = RSEL2 | RSEL1 | RSEL0;                                              // Set DCO to about 5MHz (at room temperature)
  DCOCTL  = DCO2  | DCO1  | DCO0;                                               // Set DCO to about 5MHz (at room temperature)

  // Do all tests in normal mode
  testnumber = 1;
  test_call_ret_break();
  // testnumber =  5

  if(*(unsigned int *)0xffe0 < (unsigned int)0xfe00 ||
     *(unsigned int *)0xffe0 == 0xFFFF)                                         // Don't do this test if interrupts are redirected (after a reset)
  {
    test_interrupt();
    // testnumber = 0 (changed inside the interrupt routine)
  }

  test_addr_modes();
  test_branch();
  test_jump();
  //test_stackpointer(); // (Skip the stack pointer modification test in normal mode)


  // Now do all tests using the MSP430 emulator
  testnumber = 1;

  // Take a look at memory address 0xFFE0. The interrupt vector table is original

  emul_cpu_regs = emulator(test_call_ret_break, (unsigned int *)breakpoint);    // Breakpoint test. (unsigned int *)breakpoint = address where to break the emulator

  // testnumber = 1;  unchanged because of the breakpoint
  // Take a look at memory address 0xFFE0. The interrupt vectors are redirected
  // Also look at the emulated PC register

  emul_cpu_regs = emulator(test_call_ret_break, 0);                             // Call, return and breakpoint test

  // testnumber =  5
  // The emulated PC register is now different because the breakpoint was not set

  // Note: If runnig in the IAR simulator, please be aware that the interrupt is not generated
  emul_cpu_regs = emulator(test_interrupt, 0);                                  // Interrupt and low power mode test

  // testnumber = 0 (changed inside the interrupt routine)

  emul_cpu_regs = emulator(test_addr_modes, 0);                                 // Move test (using all addressing modes)
  emul_cpu_regs = emulator(test_branch, 0);                                     // Branch test (using all addressing modes)
  emul_cpu_regs = emulator(test_jump, 0);                                       // Jump test (using all jump conditions)
  emul_cpu_regs = emulator(test_stackpointer, 0);                               // Stack pointer modification test

  // If emulator test are completed successfully run the flashing led demo in emulation mode

  emulator(flashingled, 0);                                                      // Flashing led demo

  while(1);
}

//******************************************************************************
// The following functions are emulated (interpeted by the local emulator).
// A breakpoint in the IAR emulator won't trigger a break if the code is
// emulated by the local emulator.
//******************************************************************************
void changetestnumber()
{
  testnumber += 4;
}

//******************************************************************************
// Dummy function, to be able to pass a known address to the emulator where to
// break
void breakpoint(){};

//******************************************************************************
// Test Call, Ret and breakpoint
void test_call_ret_break()
{
  breakpoint();          // <<- The emulator will break here and therefore won't execute the rest of function test_call_ret_break()
  changetestnumber();
}

//******************************************************************************
// Test interrupt
void test_interrupt()
{
  testnumber = 1;

  TACTL   = TAIE | TASSEL_2 | TACLR | ID_2;                                     // Overflow interrupt enable (TAIE), SMCLK (TASSEL_2) / 2  = 2.2MHz HFXTAL, clear TAR
  TACCTL0 = OUTMOD_3 | CCIE;                                                    // CCR0 interrupt enabled
  TACCR0  = 2000;

  __enable_interrupt();

  TACTL   |= MC1;                                                               // Start Timer_a in up continuous mode
  LPM0;                                                                         // Enter low power mode; Wait until the low power mode is exited in the interrupt

  __disable_interrupt();
}

//******************************************************************************
#pragma vector=TIMERA0_VECTOR
__interrupt void timera0()
{
  testnumber = 0;                                                               // Set testnumber
  TACTL      = 0;                                                               // stop timer
  LPM0_EXIT;                                                                    // Exit low power mode
}

//******************************************************************************
// Varibables decleration as the ones used in the MSP430 device specific datasheets examples
__no_init unsigned int EDE;
__no_init unsigned int TONI;
__no_init unsigned int MEM;
__no_init unsigned int TCDAT;

// Test all move addressing modes
void test_addr_modes()
{
  asm("mov R10, R11");                                                          // Register

  // Preperation of move instructions to have a valid source and destination addresses
  asm("mov #EDE, R5");
  asm("sub #2, R6");
  asm("mov #TONI, R6");
  asm("sub #6, R6");

  // Move indexed, symbolic and absolute
  asm("mov 2(R5), 6(R6)");                                                      // Indexed
  asm("mov EDE, TONI");                                                         // Symbolic
  asm("mov &MEM, &TCDAT");                                                      // Absolute

  // Addressing modes only available for source
  // Preperation of move instructions to have a valid source and destination addresses
  asm("mov #EDE, R10");
  asm("add #3, R6");

  // Move indirect, indirect auto increment, immediate
  asm("mov @R10, 3(R6)");                                                       // Indirect
  asm("mov @R10+, R11");                                                        // Indirect auto increment
  asm("mov #0x45, TONI");                                                       // Immediate
}

//******************************************************************************
// Varibable decleration as the one used in MSPFx1xx user guide brach examples
__no_init unsigned int EXEC;

// Test all branch addressing modes
// The branch sequence is -> branch3 -> branch2 -> branch4 -> branch7 -> branch5 -> branch6 -> branch8
void test_branch()
{
  testnumber = 1;

  // Branch to label EXEC or direct branch (e.g. #0A4h)
  asm("br #branch3");

  while(testnumber); // A while(1) to detect if the emulator is not branching correctly (testnumber variable is used because otherwise the compiler complains that the next statement is unreachable)

  asm("branch2:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov.w #branch4, &EXEC");
  // Branch to the address contained in EXEC
  asm("br EXEC");

  while(testnumber);

  asm("branch3:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov.w #branch2, &EXEC");
  // Branch to the address contained in absolute addres EXEC
  asm("br &EXEC");

  while(testnumber);

  asm("branch4:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov #branch7, R5");
  // Branch to the address contained in R5
  asm("br R5");

  while(testnumber);

  asm("branch5:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov #branch6, &EXEC");
  asm("mov #EXEC, R5");
  // Branch to the address contained in the word pointed by R5
  asm("br @R5");

  while(testnumber);

  asm("branch6:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov #branch8, &EXEC");
  asm("mov #EXEC, R5");
  // Branch to the address contained in the word pointed to by R5 and increment pointer in R5 afterwards.
  asm("br @R5+");

  while(testnumber);

  asm("branch7:");
  // Preperation of brach instructions to have a valid branch address
  asm("mov #branch6, &EXEC");
  asm("mov #EXEC, R5");
  asm("sub #2, R5");

  // Branch to the address contained in the address pointed to by R5 + 2
  asm("br 2(R5)");

  while(testnumber);

  asm("branch8:");
}

//******************************************************************************
// Jump test. Test all jump conditions
// If a jump condition is not handled correctly by the emulator it will get
// stuck inside this routine.
void test_jump()
{
  testnumber = 1;

  asm("mov #10, R5");

  asm("jumptest:");
  asm("jmp jump0");

  while(testnumber);

  asm("jump0:");
  asm("cmp R5, R5");
  asm("jne jump0");
  asm("jeq jump1");

  while(testnumber);

  asm("jump1:");
  asm("cmp #5, R5");
  asm("jeq jump1");
  asm("jne jump2");

  while(testnumber);

  asm("jump2:");
  asm("cmp #5, R5");
  asm("jnc jump2");
  asm("jc  jump3");

  while(testnumber);

  asm("jump3:");
  asm("cmp #20, R5");
  asm("jc  jump3");
  asm("jnc jump4");

  while(testnumber);

  asm("jump4:");
  asm("tst R5");
  asm("jz  jump4");
  asm("jnz jump5");

  while(testnumber);

  asm("jump5:");
  asm("mov #0, R6");
  asm("tst R6");
  asm("jnz jump5");
  asm("jz  jump6");

  while(testnumber);

  asm("jump6:");
  asm("cmp #5, R5");
  asm("jl  jump6");
  asm("jge jump7");

  while(testnumber);

  asm("jump7:");
  asm("cmp #20, R5");
  asm("jge jump7");
  asm("jl  jump8");

  while(testnumber);

  asm("jump8:");
}

//******************************************************************************
// Stack pointer modification test
void test_stackpointer()
{
  asm("mov #0x900, SP");
}

//******************************************************************************
// Flashing Led demo
void flashingled(void)
{
  WDTCTL = WDTPW + WDTHOLD;             // Stop watchdog timer
  P1DIR |= 0x01;                        // Set P1.0 to output direction

  for (;;)
  {
    volatile unsigned int i;            // volatile to prevent optimization

    P1OUT ^= 0x01;                      // Toggle P1.0 using exclusive-OR

    i = 1000;                           // SW Delay
    do i--;
    while (i != 0);
  }
}

