//******************************************************************************
//  MSP430F22x4 Demo - Executing code from RAM
//
//  Description: This program demonstrates how a parts of a program can be
//  executed from FLASH and RAM. First, the main program copies the function
//  from Flash to RAM during runtime. The main program executes from Flash and
//  it calls the RAM function and executes in RAM and finally jumps back to
//  Flash. The function first erases flash seg C, then it increments all values
//  in seg C, then it erases seg D, then copies seg C to seg D.
//
//  Important Notes:
//
//  1. CCE automatically generates a new copy of linker file in the project
//     directory. The zip file has the attached required modified linker file.
//     See below on what was added to the linker file.
//
//        MEMORY
//        {
//          ...
//          RAM_MEM         : origin = 0x0200, length = 0x0200
//          FLASH_MEM       : origin = 0x8000, length = 0x01FF
//          ...
//        }
//
//        SECTIONS
//        {
//          ...				    
//          .FLASHCODE : load = FLASH_MEM, run = RAM_MEM
//                                                /* CODE IN FLASH AND WILL BE COPIED
//                                                   TO RAM AT EXECUTION HANDLED BY
//                                                   USER                               */
//          .RAMCODE   : load = FLASH_MEM         /* CODE WILL BE IN RAM                */
//          ...
//        }
//
//  2. Unlike IAR where it can calculate the code size through the segment 
//     call functions, we have to define the allocated memory area that will
//     be copied from FLASH to RAM. In this case, user has to manually define
//     the start address of FLASH and RAM. These memory addresses has to be
//     the same as defined in the linker file origin address of FLASH_MEM and
//     RAM_MEM. The flash_code_size can be changed to however much the final
//     compiled code size is.
//
//          #define flash_start_add 0x8000              // Flash code starting address
//          #define flash_code_size 0x01ff              // Function segment size to be copied
//          #define RAM_start_add   0x0200              // RAM code starting address
//
//
//  3. Erasing and Writing to flash running from RAM requires a few checks of
//     the BUSY flag. See the user guide for the flow diagram.
//
//     while(FCTL3&BUSY);
//
//  ACLK = n/a, MCLK = SMCLK = CALxxx_1MHZ = 1MHz
//  //* Set Breakpoint on NOP in the Mainloop to avoid Stressing Flash *//
//
//
//  W. Goh & M. Morales & A. Dannenberg
//  Texas Instruments Inc.
//  December 2008
//  Built with CCE Version: 3.2.2.1.8
//******************************************************************************

#include "msp430x22x4.h"
#include "string.h"

// Define where the code is located in both Flash and RAM
#define flash_start_add 0x8000              // Flash code starting address
#define flash_code_size 0x01ff              // Function segment size to be copied
#define RAM_start_add   0x0200              // RAM code starting address

// Function prototypes
void write_SegC (char value);
void copy_C2D (void);

char value;                                 // 8-bit value to write to seg C

// This function is in flash to be copied to RAM
#pragma CODE_SECTION(write_SegC,".FLASHCODE")
void write_SegC (char value)
{
  char *Flash_ptr;                          // Flash pointer
  unsigned int i;
  
  P1OUT ^= 0x02;
  
  Flash_ptr = (char *) 0x1040;              // Initialize Flash pointer
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + ERASE;                    // Set Erase bit
  FCTL3 = FWKEY;                            // Clear Lock bit
  
  *Flash_ptr = 0;                           // Dummy write to erase Flash segment
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + WRT;                      // Set WRT bit for write operation

  for (i=0; i<64; i++)
  {
    while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
    *Flash_ptr++ = value;                   // Write value to flash
  }

  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY;                            // Clear WRT bit
  FCTL3 = FWKEY + LOCK;                     // Set LOCK bit
}


// This function is in flash to be copied to RAM
#pragma CODE_SECTION(copy_C2D,".FLASHCODE")
void copy_C2D (void)
{
  char *Flash_ptrC;                         // Segment C pointer
  char *Flash_ptrD;                         // Segment D pointer
  unsigned int i;

  Flash_ptrC = (char *) 0x1040;             // Initialize Flash segment C pointer
  Flash_ptrD = (char *) 0x1000;             // Initialize Flash segment D pointer
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + ERASE;                    // Set Erase bit
  FCTL3 = FWKEY;                            // Clear Lock bit
  *Flash_ptrD = 0;                          // Dummy write to erase Flash segment D
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + WRT;                      // Set WRT bit for write operation

  for (i=0; i<64; i++)
  {
    while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
    *Flash_ptrD++ = *Flash_ptrC++;          // copy value segment C to segment D
  }
  
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY;                            // Clear WRT bit
  FCTL3 = FWKEY + LOCK;                     // Set LOCK bit
}


//Copy function from Flash to RAM
void copy_flash_to_RAM(void)
{
  unsigned char *flash_start_ptr;           // Initialize pointers
  unsigned char *RAM_start_ptr;

  flash_start_ptr = (unsigned char *)flash_start_add;
  RAM_start_ptr = (unsigned char *)RAM_start_add;
  
  // Copy flash function to RAM 
  memcpy(RAM_start_ptr,flash_start_ptr,flash_code_size);
}


void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WatchDog timer
  if (CALBC1_1MHZ ==0xFF || CALDCO_1MHZ == 0xFF)                                     
  {  
    while(1);                               // If calibration constants erased
                                            // do not load, trap CPU!!
  }   
  BCSCTL1 = CALBC1_1MHZ;                    // Set DCO to 1MHz
  DCOCTL = CALDCO_1MHZ;
  FCTL2 = FWKEY + FSSEL_1 + FN1;            // MCLK/3 for Flash Timing Generator
  P1DIR |= 0x03;
  P1OUT |= 0x00;
  value = 0;
  
  copy_flash_to_RAM();                      // call function to copy from flash to RAM
    
  // Parts of this while loop executes from Flash. The write_SegC and copy_C2D
  // is executed from RAM. The program basically will execute from FLASH until
  // it calls write_SegC function which will jump the program to RAM.
  // After executing write_SegC in RAM, it will jump back to flash to continue
  // executing. When copy_C2D is called, it jumps to RAM and back to Flash when
  // done.
  while(1)
  {
  	  int i;
  	  
      P1OUT ^= 0x01;
      for(i=0;i<20000;i++);                 // This portion executes from FLASH
      write_SegC(value++);                  // This portion of the code is executed from RAM
      copy_C2D();                           // This portion of the code is executed from RAM
      __no_operation();                     // Jumps back to FLASH here. Set breakpoint here
  }
}



