%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% calc_3stage_FIRcoefficients_gc4016_woresampler.m
% filter generator for 3 stage DDC  
%
% Texas Instruments - JoeQuintal 11/14/09
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function calc_3stage_FIRcoefficients_gc4016_woresampler( )
   
   filtertype = 12;   % custom dec10, 160% band uwide mode

   infoldername = 'C:\Temp'; 
   
   % filter design for GC4016
   % 1 - normal % BW from <20% manual pfir and cfir Stopband rejection to get desired performance
   % 2 - normal % BW from 20 - 40%
   % 3 - normal % BW from 41 - 60%
   % 4 - normal % BW from 61 - 79%
   % 5 - normal % BW from 80 - 90% manual pfir and cfir Stopband rejection to get desired performance
   % 1-5 we change the equations for the pfir_Fs, and cfir_Fs
   
   % 6 - best performance, external PFIR - 63, CFIR - 19 design, can design
   %   63 tap PFIR, 19tap(3tap CIC corr) CFIR,
   %   readGCfilter = 1 - read GC filters with CFIR CIC corr, = 0 use low
   %   pass filters and add CFIR CIC corr
   
   % 7 - 1/2BW for UWMode from 91 - 110%
   % 8 - 1/2BW for UWMode from 111 - 120%
   % 9 - 1/2BW for UWMode from 121 - 126%
   % 10 - 1/2BW for UWMode > 126% manual pfir and cfir Stopband rejection to
   %    get desired performance, setup for 160%
   % 7-10 we change the equations for the pfir_Fs, and cfir_Fs
   
   % 11 - 1/2BW for UWMode, best performance, external PFIR - 63, CFIR - 19 design, can design
   %   63 tap PFIR, 19tap(3tap CIC corr) CFIR,
   %   readGCfilter = 1 - read GC filters with CFIR CIC corr, = 0 use low
   %   pass filters and add CFIR CIC corr

   % select the range from the above list that corresponds to the desired BW.  The 100%BW is based on the PFIR dec2, output rate/2.
   % enter the filtertype, infoldername at the top
   % enter in the filtertype section:
   %    cic_ratio - cic decimation, 
   %    GC4016 clockrate, 
   %    sig_BW - based on PFIR output rate / 2 is 100% bandwidth, more than
   %    90% bandwidth requires the UltraWide mode
   
   % you can adjust the pfir_Fs, cfir_Fs, (filter performance)
   % and ciccorrtaps (passband flatness 3 or 5 taps - normal 3)
   
   % note cic in rate = gc4016 clock
   %      cic out rate = gc4016 clock / cic_ratio 
   %      cfir in rate = cic out rate
   %      cfir out rate = cfir in rate / cfir ratio
   %      pfir in rate = cfir out rate
   %      pfir out rate = pfir in rate / 2
   %      resampler in rate = pfir out rate * (1 + UWmode)
   %      resampler out rate = resampler in rate * res_interp / res_dec
   %      sig_BW = 100 * pfir out rate / 2
   
   %%%%%%%%%%%%%%%%%%%%%%%%
   % general parameters - adjust for customer application
   cic_ratio = 8;      % CIC Decimation -4->1024 for splitIQ, 8-2048 for 4ch  
   clockrate = 32e6;   % GC4016 clock rate

   %%%%%%%%%%%%%%%%%%%%%%%
   % fixed parameters
   pfir_ratio = 2;     % PFIR Decimation
   pfir_oversample_factor = 1;   % PFIR output rate/symbol rate
   cic_stages = 5;     % this is fixed for GC4016          
   forcePFIRnum = 63;  % number of PFIR taps, this is fixed for GC4016 
   pfir_method = 1;    % 0 - LP, 1 - LP w/Window, 2 - RRC, 3- RRC w/Window, 4 - Hybrid
   pfir_Passbd = .05;  % passband ripple
   pfir_Stopbd = -90; % out of band atten
   pfir_alpha = .215;         % pulse shaping filter excess bandwidth (for RRC)
   PFIRcoef_bits = 16; % fixed point resolution
   forceCFIRnum = 21;  % number of CFIR taps, this is fixed for GC4016 
   cfir_ratio = 2;     % CFIR Decimation     
   cfir_method = 0;    % 0 - LP 
   cfir_Passbd = .05;   % passband ripple
   cfir_Stopbd = -90; % out of band atten
   cfir_coupled = 0;
   cfir_windowtype = 1.05;
   cic_corr_auto = 0;
   CFIRcoef_bits = 16; % fixed point resolution
   numRRCfil_taps = 64; % hybrid filter RRC taps
   cic_m = 1;
   m2stage_array(1: cic_stages) = [0,0,0,0,0];
   UWmode = 0;
   
   %%%%%%%%%%%%%%%%%%%%
   % note the reason we change the pfir_Fs and cfir_Fs is that the Matlab
   % remez and firls do not provide a good filter, if there are too few or
   % too many taps.  Too few and we have bad stopband rejection, too many
   % and we have bad passband response.  The CFIR is 1/6th the number of
   % taps for the same clock rate. Design goal for narrow filters is > 80db
   % stopband, wider filters > 65db stopband
   % minor pass band flatness is adjusted with cic correction.
   % note monitor correction to sig_BW is done because of windowed
   % coefficients
  
   if( filtertype == 1)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters <20%, need to adjust manual pfir_Fs, cfir_Fs
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      sig_BW = 20; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      pfir_autostop = 1;
      pfir_windowtype = 1.005;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      pfir_Fs = pfir_Fp * 3;   % stopband frequency, set for max stopband freq
  
      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
  
      cfir_Fp = pfir_Fp * 1.05;     % passband frequency
      cfir_autostop = 1;
      cfir_Fs = pfir_Fp * 12;   % stopband frequency, set for max stopband freq
      % cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * 2)) - (pfir_Fp * .1);    % reject CIC image to PFIR bandwidth
      CFIRtap_filename_in = strcat(infoldername,'\',' ' );  % used to read in CFIR coef
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
       numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
       ciccorrtaps(1:numtaps) = [-.15, 1.2, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end

   if( filtertype == 2)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters 21 - 40
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      sig_BW = 39; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      pfir_windowtype = 1.005;
      pfir_autostop = 1;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1.4 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end
      
      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
 
      cfir_Fp = pfir_Fp * 1.1;     % passband frequency
      cfir_autostop = 1;
      if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 3.0 + (1 - (sig_BW/100)));
         temp2 = pfir_Fp * 6.0;
         cfir_Fs = max(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end
      CFIRtap_filename_in = strcat(infoldername,'\',' ' );  % used to read in CFIR coef
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
       numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
       ciccorrtaps(1:numtaps) = [-.15, 1.2, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end
   
   if( filtertype == 3)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters 41 - 60
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      sig_BW = 60; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      pfir_autostop = 1;
      pfir_windowtype = 1.005;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1.2 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end
      
      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
 
      cfir_Fp = pfir_Fp * 1.05;     % passband frequency
      cfir_autostop = 1;
      if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 2.5 + (1 - (sig_BW/100)));
         temp2 = pfir_Fp * 5.0;
         cfir_Fs = max(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end
      CFIRtap_filename_in = strcat(infoldername,'\',' ' );  % used to read in CFIR coef
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
       numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
       ciccorrtaps(1:numtaps) = [-.15, 1.2, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end
   
   if( filtertype == 4)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters 61 - 79
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      sig_BW = 79; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      pfir_autostop = 1;
      pfir_windowtype = 1.005;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1.15 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end
      
      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
      cfir_Fp = pfir_Fp * 1.05;     % passband frequency
      cfir_autostop = 1;
      if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.4 + (1 - (sig_BW/100)));
         temp2 = pfir_Fp * 3.9;
         cfir_Fs = max(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end
      CFIRtap_filename_in = strcat(infoldername,'\',' ' );  % used to read in CFIR coef
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
       numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
       ciccorrtaps(1:numtaps) = [-.15, 1.2, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end

   if( filtertype == 5)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters 80 - 90%
      % manual pfir_Fs, cfir_Fs
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      sig_BW = 90; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 
      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      pfir_autostop = 1;
      pfir_windowtype = 1.005;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1.1 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end

      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
 
      cfir_Fp = pfir_Fp * 1.05;     % passband frequency
      cfir_autostop = 1;
      if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.2 + (1 - (sig_BW/100)));
         temp2 = pfir_Fp * 3.2;
         cfir_Fs = max(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end
      CFIRtap_filename_in = strcat(infoldername,'\',' ' );  % used to read in CFIR coef
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
      numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      ciccorrtaps(1:numtaps) = [-.15, 1.2, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end

   if( filtertype == 6)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % read filters from file, option to cic correct the CFIR
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      readGCfilter = 0;   % this control is used  = 1 when reading prebuilt gc4016 filters with cfir cic correction
      
      sig_BW = 68; %desired passband BW of the PFIR output signal
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      pfir_method = 5;    % 0 - LP, 1 - LP w/Window, 2 - RRC, 3- RRC w/Window, 4 - Hybrid, 5 - read taps from file
      pfir_autostop = 1;
      pfir_windowtype = 1.005;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end   
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1.1 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end

      if(readGCfilter == 1)
         PFIRtap_filename_in = strcat(infoldername,'\','pfir_68.taps' );  % used to read in CFIR coef
         CFIRtap_filename_in = strcat(infoldername,'\','cfir_68.taps' );  % used to read in CFIR coef
         no_ciccor = 1;
         numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         ciccorrtaps(1:numtaps) = [-.15, 1.15, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
      else    % correct manual low pass filter for CIC correction, % normal %bw is at -0.5db, w/ droop correction, usually want 2-3% more
         PFIRtap_filename_in = strcat(infoldername,'\','pfir_fck2_fpp34_fsp46_63.taps' );  % used to read in CFIR coef
         CFIRtap_filename_in = strcat(infoldername,'\','cfir_fck4_fpp36_fs1p1_19.taps' );  % used to read in CFIR coef
         no_ciccor = 0;
         % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         % ciccorrtaps(1:numtaps) = [0.03, -.365, 1.14, -.365, 0.03]; 
         numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         ciccorrtaps(1:numtaps) = [-.15, 1.15, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
      end   
         
      PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
      CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
      Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convrespman_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));

      cfir_method = 5;    % 0 - LP, , 5 - read taps from file 
      cfir_Fp = pfir_Fp * 1.05;     % passband frequency
      cfir_autostop = 1;
      if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.2 + (1 - (sig_BW/100)));
         temp2 = pfir_Fp * 3.2;
         cfir_Fs = max(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end

   end
   
   if( filtertype == 7)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode from 91 - 110%
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      UWmode = 1;
      sig_BW = 110; %desired passband BW of the PFIR output signal

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      CFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in CFIR coef
      PFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in PFIR coef

      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWMode_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.1;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( 1 + (1 - (sig_BW/100)))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.05);     % passband frequency
      cfir_autostop = 1;
     if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.45 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 2.75;
         cfir_Fs = min(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
      end

     no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
      numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      ciccorrtaps(1:numtaps) = [-.15, 1.0, -.15];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end

  if( filtertype == 8)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode from 111 - 120%
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      UWmode = 1;
      sig_BW = 120; %desired passband BW of the PFIR output signal

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      CFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in CFIR coef
      PFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in PFIR coef

      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWMode_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.1;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( .65 + (1 - sig_BW/100))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) );   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.05);     % passband frequency
      cfir_autostop = 1;
     if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.8 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 3.15;
         cfir_Fs = min(temp1, temp2);
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * .25) )  % stopband frequency, set for max stopband freq
     end

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
      numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      ciccorrtaps(1:numtaps) = [-.155, 1.0, -.155];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
  end
   
  if( filtertype == 9)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode, man adjust for BW 121 to 126% 
      % manual pfir_Fs, cfir_Fs
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      UWmode = 1;
      sig_BW = 126; %desired passband BW of the PFIR output signal

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      CFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in CFIR coef
      PFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in PFIR coef

      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWMode_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.05;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( .8 + (1 - sig_BW/100))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) - .05e6;   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.03);     % passband frequency
      cfir_autostop = 1;
     if( cfir_autostop == 1)
         temp1 = pfir_Fs * ( 1.55 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 2.25;
         cfir_Fs = min(temp1, temp2) 
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * 2) ) - .05e6  % stopband frequency, set for max stopband freq
     end

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
      numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      ciccorrtaps(1:numtaps) = [-.16, 1.05, -.16];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
   end

  if( filtertype == 10)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode, man adjust for BW 127 to 160% 
      % manual pfir_Fs, cfir_Fs. adjusted to balance bandwidth (PFIR Fs) and CIC
      % rejection (CFIR Fs)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      UWmode = 1;
      sig_BW = 160; %desired passband BW of the PFIR output signal

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      CFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in CFIR coef
      PFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in PFIR coef

      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWMode_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.05;
      pfir_Passbd = .2;  % passband ripple
      pfir_Stopbd = -60; % out of band atten
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( .85 + (1 - sig_BW/100))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) - .05e6;   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.02);     % passband frequency
      cfir_Passbd = .2;  % passband ripple
      cfir_Stopbd = -60; % out of band atten
      cfir_autostop = 1;
     if( cfir_autostop == 1)
%          temp1 = pfir_Fs * ( 1.35 + (1 - sig_BW/100) ); %150
%          temp2 = pfir_Fp * 2.05;
         temp1 = pfir_Fs * ( 1.23 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 1.91;
         cfir_Fs = min(temp1, temp2) 
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * 2) ) - .05e6  % stopband frequency, set for max stopband freq
     end

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
      numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      ciccorrtaps(1:numtaps) = [-.17, 1.1, -.17];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
  end
   
   if( filtertype == 11)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode, manual PFIR and CFIR tap input 
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      readGCfilter = 0;   % this control is used  = 1 when reading prebuilt gc4016 filters with cfir cic correction

      UWmode = 1;

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      pfir_method = 5;    % 0 - LP, 1 - LP w/Window, 2 - RRC, 3- RRC w/Window, 4 - Hybrid, 5 - read taps from file
      % note cic in rate = gc4016 clock
      %      cic out rate = gc4016 clock / cic_ratio 
      %      cfir in rate = cic out rate
      %      cfir out rate = cfir in rate / cfir ratio
      %      pfir in rate = cfir out rate
      %      pfir out rate = pfir in rate / 2
      %      resampler in rate = pfir out rate * (1 + UWmode)
      %      resampler out rate = resampler in rate * res_interp / res_dec
      PFIRtap_filename_in = strcat(infoldername,'\','pfir_fck2_fpp8_fsp87_63.taps' );  % used to read in PFIR coef
      cfir_method = 5;    % 0 - LP, , 5 - read taps from file 
      CFIRtap_filename_in = strcat(infoldername,'\','cfir_fck4_fpp82_fs1p2_19.taps' );  % used to read in CFIR coef

      if(readGCfilter == 1)
         sig_BW = 150; %desired passband BW of the PFIR output signal
         PFIRtap_filename_in = strcat(infoldername,'\','pfir_150.taps' );  % used to read in CFIR coef
         CFIRtap_filename_in = strcat(infoldername,'\','cfir_150.taps' );  % used to read in CFIR coef
         no_ciccor = 1;
         numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         ciccorrtaps(1:numtaps) = [-.18, 1.05, -.18];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
      else    % correct manual low pass filter for CIC correction, % normal %bw is at -0.5db, w/ droop correction, usually want 2-3% more
         sig_BW = 160; %desired passband BW of the PFIR output signal
         PFIRtap_filename_in = strcat(infoldername,'\','pfir_fck2_fpp8_fsp87_63.taps' );  % used to read in CFIR coef
         CFIRtap_filename_in = strcat(infoldername,'\','cfir_fck4_fpp82_fs1p2_19.taps' );  % used to read in CFIR coef
         no_ciccor = 0;
         % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
         numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
         ciccorrtaps(1:numtaps) = [-.18, 1.05, -.18];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
      end   
      
      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWModeman_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016man_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convrespman_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.05;
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
        % pfir_Fs =  pfir_Fp * ( .8 + (1 - sig_BW/100)) %150
         pfir_Fs =  pfir_Fp * ( .8 + (1 - sig_BW/100))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) - .05e6;   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.03);     % passband frequency
      cfir_autostop = 1;
     if( cfir_autostop == 1)
%          temp1 = pfir_Fs * ( 1.35 + (1 - sig_BW/100) ); %150
%          temp2 = pfir_Fp * 2.05;
         temp1 = pfir_Fs * ( 1.3 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 2.0;
         cfir_Fs = min(temp1, temp2) 
      else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * 2) ) - .05e6  % stopband frequency, set for max stopband freq
     end
   end

   if( filtertype == 12)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % channel / filter design parameters UW mode, man adjust for BW 127 to 160% 
      % manual pfir_Fs, cfir_Fs. adjusted to balance bandwidth (PFIR Fs) and CIC
      % rejection (CFIR Fs)
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      cic_ratio = 5;      % CIC Decimation -4->1024 for splitIQ, 8-2048 for 4ch  
      clockrate = 100e6;   % GC4016 clock rate

      UWmode = 1;
      sig_BW = 160; %desired passband BW of the PFIR output signal

      pfir_ratio = 1;     % PFIR Decimation
      chdec = cic_ratio * cfir_ratio * pfir_ratio;
      symbol_rate = clockrate / (cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor) % assumes resampler is bypassed 

      CFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in CFIR coef
      PFIRtap_filename_in = strcat(infoldername,'\','' );  % used to read in PFIR coef

      if(UWmode == 1)
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfirUW4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_UWMode_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
         sig_BW = sig_BW/2; 
      else
         PFIRtap_filename = strcat(infoldername,'\',sprintf('pfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         CFIRtap_filename = strcat(infoldername,'\',sprintf('cfir4016_%d.taps', fix(sig_BW)) );  % destination of PFIR coef file
         Convtap_filename = strcat(infoldername,'\',sprintf('gc4016convresp_dec%d_sigBW%d.taps', chdec, fix(sig_BW) ));
      end
      
      pfir_autostop = 1;
      pfir_windowtype = 1.05;
      pfir_Passbd = .2;  % passband ripple
      pfir_Stopbd = -60; % out of band atten
      if(pfir_method == 1) 
         sig_BW = sig_BW / pfir_windowtype;
      end 
          
      pfir_Fp = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) * (sig_BW / 100);   % passband frequency, % of PFIR output rate * sig_BW
      if( pfir_autostop == 1)
         pfir_Fs =  pfir_Fp * ( .85 + (1 - sig_BW/100))
      else
         pfir_Fs = (clockrate / (cic_ratio * cfir_ratio * pfir_ratio * 2) ) - .05e6;   % stopband frequency, set for max stopband freq
      end
      
      cfir_Fp = (pfir_Fp * 1.02);     % passband frequency
      cfir_Passbd = .2;  % passband ripple
      cfir_Stopbd = -60; % out of band atten
      cfir_autostop = 1;
     if( cfir_autostop == 1)
%          temp1 = pfir_Fs * ( 1.35 + (1 - sig_BW/100) ); %150
%          temp2 = pfir_Fp * 2.05;
         temp1 = pfir_Fs * ( 1.23 + (1 - sig_BW/100) );
         temp2 = pfir_Fp * 1.91;
         cfir_Fs = min(temp1, temp2) 
     else
         cfir_Fs = (clockrate / (cic_ratio * cfir_ratio * 2) ) - .05e6  % stopband frequency, set for max stopband freq
     end

      no_ciccor = 0;
      % numtaps = 5; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
      % ciccorrtaps(1:numtaps) = [0.04, -.30, 1.24, -.30, 0.04]; 
       numtaps = 3; %number of CIC compensation taps 3 (narrower band/higher CIC dec) or 5 taps (wider band/low CIC decimation, WiMAX)
       ciccorrtaps(1:numtaps) = [-.173, 1.0, -.173];  % note this has to be manually adjusted to flatten the overall passband (CIC correction)
      
  end   

 
   % END OF INPUT PARAMETERS
   % CALCULATIONS BEGIN

   clock_rate = cic_ratio * cfir_ratio * pfir_ratio * pfir_oversample_factor * symbol_rate;
   
   %%%%%%%%%%%%%%%%%%% start of code section %%%%%%%%%%%%%%%%%%%%%%%%   
   % constants and fudge factors 
   Remez_fudge = 60;  % fudge factor for Remez versus Least Squares calc
   
   LPFilter = 0;
   LPFilter_Window = 1;
   RRC = 2;
   RRC_Window = 3;
   Hybrid = 4;
   Readtaps_fromfile = 5;
   
   Firres_bits = PFIRcoef_bits;
   FIRmax = 2^ (Firres_bits - 1) - 1;
   
   numPFIRtaps = forcePFIRnum;
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
   % determine Filter clock rates
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   CFIRClk = clock_rate / cic_ratio;
   PFIRClk = clock_rate / (cic_ratio * cfir_ratio);
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % generate CIC correction array
   % and CIC filter response
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   
   CICclockrate = clock_rate;  %Rate of CIC clock, i.e. after interpolation or before decimation = CIC I/O rate
   bandwidth = sig_BW;         %Desired Passband bandwidth
   CICstage = cic_stages;      %5 for DDC, 6 for DUC
   m=cic_m;                    %CIC delay
   CICrate = cic_ratio;        %Interpolation/Decimation rate
   resolution = 200;           %resolution of search matrix
   plotripple = 1;             %1 = plot ripple across bandwidth
   savetaps = 0;               %1 = save text file of taps
   filepath='C:\Temp\CICcomp.txt'; %CIC Compensation filter coefficients

   CICtaps = GenCICTaps_indivm2( cic_stages, cic_ratio, cic_m, m2stage_array);
   if(no_ciccor == 0)
      numCFIRtaps = forceCFIRnum - numtaps + 1;
   else
      numCFIRtaps = forceCFIRnum;
   end    
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % normalize frequency and amplitude
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(cfir_coupled == 1)
      cfir_Fp = pfir_Fp * 1.05;    
      cfir_Fs = PFIRClk - (pfir_Fp * 1.1);  % CFIR stopband set to attn PFIR image
   end
   freq(1:4) = [0, cfir_Fp, cfir_Fs, (CFIRClk /2)];
   ampldbV(1:4) = [0, cfir_Passbd, cfir_Stopbd, cfir_Stopbd];

   % convert freq[] to normalized freq
   % convert amplitude in dbV to ratio, assumes same number of points as freq[]
   for n = 1: 4
      norm_freq(n) = freq(n) / (CFIRClk /2);  % normalize to filter samplerate
      if(n < 4)
         ratio_ampl(n) = 10^( ampldbV(n) / 20);
      else
         ratio_ampl(n) = 0;
      end   
   end
         
   % weighting vector uses passband ripple for quality factor
   w(1: 2) = [10, 10];

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % run specific filter mode
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(cfir_method == LPFilter)
      norm_freq      
      % generate taps  
      if( numCFIRtaps < Remez_fudge)
         CFIRtaps = remez( numCFIRtaps-1, norm_freq, ratio_ampl, w); % remez exchange   
      else    
         CFIRtaps = firls( numCFIRtaps-1, norm_freq, ratio_ampl, w); % Least Square error
      end  
   end 
      
   if(cfir_method == LPFilter_Window) 

      % generate taps  
      if( numCFIRtaps < Remez_fudge)
         CFIRtaps = remez( numCFIRtaps-1, norm_freq, ratio_ampl, w); % remez exchange   
      else    
         CFIRtaps = firls( numCFIRtaps-1, norm_freq, ratio_ampl, w); % Least Square error
      end  
 
      % windowing
      % 0 - none, 1 - Hamming, >1 - KaiserBessel Alpha
      flen = length(CFIRtaps);
      w(1: flen) = 1;

      if(cfir_windowtype == 1)
         w = hamming(flen);                      
      end       
      if( cfir_windowtype > 1)
         w = kaiser( flen, cfir_windowtype);                      
      end    
      for n = 1: flen
         CFIRtaps(n) = CFIRtaps(n) * w(n);    
      end
   end
   
   if(cfir_method == Readtaps_fromfile)
      CFIRtaps = read_filterdat( CFIRtap_filename_in);    
   end
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % PFIR section
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   freq(1:4) = [0, pfir_Fp, pfir_Fs, (PFIRClk /2)];
   ampldbV(1:4) = [0, pfir_Passbd, pfir_Stopbd, pfir_Stopbd];

   % convert freq[] to normalized freq
   % convert amplitude in dbV to ratio, assumes same number of points as freq[]
   for n = 1: 4
      norm_freq(n) = freq(n) / (PFIRClk /2);  % normalize to filter samplerate
      if(n < 4)
         ratio_ampl(n) = 10^( ampldbV(n) / 20);
      else
         ratio_ampl(n) = 0;
      end   
   end
         
   % weighting vector uses passband ripple for quality factor
   % RRC filter parameters
   w(1: 2) = [1/pfir_Passbd, 10];
   numPFIRtaps = forcePFIRnum;
   sampleratio = symbol_rate / PFIRClk;
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % run specific filter mode
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(pfir_method == LPFilter)
            
      % generate taps  
      if( numPFIRtaps < Remez_fudge)
         PFIRtaps = remez( numPFIRtaps-1, norm_freq, ratio_ampl, w); % remez exchange   
      else    
         PFIRtaps = firls( numPFIRtaps-1, norm_freq, ratio_ampl, w); % Least Square error
      end  
   end 
      
   if(pfir_method == LPFilter_Window) 

      % generate taps
      norm_freq
      if( numPFIRtaps < Remez_fudge)
         PFIRtaps = remez( numPFIRtaps-1, norm_freq, ratio_ampl, w); % remez exchange   
      else    
         PFIRtaps = firls( numPFIRtaps-1, norm_freq, ratio_ampl, w); % Least Square error
      end  
 
      % windowing
      % 0 - none, 1 - Hamming, >1 - KaiserBessel Alpha
      flen = length(PFIRtaps);
      w(1: flen) = 1;

      if(pfir_windowtype == 1)
         w = hamming(flen);                      
      end       
      if(pfir_windowtype > 1)
         w = kaiser( flen, pfir_windowtype);                      
      end    
      for n = 1: flen
         PFIRtaps(n) = PFIRtaps(n) * w(n);    
      end
   end
   
   if(pfir_method == RRC)
      % generate closed form taps 
      PFIRtaps = Calc_RRCTaps( sampleratio, pfir_alpha, FIRmax, numPFIRtaps);
   end
      
   if(pfir_method == RRC_Window)
      % windowing excessBW compensation
       pfir_alpha =  pfir_alpha / pfir_windowtype; 
               
      % generate closed form taps 
      PFIRtaps = Calc_RRCTaps( sampleratio, pfir_alpha, FIRmax, numPFIRtaps);
             
      % windowing
      % 0 - none, 1 - Hamming, >1 - KaiserBessel Alpha
               
      flen = length(PFIRtaps);
      w(1: flen) = 1;
      if(pfir_windowtype == 1)
         w = hamming(flen);                      
      end       
      if(pfir_windowtype > 1)
         w = kaiser( flen, pfir_windowtype);                      
      end
      for n = 1: flen
         PFIRtaps(n) = PFIRtaps(n) * w(n);    
      end
   end
   
   if(pfir_method == Hybrid) 
      % divide the number of taps between the RRC filter and the Low Pass Filter
      num_LPtaps = numPFIRtaps - numRRCfil_taps + 1;
      
      % generate closed form taps
      RRCtaps = Calc_RRCTaps( sampleratio, pfir_alpha, FIRmax, numRRCfil_taps);
            
      symbolrate = PFIRClk * sampleratio;
      freq(1:4) =    [0, pfir_Fp, pfir_Fs, (PFIRClk /2)];
      ampldbV(1:4) = [0,  -.1, pfir_Stopbd, pfir_Stopbd]; 

      % convert freq[] to normalized freq
      % convert amplitude in dbV to ratio, assumes same number of points as freq[]
      for n = 1: 4
         norm_freq(n) = freq(n) / (PFIRClk /2);  % normalize to filter samplerate
         if(n < 4)
            ratio_ampl(n) = 10^( ampldbV(n) / 20);
         else
            ratio_ampl(n) = 0;
         end   
      end
      w(1: 2) = [10, 10];
             
      if( num_LPtaps < Remez_fudge)
         LPtaps = remez( num_LPtaps -1, norm_freq, ratio_ampl, w); % remez exchange   
      else    
         LPtaps = firls( num_LPtaps -1, norm_freq, ratio_ampl, w); % Least Square error
      end 
            
      % combine the RRC + Low Pass Stopband suppression
      PFIRtaps = conv(LPtaps, RRCtaps);
   end
   
   if(pfir_method == Readtaps_fromfile)
      PFIRtaps = read_filterdat( PFIRtap_filename_in);    
   end

   % round and scale
   sigstat(1:2) = [max(PFIRtaps) (min(PFIRtaps) * -1) ];
   PFIRtaps = convrnd( PFIRtaps * FIRmax / max(sigstat), PFIRcoef_bits );
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % plot filter results
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   flen = max(size(PFIRtaps));
      
   X(1: flen) = 1: 1: flen;
   figure(2);
   plot(X, PFIRtaps,'.b');
   titletext = sprintf('PFIR filter, %d taps', flen);
   title( titletext);
   zoom on;
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % real FFT of taps
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   figure(3);
   titletext = sprintf('PFIR filter');

   FFT64KNoFig( PFIRtaps, PFIRClk, titletext); 
   zoom on;
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % interpolate PFIR taps by CFIR_ratio
   % Convolve Interp_PFIR with CFIR
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   InterpPFIRtaps= zeros(1,length(PFIRtaps) * cfir_ratio);
   InterpPFIRtaps( 1 : cfir_ratio : length(PFIRtaps) * cfir_ratio)= PFIRtaps;
   InterpPFIRCFIR = conv(CFIRtaps, InterpPFIRtaps); 
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % finish the CIC correction, repeat the previous convolution
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(cic_corr_auto  == 1) 
%      compfilter = ciccomp3or5taps(CICclockrate,bandwidth,CICstage,m,CICrate,numtaps,resolution,plotripple,savetaps,filepath);
      compfilter = ciccomp3or5taps(CICclockrate,bandwidth,CICstage,m,CICrate,numtaps,resolution,plotripple,savetaps,filepath, InterpPFIRCFIR, m2stage_array)
   else
      compfilter = ciccorrtaps;
   end   

   %%%%%%%%%%%%%%%%%%%%%%%%%
   % convolve CIC Correction
   %%%%%%%%%%%%%%%%%%%%%%%%%
   if(no_ciccor == 0)
      CFIRtaps = conv( CFIRtaps, compfilter);
   end
   
   %%%%%%%%%%%%%%%%%%%%%%%%%
   % round and scale
   %%%%%%%%%%%%%%%%%%%%%%%%%
   sigstat(1:2) = [max(CFIRtaps) (min(CFIRtaps) * -1) ];
   CFIRtaps = convrnd( CFIRtaps * FIRmax / max(sigstat), CFIRcoef_bits );

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % plot filter results
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   flen = max(size(CFIRtaps));
      
   X = 1: 1: flen;
   figure(4);
   plot(X, CFIRtaps,'.b');
   titletext = sprintf('CFIR filter, %d taps', flen);
   title( titletext);
   zoom on;

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % interpolate PFIR taps by CFIR_ratio
   % Convolve Interp_PFIR with CFIR
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   InterpPFIRtaps= zeros(1,length(PFIRtaps) * cfir_ratio);
   InterpPFIRtaps( 1 : cfir_ratio : length(PFIRtaps) * cfir_ratio)= PFIRtaps;
   InterpPFIRCFIR = conv(CFIRtaps, InterpPFIRtaps); 
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%
   % interpolate by CIC, convolve, and decimate to show true CFIR
   % CIC-correction response
   %%%%%%%%%%%%%%%%%%%%%%%%%%
   
   InterpCFIRtaps= zeros(1,length(CFIRtaps) * cic_ratio);
   InterpCFIRtaps( 1 : cic_ratio : length(CFIRtaps) * cic_ratio)= CFIRtaps;
   Convtaps = conv(CICtaps, InterpCFIRtaps); 
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % generate equivalent PFIR taps by decimating by cic_ratio
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   dectaps = polyphase_dec_max(Convtaps, cic_ratio);

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % real FFT of taps
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   figure(5);
   titletext = sprintf('CFIR filter, convolved with CIC, and decimated back to CFIR rate');
   dectaps = scalereal( dectaps, FIRmax); 

   FFT64KNoFig( dectaps, CFIRClk, titletext); 
   zoom on;   

   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % interpolation PFIR/CFIR taps by CIC_ratio
   % convolve Interp_PFIR/CFIR with CIC
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   InterpPFIRCFIRtaps = zeros(1, length(InterpPFIRCFIR) * cic_ratio);
   InterpPFIRCFIRtaps( 1 : cic_ratio : length(InterpPFIRCFIR) * cic_ratio)= InterpPFIRCFIR;
   InterpPFIRCFIRCIC = conv( CICtaps, InterpPFIRCFIRtaps);
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % real FFT of taps
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   figure(6);
   titletext = sprintf('PFIR filter, convolved with CFIR, convolved with CIC' );
   FFTC64KWinNoFig( InterpPFIRCFIRCIC,  clock_rate, titletext);
   zoom on;

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % plot and FFT Convolved Taps
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   figure(7);
   titletext = 'Convolved channel taps, estimated delay, w/o pipeline and program delays';
   clen = length(InterpPFIRCFIRCIC);
   ck_period = 1/ clock_rate;
   X(1: clen) = 0: ck_period: (clen - 1) * ck_period;
   plot( X, InterpPFIRCFIRCIC,'b.');
   title( titletext);
   zoom on;
      
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % export FIR coefficients to file
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   write_filterd( PFIRtap_filename, PFIRtaps);
   write_filterd( CFIRtap_filename, CFIRtaps);
   write_filter( Convtap_filename, InterpPFIRCFIRCIC);
  
return;
      
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write filter taps, single column vector
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function write_filter( filename, filtertaps)
   fid= fopen(filename,'w');
   DataLen = length(filtertaps);

   % print to file a line at a time
   for n = 1: DataLen
      fprintf( fid, '%12.5e \n', filtertaps(n) );
   end
   fclose(fid);
return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write filter taps dec, single column vector
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function write_filterd( filename, filtertaps)
   fid= fopen(filename,'wt');
   DataLen = length(filtertaps);

   % print to file a line at a time
   for n = 1: DataLen
      fprintf( fid, '%d \n', filtertaps(n) );
   end
   fclose(fid);
return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% computes Root Raised Cosine array, windows, scales
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function RRCtaps = Calc_RRCTaps( sampleratio, alpha, centertap, numtaps)
   
   titletext = sprintf('RRC FIR Response %d taps', numtaps);
   
   % RRC calculation
   % formula is:
   %   
   % tap(k) = [CT/((pi(1-A)+4A)]*
   %          [1/(1-(4AkR)**2)]* 
   %          [sin(kRpi(1-A))/kR + 4Acos(kRpi(1+A))]
   %    

   gain = centertap / ( pi*(1.0 - alpha) + (4.0 * alpha) );
 
   RRCtaps(1: numtaps) = 0.0;
   Fudge = 1.0e-10;

   totaltaps = 0.0;
   for k = 0 : (numtaps - 1)
      
      % second term 
      a = Fudge + sampleratio * (numtaps - (2*k) - 1.0)/2.0;
      t = 4.0 * alpha * a;
      tg = gain / (1.0-(t*t)+Fudge);
      s= sin(pi * (1.0 - alpha) * a);
      c = cos(pi * (1.0 + alpha) * a);
      
      % third term and combination
      RRCtaps(k+1) = tg * ((s/a) + (4.0*alpha*c));
   end  % next, completes loop calculation

   % re-scale for centertap value
   maxtap = max(RRCtaps);
   tapscale = centertap / maxtap;
   RRCtaps(1: numtaps) = RRCtaps(1: numtaps) * tapscale;

return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calc CIC Taps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function CICtaps = GenCICTaps( cic_stages, cic_ratio, cic_m)

   % determine number of loops
   numConv = cic_stages - 1;
   
   % determine boxcar equiv
   boxtaps = ones((cic_ratio * cic_m), 1);
   
   % convolve n - (1,2) stages
   a = boxtaps;
   for n = 1 : numConv
      a = conv(a, boxtaps);
   end   
   
   CICtaps = a;
return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calc CIC Taps - allows for CICM=2 on individual stages
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function CICtaps = GenCICTaps_indivm2( cic_stages, cic_ratio, cic_m, m2stage_array)

   % determine boxcar equiv
   boxtaps1 = ones((cic_ratio * 1), 1);
   boxtaps2 = ones((cic_ratio * 2), 1);
   
   % convolve n - (1,2) stages
   if(m2stage_array(1) == 1)
      a = boxtaps2;
   else
      a = boxtaps1;
   end
   
   for n = 2 : cic_stages
      if( m2stage_array(n) == 1)
         a = conv(a, boxtaps2);
      else
         a = conv(a, boxtaps1);
      end   
   end   
   
   CICtaps = a;
return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% rescale complex array to max value
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function Cscaled_array = scalecomplex( inputC_array, newmaxvalue)
   maxI = real( max( inputC_array) );
   minI = -1 * real( min( inputC_array) );
   maxQ = imag( max( inputC_array) );
   minQ = -1 * imag( min( inputC_array) );
   
   arraystat(1: 4) = [maxI, minI, maxQ, minQ];
   
   scalef = newmaxvalue / max(arraystat);
   Cscaled_array = scalef * inputC_array;
return; 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RealFFT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function FFT64KNoFig( d, FClk, titletext)
   
   FFTsize = 65536;
   lend = length(d);
   b(1: FFTsize) = 0;                       % zero fill the array
   b(1: lend) = d(1: lend);

   F= fft(b, FFTsize);                      % FFT h(t)
   F= F(1: FFTsize/2);                       % We only need the first half of the FFT

   Xmax = FClk/2;
   X= linspace(0, Xmax, FFTsize/2);

   F= F + (realmin + realmin*i);             % This prevents log(0) warnings
   FF= 20*log10(abs(F));                     % Compute log of magnitude

   maxFF = max(FF);
   FF = FF - maxFF;                          % dBc 

   plot(X, FF);
   title( titletext);
   axis( [0 Xmax (max(FF)-120) max(FF)]); grid;            % Plot the data
return;

function rnd_array = convrnd(in_array, numbits)

   dlen = length(in_array);
   rnd_array(1: dlen) = 0;
   
   maxpos = 2^(numbits-1) - 1;
   maxneg = (-1 * maxpos) - 1;
   halfpos = 2^(numbits-2);
   halfneg = -1 * halfpos;
   pos_status = 0;
   neg_status = 0;
   
   for n = 1: dlen
      if (in_array(n) >= maxpos)
         rnd_array(n) = maxpos;
      elseif (in_array(n) <= maxneg)
         rnd_array(n) = maxneg;
      elseif (in_array(n) == halfpos)
         if(pos_status == 1)
             rnd_array(n) = fix(in_array(n));
             pos_status = 0;
         else
             rnd_array(n) = fix(in_array(n) + 0.5);
             pos_status = 1;
         end
      elseif (in_array(n) == halfneg)
          if(neg_status == 1)
             rnd_array(n) = fix(in_array(n));
             neg_status = 0;
         else
             rnd_array(n) = fix(in_array(n) + 0.5);
             neg_status = 1;
          end
      else
          rnd_array(n) = fix(in_array(n) + 0.5);
      end
   end   

return;

function dectaps = polyphase_dec_max(Convtaps, decimation_ratio) 
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   % decimate the convolved response of the PFIR and CIC back to the PFIR
   % rate, using the decimation phase, select the highest sum of abs(taps)
   % as the reference phase for the test
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   dlen = length(Convtaps);
   polyphaselen = fix(dlen / decimation_ratio);
   endarray = polyphaselen * decimation_ratio;
   
   % there are cic_interpolation polyphases, only one will be processed for
   % testing, the max sum(abs(taps)) 
   index = 1;
   maxsum = 0;
      
   for n = 1: decimation_ratio
      Decphase_array = CDecimate(Convtaps(1: endarray), decimation_ratio, n);
      dlen = length(Decphase_array);
      temp = 0;
      for m = 1: dlen
         temp = temp + abs( Decphase_array(m) );   
      end    

      if(temp > maxsum)
         index = n;
         maxsum = temp;
      end   
   end   
   dectaps = CDecimate(Convtaps(1: endarray), decimation_ratio, index);
return;

function [sq_err, sqrt_err] = mse(array1, array2)
   dlen1 = length(array1);
   dlen2 = length(array2);
   
   error = 0;
   if( dlen1 == dlen2)
      for n = 1: dlen1    
         error = error + (array1(n) - array2(n))^2; 
      end 
      sq_err = error / dlen1;
      sqrt_err = sqrt(error) / dlen1;
   else
      sq_err = -999;
      sqrt_err = -999;
   end
return; 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CDecimate
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function CDecData = CDecimate(CData, decimation_ratio, decimation_phase) 
   
   %output data array size
   filtlength = length(real(CData));
   declength = fix(length(real(CData)) / decimation_ratio) - 1;
   
   % matrix decimation method?
   CDecData(1: declength) = CData( decimation_phase: decimation_ratio: ((declength-1) * decimation_ratio) + decimation_phase);
   clear CData;
return;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% rescale real array to max value
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function scaled_array = scalereal( input_array, newmaxvalue)
   maxI = max( input_array);
   minI = -1 * min( input_array);
   
   arraystat(1: 2) = [maxI, minI];
   
   scalef = newmaxvalue / max(arraystat);
   scaled_array = scalef * input_array;
return;

function sel_tap_array = LargestAbsSumTaps( Taparray, num_extractedtaps)
  % loop through the taps, looking for the largest abs sum
   testloops = length(Taparray) - num_extractedtaps;
   test_sum(1: testloops) = 0;
   testindex = 1;
   testmax = 0;
   for n = 1: testloops
      for m = 1: num_extractedtaps
         test_sum(n) = test_sum(n) + sqrt( (Taparray( n + m) )^2);       
      end
  
      if( test_sum(n) > testmax)
         testindex = n;
         testmax = test_sum(n);
      end   
   end
  
   % provide the subset indices
   start_index = testindex;
   stop_index = testindex + num_extractedtaps - 1;
   sel_tap_array(1: num_extractedtaps) =  Taparray(start_index: stop_index);
return;

function compfilter = ciccomp3or5taps(CICclockrate,bandwidth,CICstage,m,CICrate,numtaps,resolution,plotripple,savetaps,filepath, PFIRtaps, m2stage_array);

    % CICclockrate = 160e6;    %Rate of PFIR clock, i.e. after interpolation or before decimation = CIC I/O rate
    % bandwidth = 10e6;           %Passband bandwidth
    % CICstage = 5;               %5 for DDC, 6 for DUC
    % m=1;                        %CIC delay
    % CICrate = 6;                %Interpolation/Decimation rate
    % numtaps = 5;                %number of compensation taps 3 or 5 taps
    % resolution = 200;           %resolution of search matrix
    % plotripple = 1;             %1 = plot ripple across bandwidth
    % savetaps = 1;               %1 = save text file of taps
    % filepath='C:\Documents and Settings\A0188431\My Documents\My GC Studio Projects\GC5016\TSW4100\DDC Experiments\GC5016_RX_CIC6_PFIR2_4CH\GC5016_RX_CIC6_PFIR2_4CH_10MHz\CICcomp.txt';
    % PFIRtaps = []               % array of PFIR taps
    
    PFIRclockrate = CICclockrate/CICrate;
    
    cictaps=GenCICTaps_indivm2(CICstage,CICrate,m, m2stage_array);
    numtaps_int = max(size(cictaps));
    % numtaps_int = CICrate*5 - 1; 
    
    firlen = max(size(PFIRtaps));
    InterpPFIRtaps= zeros(1, firlen * CICrate);
    InterpPFIRtaps( 1 : CICrate : firlen * CICrate)= PFIRtaps;
    Convtaps = conv(cictaps, InterpPFIRtaps); 
    numconvtaps = max(size(Convtaps));

    BW = bandwidth/(2*PFIRclockrate*CICrate);
    numfft=round(25/BW);
    BWpoints=round(BW*numfft)+1;

    % 3tap correction only uses x
    % 5tap correction uses x,y
    % 7 tap correction uses x,y,z

    if(numtaps == 3)
       xmin= .004;
       xmax= .4;
       xresolution=resolution;
        
       ymin=0;
       ymax=0;
       yresolution=1;  
        
       zmin=0;
       zmax=0;
       zresolution=1;  
        
    elseif(numtaps == 5)
       xmin= .004;
       xmax= .4;
       xresolution=resolution;
        
       ymin= .002;
       ymax= .2;
       yresolution=resolution;
        
       zmin=0;
       zmax=0;
       zresolution=1;  
        
    elseif(numtaps == 7)    
       xmin= .004;
       xmax= .4;
       xresolution=resolution;

       ymin= .002;
       ymax= .2;
       yresolution=resolution;

       zmin=.001;
       zmax=.1;
       zresolution=resolution;
       
    else
        disp(sprintf('numtaps should be 3, 5, or 7'));
        xresolution = 1;
        yresolution = 1;
        zresolution = 1;
        
    end
    BWminmax=zeros(xresolution,yresolution,zresolution);

    for xindex = 1: xresolution
        for yindex = 1: yresolution
           for zindex = 1: zresolution
              xtap = -(xmin+(xmax-xmin)*xindex/xresolution);
              ytap = (ymin+(ymax-ymin)*yindex/yresolution);
              ztap = (zmin+(zmax-zmin)*zindex/zresolution);
              
              comparray=[ztap ytap xtap 1 xtap ytap ztap];
              % comparray_int=zeros(numtaps_int,1);
              comparray_int=zeros(CICrate*max(size(comparray)),1);
              % comparray_int(1:CICrate:numtaps_int)=comparray;
              comparray_int(1:CICrate:max(size(comparray_int)))=comparray;
               % convolved=conv(comparray_int,cictaps);
              convolved=conv(comparray_int,Convtaps);
 
              convolved_fft=fftextendlocal(convolved,numfft);
              BWarray=20*log10(abs(convolved_fft(1:BWpoints)));
              BWminmax(xindex,yindex, zindex)=max(BWarray)-min(BWarray);
           end
        end   
    end

%    [minxindex,minyindex]=find(BWminmax==min(min(BWminmax)));
    [minxindex,minyindex,minzindex]=find(BWminmax==min(min(min(BWminmax))));
    xtap = -(xmin+(xmax-xmin)*minxindex/xresolution);
    ytap = (ymin+(ymax-ymin)*minyindex/yresolution);
    ztap = (zmin+(zmax-zmin)*minzindex/zresolution);

    clear comparray;
    if(numtaps == 7)
        comparray=[ztap ytap xtap 1 xtap ytap ztap];
    elseif(numtaps == 5)
        comparray=[ytap xtap 1 xtap ytap];
    elseif(numtaps == 3)
        comparray=[xtap 1 xtap];
    else
        comparray = [1];
    end
    
    % comparray_int=zeros(numtaps_int,1);
    % comparray_int(1:CICrate:numtaps_int)=comparray;
    comparray_int=zeros(CICrate*max(size(comparray)),1);
    comparray_int(1:CICrate:max(size(comparray_int)))=comparray;
    
    if plotripple == 1
        % convolved=conv(comparray_int,cictaps);
        convolved=conv(comparray_int,Convtaps);
        convolved_fft=fftextendlocal(convolved,numfft);
        BWarray=20*log10(abs(convolved_fft(1:BWpoints)));
        x=(0:BWpoints-1)*bandwidth/((BWpoints-1)*2);
        figure(1);plot(x,BWarray-mean(BWarray));
        grid on;
        xlabel('Frequency (Hz)');
        ylabel('Ripple (dB)')
    end

    minxindex
    minyindex
    minzindex
    ripple=BWminmax(minxindex,minyindex,minzindex)
    comparray = comparray_int(1:CICrate:length(comparray_int));
    
 %   if numtaps == 3
 %       comparray(5)=[];
 %       comparray(1)=[];
 %   end
    
    comparray
    
    if savetaps == 1
        filepath
        write_column(comparray, filepath);
    end

    compfilter = comparray;

return;

 function outfft=fftextendlocal(inarray,numfft);
     inarray_extend=zeros(numfft,1);
     inarray_extend(1:length(inarray))=inarray;
     outfft=fft(inarray_extend);
 return;
 
 function FFTC64KWinNoFig( d, samplerate, titletext)

windowflag = 1;

lendata = length(d);               % size of data
sizeFFT = 2^( ( round(log2(lendata))));       % set FFT size to next larger power of 2

if( sizeFFT < (2^16) )                     % at least 64K FFT 
   sizeFFT = (2^16);
end

if( lendata < sizeFFT)
   temp = lendata;
else
   temp = sizeFFT;
end   
b(1: sizeFFT) = 0 + i*0;
b(1: temp) = d(1: temp);  % put in input data

if( windowflag == 1)
   numbins = temp;                     % apply window 
   w(1: sizeFFT) = 0;
   w(1: numbins) = hamming(numbins); 
   
   bi(1: sizeFFT) = real(b(1: sizeFFT) );
   bq(1: sizeFFT) = imag(b(1: sizeFFT) );
   bi(1: sizeFFT) = bi(1: sizeFFT) .* w(1: sizeFFT);
   bq(1: sizeFFT) = bq(1: sizeFFT) .* w(1: sizeFFT);
   b(1: sizeFFT) = bi(1: sizeFFT) + (i*bq(1: sizeFFT));
end

xmin = -.5 * samplerate;                   % plotted at the input rate, shifted for complex representation
xmax = .5 * samplerate;
X= linspace(xmin, xmax , sizeFFT);               % Create the X scale frequency axis

% b= finalTaps/sum(finalTaps);             % fraction of sum(? needed)
% F = fft(b, M);

F(1: sizeFFT) = 0 + i*0;
F = fft(b, sizeFFT);                    % FFT of convolved filter response
F= F + (realmin + realmin*i);             % This prevents log(0) warnings

FF= 20*log10(abs(F));                     % Compute log of magnitude

maxFF = max(FF);
% maxFF = 20*log10(32767*sqrt(sizeFFT));
FF = FF - maxFF;                          % dBc 

outFFT(1:sizeFFT) = 0 + 0*i;
hsizeFFT = fix(sizeFFT/2);

outFFT(1: hsizeFFT ) = FF( (hsizeFFT+1): sizeFFT);         % lower half of FFT plot is neg freq
outFFT( (hsizeFFT+1) : sizeFFT) = FF(1: hsizeFFT);          % upper half of FFT plot is pos freq

% figure(1);

plot( X, outFFT);

title( titletext);
axis( [xmin xmax (max(FF) - 119) (max(FF) + 1) ]); 
grid;            % Plot the data

grid on;
zoom on;

 function [taps, DataLen] = read_filterdat( filename)
% filename = 'C:\Engineering\Fs48xFpp4xFsp5x_1485.taps';

   fid= fopen(filename,'r');

   taps = fscanf(fid, '%g');
   DataLen = length(taps);
   fclose(fid);
return;
   