/******************************************************************************
All values transmitted to and received from the Compass Module are most significant bit first, with the bit
value valid after the clock signal's rising edge. For the BASIC Stamp 2, this means set the SHIFTOUT
commands Mode argument to MSBFIRST, and the SHIFTIN commands Mode argument to MSBPOST.

To reset the HM55B, take /EN from high to low, and shift-out %0000, then set /EN high again.

After reset, start a measurement by taking /EN low again, then shift-out %1000. Leave /EN low until
checking the measurement status.

To check the measurement status, start by sending a positive pulse to /EN. Then, shift-out %1100, and
shift-in the status flags. While the measurement is in progress, the end flag and error flag will both be
00. The compass Module may be polled for status repeatedly until the measurement is complete, at
which point the end flag will change to 11. Upon receipt of %1100, discontinue polling. Leave /EN low,
and move on to shifting-in the x and y-axis values.

Shifting-in the x and y-axis values is a simple matter of shifting-in 11 bits for the x-axis measurement
followed by 11 more bits for the y-axis measurement. After completing the y-axis shift-in, set the /EN pin
high again.

S. K. 5/2010
******************************************************************************/
//INCLUDES  These are files that define some of the labels we use below. 
//These are made by Ti and pretty much follow the labels in the CC2430 spec.
 
#include <hal_types.h>
#include <hal_defs.h>
#include <hal_cc8051.h>
#include <ioCC2430.h>

/******************************************************************************/
//CONSTANTS

#define IO_GIO = 0    /* General IO */
#define IO_PER  1     /* Perepheral IO */   
#define IO_IN   0     /* Pin Direction - Input */
#define IO_OUT  1     /* Pin Direction - Output */ 




/*********structure to store axis information**********/

struct axis{
      uint16 x;
      uint16 y;
      };        


/*************************************************/
/********** Compass functions*/



struct axis getMeasurement(void);
void resetDevice(void);
void startMeasurement(void);
uint8 checkStatus(void);
void Shiftout(uint8 num, uint8 num_bits); 
uint16 Shiftin( uint8 num_bits );
void setIOPins(void);
void clearBits( void );




/**********************************************/ 


/***********************************************************************************
* LOCAL VARIABLES*/

// We will put variable we make up here: example might be unsigned char door_range = 0;
// unsigned char just tells the compiler it is a 8 bit number (0 -255)

/***********************************************************************************
* LOCAL FUNCTIONS*/

static void Wait_uS(uint16 usec);  //Functions (aka subroutines) are called out before main

static void Wait_mS(uint16 msec);  // static says they do not change often so they can be in flash memory
                                   // void says nothing is sent back to the calling program
                                   // Wait_uS is a name I made up
                                   // (uint16 msec) msec is a name I made up of type unsigned interger 16 bits long
/*******************************************************************************
* Global Variables
*/
/**************************************************************************************************
void MSA_PowerMgr(uint8 enable)
{
  // enable OSAL power management 
  if (enable)
   osal_pwrmgr_device(PWRMGR_BATTERY);
  else
   osal_pwrmgr_device(PWRMGR_ALWAYS_ON);
}


**************************************************************************************************
@fn: getMeasurement() 

@brief: gettMeasurment() Function is called to get measurment from the compass. It 
also clears any prvious Bits on Port 0 Pin 6 (Pins are numbered 0 -7), resets the device,
Start a new measurement, checks the status of the measurement in progress, it measurements were 
sucessfully completed it Shifts in (inputs) the measurement data from the compass

@param: void 

@return: struct axis - Record of field intensities of the axis ( x and y )
**************************************************************************************************/


/***********************************************************************************
* c programs always start at "main" the void says nothing is carrier in, int says a interger (0-255)
can be returned. Some programs end with return(0). Say another program calls this one. When it see the 0
it knows the program worked. Our program loops forever doing it thing so we will not use this feature.
***********************************************************************************/
int main(void)

// This section wakes up the MCU and turns on the crystal oscilator and then turn off the RC oscillator inside the 2430
{
        asm("NOP");                 // Put here as a place to put a break point at the start of the code, it does nothing
                                    // It is a assembly instruction (non opperation)that does nothing
        
        SLEEP &= ~0x04;             // Set SLEEP (page 67) Bit 2 to zero turning on both the RC and xtal osc
                                    // The 8 bit register SLEEP is AND with Not 0x04
                                    // 0x04 is 00000100 so not (~) 0x04 is 11111011, when anded with something all bit stay as 
                                    // they are except bit 2 (remember we start counting at 0). Anything AND 0 is 0.
        
        while( !(SLEEP & BIT3));    // Loop until SLLEP Bit 3 (page 67) goes low saying xtal osc is stable
        
        Wait_uS(1000);              // go down to the function and loop a 1000 times becasue Ti screwed up and the xtal osc
                                    // isn't stable when the bit says it is. (see page 67 Bit 6 note)
        
        CLKCON = (CLKCON & ~0x40);  // CLKCON (page 70) AND not Bit 6 (01000000) which is 10111111 to select xtal osc

        SLEEP |= 0x04;              // Sleep (page 67) OR with Bit 2 (00000100) which makes bit 2 a 1 turning off the osc not selected
                                    // in the step above Note 0 OR 0 is 0, 0 OR 1 is 1, 1 OR 1 is 1 so all bits in sleep stay the same 
                                    // except bit 2 is now a 1
/**************************************************************************************************/
        
//PORT 0        
    // Initialize Port 0
P0SEL &= ~(BIT7 | BIT6 | BIT5 | BIT4);  // Set bits 7:4 to GPIO
P0DIR |=  (BIT7 | BIT6 | BIT5);         // Set bits 7:5 to output
P0DIR &= ~BIT4;                         // Ste bit 4 as input for Compass output
    
//PORT 1
    
    // Initialize P1-0 LED1, P1-1 LED2
    P1SEL &= ~ BIT0;  // P0_0 Set to GPIO
    P1DIR |=  BIT0;   // P0_0 Set to output  
    
    P1_0 = 0; // Turn on SOC-BB LED Red, 0 is on

/**************************************************************************************************
@fn: getMeasurement() 

@brief: gettMeasurment() Function is called to get measurment from the compass. It 
also clears any prvious Bits on Port 0 Pin 6 (Pins are numbered 0 -7), resets the device,
Start a new measurement, checks the status of the measurement in progress, it measurements were 
sucessfully completed it Shifts in (inputs) the measurement data from the compass

@param: void 

@return: struct axis - Record of field intensities of the axis ( x and y )
**************************************************************************************************/

//struct axis getMeasurement()
  { 
    uint16 measure; 
    struct axis a; 
     
     setIOPins();
     clearBits();
     resetDevice(); 
     startMeasurement();
     while( checkStatus() );  
       
     measure=Shiftin(11);
     a.x = measure;    
  
     measure=Shiftin(11);
     a.y=measure;
     P0_5= 1;                /*set enable (Port 0 pin 5) to High after measure*/
     //return a; 
  }   
}
/*---------------------------------  END of Program   -----------------------------------------------------*/

/**************************************************************************************************
@fn: resetDevice() 

@brief: Resets the device 

@param: void 

@return: void
**************************************************************************************************/

void resetDevice()
 { 
   
   P0_5 = 1;               /*Set enable line High*/  
   
   asm("NOP");            /* Waits - delay of 1 instruction which does nothing */
   
   P0_5 = 0;             /* Set enable line to low */   
   
   Shiftout(0, 4);      /* ShiftOut 0 0 0 0 to reset device */
   
   
   
   P0_5 = 1;          /* Set enable line high again after Shiftout*/      
   
 }



/**************************************************************************************************
@fn: startMeasurement() 

@brief: startMeasurment() Function is used to invoke the compass to start taking a measurement. 
This is accomplished by setting the Enable line (P0_5 in the CC2431's port 0 pin 5) to  HIGH (binary 1)
After that the clock is set to high and 1 bit of data is pumped outwards with each rising edge of the clock
Check HM55B compass module documentation for further details

@param: void 

@return: void
**************************************************************************************************/


void startMeasurement(void)
  {
	
	         
        P0_5 = 0;           /* Take enable to low */ 
        
        Shiftout(8 , 4);   /* 8 in bainary is equal to 1 0 0 0  */
	                  /* tells the compass to start measuring */
  } 
     

/**************************************************************************************************
@fn: checkStatus() 

@brief: Checks the status of a measuremnt in progress  
@param: void 

@return: uint8 - 0 if sucess 1 if failure 
**************************************************************************************************/

uint8 checkStatus(void)
    {

       
       P0_5 = 1;      /* Start by sending a positive pulse to Enable line */ 
       asm("NOP");    /* wait for a period of 1 instruction doing nothing*/ 
       P0_5 = 0;      /* Set Enable to low again ( complete the pulse ) */
       
       Shiftout(12, 4);  /* 1 1 0 0 is shifted to the compass to check measurement status */
       
      /*Start Polling - Sucessful measurement when 12 ( 1 1 0 0 ) is read */
       
       if ( Shiftin(4) != 12 )   /* 12 = 1 1 0 0 binary   */     
          return 1;             /* If no sucess return 1 */ 
       
       else 
          return 0;
     
     } 
     
     
/**************************************************************************************************
@fn: Shiftout() 

@brief: Shifts out data bits (MSBFirst) to the compass Data line (P0_6)  
@param: uint8 num - the value to shiftout, uint8 num_bits - number of bits to shift 

@return: void
**************************************************************************************************/
     
     
void Shiftout(uint8 num, uint8 num_bits) 
{


P0DIR |= ( 1 << 5 );   /* set Port 0 pin 6 (Data) as output - (Output - 1 Input - 0) */


/* General algorithm for stripping bits Most significant bit first */ 
for(int i = num_bits -1; i >= 0; i -- )
   {                                    
     
     if ( (num & (1 << i))  >  1   )  
	  {  
	    P0_6 = 1;      /*Send High = 1 to data pin P0_6  */
            
            P0_7 = 1;    /*Set the clock high */            
	    asm("NOP"); /* Let the Clock Settle */  
            P0_7 = 0;   /*Set clock low - End of pumping a "bit" */ 
	   
          }	
	
	else 
           { 	
	    P0_6 = 0;      /* Send Low = 0 to data pin */
            
            P0_7 = 1;    /*Set the clock high */
	    asm("NOP"); /* Let the Clock Settle */  
	    P0_7 = 0;  /*Set clock low - End of pumping a "bit" */ 

	   } 		

   


    }    
 

}

/**************************************************************************************************
@fn: Shiftin() 

@brief: Shifts in data bits  from the compass Data line (P0_6)  
@param: uint8 num_bits - number of bits to shift in  

@return: uint16 - The bits shifted in 
**************************************************************************************************/

uint16 Shiftin(uint8 num_bits ) 
  {
    
     
     P0DIR &= ~( 1 << 5 );      /* Set Port 0 pin 6 for input
                               / Bits in this register indicate which Pins in                                   
                              / Port 0 are set to input or output. It is a 8 bit 
                             / rprsenting Pins 0 -7 on Port 0. 
                            / 0 - input 1 - output */ 
     
     uint16 result=0, temp=0; 
     uint8 i=0;
     
     P0_7 = 1;         /* set clock high */    
     
     asm("NOP");     
     
     temp =  P0_6;     /* Grab the first bit */ 
     
     P0_7 = 0;        /* set clock low */     
     
     result = temp;  /* set result to the first bit (MSBFirst) */
     
     
     
     for(i= 1; i< num_bits; i++)     /* Grab rest of the bits Shiftd in */ 
       { 
       P0_7 = 1;                     /* Set clock high */   
       asm("NOP");                  /* Shiftin data with the rising and falling of clock */  
       temp = P0_6;                 /* Grab bits */
       result = (result << 1) + temp; /* Shift gathered bits to the left */       
       P0_7 = 0;              /* Set clock low */
       }    
    
   return result;  /* Return the result of Shitin */
   
  
}

/**************************************************************************************************
@fn: setIOPins()

@brief: Sets Pins 5, 6 and 7 (Pins start from 0 -7 ) to be General IO, insted of perepheral IO 


@param: void 

@return: void
**************************************************************************************************/


void setIOPins(void)         /* Set Pins to General IO */
{                           
                            /* Set the POSEL register's sixth, seventh and eighth 
                           /   bits to  0 - General IO. 
                          /    1 is for perpheral IO but we only use General IO  */
                                  

P0SEL &= ~(0xE0);       
                            /*       x  x  x  x  x  x  x  x      
                                   & 0  0  0  1  1  1  1  1
                                     ----------------------
                                     0  0  0  x  x  x  x  x      */
}



/**************************************************************************************************
@fn: setIOPins()
@brief: Sets pins to output ( 1 ) and clears pins 5, 6 and 7 - Sets to Zero  
@param: void 
@return: void
**************************************************************************************************/

void clearBits(void) 
{
  

P0DIR |= (0xE0) ;     /* set PODIR = 1 1 1 X X X X X  */         
  
  P0_7 = 0;      
  Wait_mS(1);  
  P0_6 = 0;
  P0_5 = 0;
  
  asm("NOP");
  asm("NOP");
  asm("NOP");
} 



void  Wait_uS(uint16 usec)  // The function noted (correct word is prototyped) before main and used to wait for the xtal to stabize 
{
    while(usec--)          //  -- means decrement. Above we entered 1000 so the loop will go around 1000 times
                           // each time subtracting 1 until it reachs 0 where it will not be "True" and the program will
                           // return to the next line of code below where it was called
   {
        asm("nop");       // 10 non ops in a row to waste time so each loop about is close to 1 micro second
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
        asm("nop");
    }
}


void Wait_mS(uint16 msec) // The function prototyped before main and used to wait before changing the LEDs from on to off to on 
{
    while(msec--)         // Loop untill msec from its starting value given above is decremented to 0
    Wait_uS(1000);        // In each loop go to WaituS and loop a 1000 times.  Since each loop is a usec we get 1000usec or 1 msec
                          // So if msec is 250 it will go to the usec function above where it will loop 1000 times for 
                          // a total of 250 x 1000 = 250,000 times each loop being about 1us so 250ms or 1/4 of a second
}