/** <!-------------------------------------------------------------------------------------------->
 * Transfers one (or many) dwords from source to destination adress via EDMA. Note that
 * many transfers (iNumber > 1) are still done from the same source address to provide
 * transfer from a fifo.
 * This function will block until all data is transferred.
 *
 * @param   u32SourceAddress	(i) source address
 * @param   u32Destination		(i) destination address. Has to be aligned to
 *                                  EDMA3_CACHE_LINE_SIZE_IN_BYTES
 * @param   iNumber				(i) number of transfers in dwords
 * @return
 <!--------------------------------------------------------------------------------------------> */
Void edmaTransfer( UInt32* pu32SourceAddress, UInt32* pu32Destination, SizeT iNumber )
{
    const UInt32 U32_ACNT = 4; // one float
    const UInt32 U32_BCNT = iNumber;
    const UInt32 U32_CCNT = 1;

    const UInt16 U16_DMA_WAIT_TIMEOUT = 0x8000;

	EDMA3_DRV_Result sEDMAResult=EDMA3_DRV_SOK;
    UInt32 u32WaitCounter=0;
    static UInt32 u32LastWaitCounter=0;

    UInt32 u32SrcAddr = (UInt32) pu32SourceAddress;
    UInt32 u32DstAddr = (UInt32) pu32Destination;

    // Take care of cached source and destination
    Cache_wbInv((Ptr) u32DstAddr, U32_ACNT * iNumber, Cache_Type_ALL, TRUE);
    Cache_wbInv((Ptr) u32SrcAddr, U32_ACNT * iNumber, Cache_Type_ALL, TRUE);

    if (sEDMAResult == EDMA3_DRV_SOK)
    {
        /* AB Sync Transfer Mode */
        sEDMAResult = EDMA3_DRV_setTransferParams (g_hEdma, g_sFPGAChannel.chId,
                                                   U32_ACNT, U32_BCNT, U32_CCNT, 0,
                                                   EDMA3_DRV_SYNC_AB);
    }

    if (sEDMAResult == EDMA3_DRV_SOK)
    {
        irqRaised1 = 0u;

        /* Setup the Source Address */
        sEDMAResult = EDMA3_DRV_setSrcParams (g_hEdma, g_sFPGAChannel.chId,
                                              u32SrcAddr,
                                              EDMA3_DRV_ADDR_MODE_INCR,
                                              EDMA3_DRV_W32BIT);

        /*
         * Now trigger the QDMA channel by writing to the Trigger
         * Word which is set as Destination Address.
         */
        if (sEDMAResult == EDMA3_DRV_SOK)
        {
            sEDMAResult = EDMA3_DRV_setPaRAMEntry (g_hEdma, g_sFPGAChannel.chId,
                                                   EDMA3_DRV_PARAM_ENTRY_DST,
                                                   u32DstAddr);
        }

        if (sEDMAResult == EDMA3_DRV_SOK)
        {
            // wait for transfer to finish
            while( irqRaised1==0 && u32WaitCounter <= U16_DMA_WAIT_TIMEOUT )
            {
                ++u32WaitCounter;
            };

            // Ensure that destination data will be cache coherent
            Cache_inv((Ptr) u32DstAddr, U32_ACNT * iNumber, Cache_Type_ALL, TRUE);

            /* Check the status of the completed transfer */
            if (irqRaised1 < 0)
            {
                /* Some error occured. */
                CPTErrorHandler::Instance().Raise( CPTErrorHandler::DEC_DMA_EVENT_MISSED, irqRaised1,
                                                   "(Q)DMA Event missed!", irqRaised1 );

                /* Clear the error bits first */
                sEDMAResult = EDMA3_DRV_clearErrorBits (g_hEdma, g_sFPGAChannel.chId);
            }

            if( u32WaitCounter > u32LastWaitCounter  )
            {
                Log_print2( Diags_USER1, "Waiting for DMA took %d counts. Last: %d.",
                            u32WaitCounter,
                            u32LastWaitCounter);
                u32LastWaitCounter = u32WaitCounter;
            }
        }
    }

} // END edmaTransfer
