/*
 * main.c
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2009
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <strings.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>

#include <xdc/std.h>

#include <ti/sdo/ce/CERuntime.h>

#include <ti/sdo/dmai/Dmai.h>
#include <ti/sdo/dmai/Fifo.h>
#include <ti/sdo/dmai/Pause.h>
#include <ti/sdo/dmai/BufferGfx.h>
#include <ti/sdo/dmai/Rendezvous.h>

#include "display.h"
#include "video.h"
#include "speech.h"
#include "loader.h"
#include "../ctrl.h"
#include "../demo.h"
#include "../ui.h"

#include <ti/sdo/fc/rman/rman.h>

/* The levels of initialization */
#define LOGSINITIALIZED         0x1
#define DISPLAYTHREADCREATED    0x2
#define VIDEOTHREADCREATED      0x4
#define SPEECHTHREADCREATED     0x8
#define LOADERTHREADCREATED     0x10

/* Thread priorities */
#define LOADER_THREAD_PRIORITY  sched_get_priority_max(SCHED_FIFO) - 2
#define SPEECH_THREAD_PRIORITY  sched_get_priority_max(SCHED_FIFO) - 2
#define VIDEO_THREAD_PRIORITY   sched_get_priority_max(SCHED_FIFO) - 1
#define DISPLAY_THREAD_PRIORITY sched_get_priority_max(SCHED_FIFO) 

typedef struct Args {
    Display_Output displayOutput;
    VideoStd_Type  videoStd;
    Char          *videoStdString;
    Char          *speechFile;
    Char          *videoFile;
    Codec         *speechDecoder;
    Codec         *videoDecoder;
    Int            loop;
    Int            keyboard;
    Int            time;
    Int            osd;
    Int            interface;
} Args;

#define DEFAULT_ARGS { Display_Output_COUNT, VideoStd_D1_NTSC, "D1 NTSC", NULL, NULL, NULL, NULL, \
                       FALSE, FALSE, FOREVER, FALSE, FALSE }

/* Global variable declarations for this application */
GlobalData gbl = GBL_DATA_INIT;

/******************************************************************************
 * Signal handler
 ******************************************************************************/
Void signalHandler(int sig)
{
    signal(SIGINT, SIG_DFL);
    gblSetQuit();
}

/******************************************************************************
 * getCodec
 ******************************************************************************/
static Codec *getCodec(Char *extension, Codec *codecs)
{
    Codec *codec = NULL;
    Int i, j;

    i = 0;
    while (codecs[i].codecName) {
        j = 0;
        while (codecs[i].fileExtensions[j]) {
            if (strcmp(extension, codecs[i].fileExtensions[j]) == 0) {
                codec = &codecs[i];
            }
            j++;
        }
        i++;
    }

    return codec;
}

/******************************************************************************
 * usage
 ******************************************************************************/
static Void usage(void)
{
    fprintf(stderr, "Usage: decode [options]\n\n"
      "Options:\n"
      "-s | --speechfile       Speech file to play\n"
      "-v | --videofile        Video file to play\n"
      "-y | --display_standard Video standard to use for display (see below).\n"
      "-O | --display_output   Video output to use (see below).\n"
      "-k | --keyboard         Enable keyboard interface [off]\n"
      "-t | --time             Number of seconds to run the demo [infinite]\n"
      "-l | --loop             Loop to beginning of files when done [off]\n"
      "-o | --osd              Show demo data on an OSD [off]\n"
      "-i | --interface        Launch the demo interface when exiting [off]\n"
      "-h | --help             Print this message\n\n"
      "Video standards available:\n"
      "\t1\tD1 @ 30 fps (NTSC) [Default]\n"
      "\t2\tD1 @ 25 fps (PAL)\n"
      "\t3\t720P @ 60 fps\n"
      "Video outputs available:\n"
      "\tcomposite [Default]\n"
      "\tcomponent (Only 720P available)\n"
      "You must supply at least a video or a speech file\n"
      "with appropriate extensions for the file formats.\n\n");
}

/******************************************************************************
 * parseArgs
 ******************************************************************************/
static Void parseArgs(Int argc, Char *argv[], Args *argsp)
{
    const Char shortOptions[] = "s:v:y:O:kt:lfoih";
    const struct option longOptions[] = {
        {"speechfile",       required_argument, NULL, 's'},
        {"videofile",        required_argument, NULL, 'v'},
        {"display_standard", required_argument, NULL, 'y'},
        {"display_output",   required_argument, NULL, 'O'},
        {"keyboard",         no_argument,       NULL, 'k'},
        {"time",             required_argument, NULL, 't'},
        {"loop",             no_argument,       NULL, 'l'},
        {"osd",              no_argument,       NULL, 'o'},
        {"interface",        no_argument,       NULL, 'i'},
        {"help",             no_argument,       NULL, 'h'},
        {"exit",             no_argument,       NULL, 'e'},            
        {0, 0, 0, 0}
    };

    Int   index;
    Int   c;
    Char *extension;

    for (;;) {
        c = getopt_long(argc, argv, shortOptions, longOptions, &index);

        if (c == -1) {
            break;
        }

        switch (c) {
            case 0:
                break;

            case 's':
                extension = rindex(optarg, '.');
                if (extension == NULL) {
                    fprintf(stderr, "Speech file without extension: %s\n",
                            optarg);
                    exit(EXIT_FAILURE);
                }

                argsp->speechDecoder =
                    getCodec(extension, engine->speechDecoders);

                if (!argsp->speechDecoder) {
                    fprintf(stderr, "Unknown speech file extension: %s\n",
                            extension);
                    exit(EXIT_FAILURE);
                }
                argsp->speechFile = optarg;

                break;

            case 'v':
                extension = rindex(optarg, '.');
                if (extension == NULL) {
                    fprintf(stderr, "Video file without extension: %s\n",
                            optarg);
                    exit(EXIT_FAILURE);
                }

                argsp->videoDecoder =
                    getCodec(extension, engine->videoDecoders);

                if (!argsp->videoDecoder) {
                    fprintf(stderr, "Unknown video file extension: %s\n",
                            extension);
                    exit(EXIT_FAILURE);
                }
                argsp->videoFile = optarg;

                break;

            case 'y':
                switch (atoi(optarg)) {
                    case 1:
                        argsp->videoStd = VideoStd_D1_NTSC;
                        argsp->videoStdString = "D1 NTSC";
                        break;
                    case 2:
                        argsp->videoStd = VideoStd_D1_PAL;
                        argsp->videoStdString = "D1 PAL";
                        break;
                    case 3:
                        argsp->videoStd = VideoStd_720P_60;
                        argsp->videoStdString = "720P 60Hz";
                        break;
                    case 4:
                        argsp->videoStd = VideoStd_720P_50;
                        argsp->videoStdString = "720P 50Hz";
                        break;
                    case 5:
                        argsp->videoStd = VideoStd_1080I_30;
                        argsp->videoStdString = "1080I 30Hz";
                        break;
                    case 6:
                        argsp->videoStd = VideoStd_1080I_25;
                        argsp->videoStdString = "1080I 25Hz";
                        break;
                    default:
                        fprintf(stderr, "Unknown display resolution\n");
                        usage();
                        exit(EXIT_FAILURE);
                }
                break;

            case 'O':
                if (strcmp(optarg, "component") == 0) {
                    argsp->displayOutput = Display_Output_COMPONENT;
                } else if (strcmp(optarg, "composite") == 0) {
                    argsp->displayOutput = Display_Output_COMPOSITE;
                } else {
                    fprintf(stderr, "Unknown video output: %s\n", optarg);
                    usage();
                    exit(EXIT_FAILURE);
                }
                break;

            case 'k':
                argsp->keyboard = TRUE;
                break;

            case 't':
                argsp->time = atoi(optarg);
                break;

            case 'l':
                argsp->loop = TRUE;
                break;

            case 'o':
                argsp->osd = TRUE;
                break;

            case 'i':
                argsp->interface = TRUE;
                break;

            case 'h':
                usage();
                exit(EXIT_SUCCESS);

            default:
                usage();
                exit(EXIT_FAILURE);
        }
    }

    if (argsp->displayOutput == Display_Output_COUNT) {
        if ((argsp->videoStd == VideoStd_D1_NTSC) || (argsp->videoStd == VideoStd_D1_PAL)) {
            argsp->displayOutput = Display_Output_COMPOSITE;
        } else {
            argsp->displayOutput = Display_Output_COMPONENT;
        }
    }
    /* Need at least one file to decode and only one sound file */
    if (!argsp->videoFile && !argsp->speechFile) {
        usage();
        exit(EXIT_FAILURE);
    }
}

/******************************************************************************
 * uiSetup
 ******************************************************************************/
static UI_Handle uiSetup(Args *argsp)
{
    UI_Attrs  uiAttrs;
    UI_Handle hUI;

    /* Create the user interface */
    uiAttrs.osd = argsp->osd;
    uiAttrs.videoStd = argsp->videoStd;

    hUI = UI_create(&uiAttrs);

    if (hUI == NULL) {
        ERR("Failed to create UI\n");
        return NULL;
    }

    /* Initialize values */
    UI_updateValue(hUI, UI_Value_DemoName, "Decode");
    UI_updateValue(hUI, UI_Value_DisplayType, argsp->videoStdString);

    if (argsp->videoDecoder) {
        UI_updateValue(hUI, UI_Value_VideoCodec, argsp->videoDecoder->uiString);
    }
    else {
        UI_updateValue(hUI, UI_Value_VideoCodec, "N/A");
    }

    UI_updateValue(hUI, UI_Value_ImageResolution, "N/A");

    if (argsp->speechDecoder) {
        UI_updateValue(hUI, UI_Value_SoundCodec,
                       argsp->speechDecoder->uiString);
    }
    else {
        UI_updateValue(hUI, UI_Value_SoundCodec, "N/A");
    }

    UI_updateValue(hUI, UI_Value_SoundFrequency, "N/A");

    /* Initialize the user interface */
    UI_init(hUI);

    return hUI;
}

/******************************************************************************
 * main
 ******************************************************************************/
Int main(Int argc, Char *argv[])
{
    Args                    args                = DEFAULT_ARGS;
    Uns                     initMask            = 0;
    Int                     status              = EXIT_SUCCESS;
    Pause_Attrs             pAttrs              = Pause_Attrs_DEFAULT;
    Rendezvous_Attrs        rzvAttrs            = Rendezvous_Attrs_DEFAULT;
    Fifo_Attrs              fAttrs              = Fifo_Attrs_DEFAULT;
    Rendezvous_Handle       hRendezvousInit     = NULL;
    Rendezvous_Handle       hRendezvousCleanup  = NULL;
    Rendezvous_Handle       hRendezvousLoop     = NULL;
    Rendezvous_Handle       hRendezvousLoader   = NULL;
    Pause_Handle            hPauseProcess       = NULL;
    Pause_Handle            hPausePrime         = NULL;
    UI_Handle               hUI                 = NULL;
    Int                     syncCnt             = 0;
    struct sched_param      schedParam;
    pthread_attr_t          attr;
    pthread_t               displayThread;
    pthread_t               videoThread;
    pthread_t               speechThread;
    pthread_t               loaderThread;
    LoaderEnv               loaderEnv;
    DisplayEnv              displayEnv;
    VideoEnv                videoEnv;
    SpeechEnv               speechEnv;
    CtrlEnv                 ctrlEnv;
    Int                     numThreads;
    Void                   *ret;

    /* Zero out the thread environments */
    Dmai_clear(loaderEnv);
    Dmai_clear(displayEnv);
    Dmai_clear(videoEnv);
    Dmai_clear(speechEnv);
    Dmai_clear(ctrlEnv);

    /* Parse the arguments given to the app and set the app environment */
    parseArgs(argc, argv, &args);

    printf("Decode demo started.\n");

    /* Initialize the mutex which protects the global data */
    pthread_mutex_init(&gbl.mutex, NULL);

    /* Set the priority of this whole process to max (requires root) */
    setpriority(PRIO_PROCESS, 0, -20);

    /* Initialize Codec Engine runtime */
    CERuntime_init();

    /* Initialize signal handler for SIGINT */
    signal(SIGINT, signalHandler);
    
    /* Initialize Davinci Multimedia Application Interface */
    Dmai_init();

    initMask |= LOGSINITIALIZED;

    /* Set up the user interface */
    hUI = uiSetup(&args);

    if (hUI == NULL) {
        cleanup(EXIT_FAILURE);
    }

    /* Create the Pause objects */
    hPauseProcess = Pause_create(&pAttrs);
    hPausePrime = Pause_create(&pAttrs);

    if (hPauseProcess == NULL || hPausePrime == NULL) {
        ERR("Failed to create Pause objects\n");
        cleanup(EXIT_FAILURE);
    }

    /* Determine the number of threads needing synchronization */
    numThreads = 1;

    if (args.videoFile) {
        numThreads += 3;
        syncCnt++;
    }

    if (args.speechFile) {
        numThreads += 1;
        syncCnt++;
    }

    /* Create the objects which synchronizes the thread init and cleanup */
    hRendezvousInit = Rendezvous_create(numThreads, &rzvAttrs);
    hRendezvousCleanup = Rendezvous_create(numThreads, &rzvAttrs);
    hRendezvousLoop = Rendezvous_create(syncCnt, &rzvAttrs);
    hRendezvousLoader = Rendezvous_create(2, &rzvAttrs);

    if (hRendezvousInit == NULL || hRendezvousCleanup == NULL ||
        hRendezvousLoop == NULL || hRendezvousLoader == NULL) {

        ERR("Failed to create Rendezvous objects\n");
        cleanup(EXIT_FAILURE);
    }

    /* Initialize the thread attributes */
    if (pthread_attr_init(&attr)) {
        ERR("Failed to initialize thread attrs\n");
        cleanup(EXIT_FAILURE);
    }

    /* Force the thread to use custom scheduling attributes */
    if (pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED)) {
        ERR("Failed to set schedule inheritance attribute\n");
        cleanup(EXIT_FAILURE);
    }

    /* Set the thread to be fifo real time scheduled */
    if (pthread_attr_setschedpolicy(&attr, SCHED_FIFO)) {
        ERR("Failed to set FIFO scheduling policy\n");
        cleanup(EXIT_FAILURE);
    }

    /* Create the video threads if a file name is supplied */
    if (args.videoFile) {
        /* Create the display fifos */
        displayEnv.hInFifo = Fifo_create(&fAttrs);
        displayEnv.hOutFifo = Fifo_create(&fAttrs);

        if (displayEnv.hInFifo == NULL || displayEnv.hOutFifo == NULL) {
            ERR("Failed to create display fifos\n");
            cleanup(EXIT_FAILURE);
        }

        /* Set the display thread priority */
        schedParam.sched_priority = DISPLAY_THREAD_PRIORITY;
        if (pthread_attr_setschedparam(&attr, &schedParam)) {
            ERR("Failed to set scheduler parameters\n");
            cleanup(EXIT_FAILURE);
        }

        /* Create the display thread */
        displayEnv.displayOutput      = args.displayOutput;        
        displayEnv.videoStd           = args.videoStd;
        displayEnv.hRendezvousInit    = hRendezvousInit;
        displayEnv.hRendezvousCleanup = hRendezvousCleanup;
        displayEnv.hPauseProcess      = hPauseProcess;
        displayEnv.hPausePrime        = hPausePrime;
        displayEnv.osd                = args.osd;

        if (pthread_create(&displayThread, &attr, displayThrFxn, &displayEnv)) {
            ERR("Failed to create display thread\n");
            cleanup(EXIT_FAILURE);
        }

        initMask |= DISPLAYTHREADCREATED;

        /* Set the video thread priority */
        schedParam.sched_priority = VIDEO_THREAD_PRIORITY;
        if (pthread_attr_setschedparam(&attr, &schedParam)) {
            ERR("Failed to set scheduler parameters\n");
            cleanup(EXIT_FAILURE);
        }

        /* Create the video thread */
        videoEnv.hRendezvousInit    = hRendezvousInit;
        videoEnv.hRendezvousCleanup = hRendezvousCleanup;
        videoEnv.hRendezvousLoop    = hRendezvousLoop;
        videoEnv.hRendezvousLoader  = hRendezvousLoader;
        videoEnv.hPauseProcess      = hPauseProcess;
        videoEnv.hPausePrime        = hPausePrime;
        videoEnv.hDisplayInFifo     = displayEnv.hInFifo;
        videoEnv.hDisplayOutFifo    = displayEnv.hOutFifo;
        videoEnv.videoFile          = args.videoFile;
        videoEnv.videoDecoder       = args.videoDecoder->codecName;
        videoEnv.params             = args.videoDecoder->params;
        videoEnv.dynParams          = args.videoDecoder->dynParams;
        videoEnv.loop               = args.loop;
        videoEnv.engineName         = engine->engineName;
        videoEnv.videoStd           = args.videoStd;        

        if (pthread_create(&videoThread, &attr, videoThrFxn, &videoEnv)) {
            ERR("Failed to create video thread\n");
            cleanup(EXIT_FAILURE);
        }

        initMask |= VIDEOTHREADCREATED;

        /*
         * Wait for the Loader to be created in the video thread before
         * launching the loader thread.
         */
        Rendezvous_meet(hRendezvousLoader);

        /* Set the loader thread priority */
        schedParam.sched_priority = LOADER_THREAD_PRIORITY;
        if (pthread_attr_setschedparam(&attr, &schedParam)) {
            ERR("Failed to set scheduler parameters\n");
            return -1;
        }

        /* Create the loader thread */
        loaderEnv.hRendezvousInit    = hRendezvousInit;
        loaderEnv.hRendezvousCleanup = hRendezvousCleanup;
        loaderEnv.loop               = args.loop;
        loaderEnv.hLoader            = videoEnv.hLoader;

        if (pthread_create(&loaderThread, &attr, loaderThrFxn, &loaderEnv)) {
            ERR("Failed to create loader thread\n");
            cleanup(EXIT_FAILURE);
        }

        initMask |= LOADERTHREADCREATED;
    }

    /* Create the speech thread if a file name is supplied */
    if (args.speechFile) {
        /* Set the thread priority */
        schedParam.sched_priority = SPEECH_THREAD_PRIORITY;
        if (pthread_attr_setschedparam(&attr, &schedParam)) {
            ERR("Failed to set scheduler parameters\n");
            cleanup(EXIT_FAILURE);
        }

        /* Create the speech thread */
        speechEnv.hRendezvousInit       = hRendezvousInit;
        speechEnv.hRendezvousCleanup    = hRendezvousCleanup;
        speechEnv.hRendezvousLoop       = hRendezvousLoop;
        speechEnv.hPauseProcess         = hPauseProcess;
        speechEnv.speechFile            = args.speechFile;
        speechEnv.speechDecoder         = args.speechDecoder->codecName;
        speechEnv.params                = args.speechDecoder->params;
        speechEnv.dynParams             = args.speechDecoder->dynParams;
        speechEnv.loop                  = args.loop;
        speechEnv.engineName            = engine->engineName;

        if (pthread_create(&speechThread, &attr, speechThrFxn, &speechEnv)) {
            ERR("Failed to create speech thread\n");
            cleanup(EXIT_FAILURE);
        }

        initMask |= SPEECHTHREADCREATED;
    }

    /* Main thread becomes the control thread */
    ctrlEnv.hRendezvousInit    = hRendezvousInit;
    ctrlEnv.hRendezvousCleanup = hRendezvousCleanup;
    ctrlEnv.hPauseProcess      = hPauseProcess;
    ctrlEnv.keyboard           = args.keyboard;
    ctrlEnv.time               = args.time;
    ctrlEnv.hUI                = hUI;
    ctrlEnv.engineName         = engine->engineName;

    ret = ctrlThrFxn(&ctrlEnv);

    if (ret == THREAD_FAILURE) {
        status = EXIT_FAILURE;
    }

cleanup:
    /* Make sure the other threads aren't waiting for us */
    if (hRendezvousInit) Rendezvous_force(hRendezvousInit);
    if (hRendezvousLoader) Rendezvous_force(hRendezvousLoader);
    if (hRendezvousLoop) Rendezvous_force(hRendezvousLoop);
    if (hPauseProcess) Pause_off(hPauseProcess);
    if (hPausePrime) Pause_off(hPausePrime);

    if (initMask & SPEECHTHREADCREATED) {
        if (pthread_join(speechThread, &ret) == 0) {
            if (ret == THREAD_FAILURE) {
                status = EXIT_FAILURE;
            }
        }
    }

    if (initMask & LOADERTHREADCREATED) {
        if (pthread_join(loaderThread, &ret) == 0) {
            if (ret == THREAD_FAILURE) {
                status = EXIT_FAILURE;
            }
        }
    }

    if (initMask & VIDEOTHREADCREATED) {
        if (pthread_join(videoThread, &ret) == 0) {
            if (ret == THREAD_FAILURE) {
                status = EXIT_FAILURE;
            }
        }
    }

    if (initMask & DISPLAYTHREADCREATED) {
        if (pthread_join(displayThread, &ret) == 0) {
            if (ret == THREAD_FAILURE) {
                status = EXIT_FAILURE;
            }
        }
    }

    if (displayEnv.hOutFifo) {
        Fifo_delete(displayEnv.hOutFifo);
    }

    if (displayEnv.hInFifo) {
        Fifo_delete(displayEnv.hInFifo);
    }

    if (hRendezvousLoop) {
        Rendezvous_delete(hRendezvousLoop);
    }

    if (hRendezvousCleanup) {
        Rendezvous_delete(hRendezvousCleanup);
    }

    if (hRendezvousInit) {
        Rendezvous_delete(hRendezvousInit);
    }

    if (hPauseProcess) {
        Pause_delete(hPauseProcess);
    }

    if (hPausePrime) {
        Pause_delete(hPausePrime);
    }

    if (hUI) {
        UI_delete(hUI);
    }

    system("sync");
    system("echo 3 > /proc/sys/vm/drop_caches");
    
    pthread_mutex_destroy(&gbl.mutex);

    if (args.interface) {
        /* Launch the demo selection interface when exiting */
        if (execl("./interface", "interface", "-l 4", (char *) NULL) == -1) {
            status = EXIT_FAILURE;
        }
    }

    exit(status);
}

