/*
 * video.c
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2009
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

#include <stdio.h>
#include <string.h>

#include <xdc/std.h>

#include <ti/sdo/ce/Engine.h>
#include <ti/sdo/ce/osal/Memory.h>

#include <ti/sdo/dmai/Fifo.h>
#include <ti/sdo/dmai/Pause.h>
#include <ti/sdo/dmai/BufTab.h>
#include <ti/sdo/dmai/Loader.h>
#include <ti/sdo/dmai/VideoStd.h>
#include <ti/sdo/dmai/ce/Vdec2.h>
#include <ti/sdo/codecs/h264dec/ih264vdec.h>
#include <ti/sdo/dmai/BufferGfx.h>
#include <ti/sdo/dmai/Rendezvous.h>
#include <ti/sdo/dmai/priv/_Buffer.h>

#include "video.h"
#include "../demo.h"

/* Number of buffers between the video and display threads */
#define DISPLAY_PIPE_SIZE       18

/* The masks to use for knowing when a buffer is free */
#define CODEC_FREE              0x1
#define DISPLAY_FREE            0x2

#ifndef YUV_420SP
#define YUV_420SP 256
#endif 

#define CorruptFrm			//Sithara --> Added to overcome buffer loss after corrupted frame

/******************************************************************************
 * resizeBufTab
******************************************************************************/
static Int resizeBufTab(Vdec2_Handle hVd2, Int displayBufs)
{
    BufTab_Handle hBufTab = Vdec2_getBufTab(hVd2);
    Int numBufs, numCodecBuffers, numExpBufs;
    Buffer_Handle hBuf;
    Int32 frameSize;

    /* How many buffers can the codec keep at one time? */
    numCodecBuffers = Vdec2_getMinOutBufs(hVd2);
		printf("numCodecBuffers = %d\n",numCodecBuffers);
    if (numCodecBuffers < 0) {
        ERR("Failed to get buffer requirements\n");
        return FAILURE;
    }

    /*
     * Total number of frames needed are the number of buffers the codec
     * can keep at any time, plus the number of frames in the display pipe.
     */
    numBufs = numCodecBuffers + displayBufs;

    /* Get the size of output buffers needed from codec */
    frameSize = Vdec2_getOutBufSize(hVd2);

    /*
     * Get the first buffer of the BufTab to determine buffer characteristics.
     * All buffers in a BufTab share the same characteristics.
     */
    hBuf = BufTab_getBuf(hBufTab, 0);

    /* Do we need to resize the BufTab? */
    if (numBufs > BufTab_getNumBufs(hBufTab) ||
        frameSize < Buffer_getSize(hBuf)) {

        /* Should we break the current buffers in to many smaller buffers? */
        if (frameSize < Buffer_getSize(hBuf)) {

            /*
             * Chunk the larger buffers of the BufTab in to smaller buffers
             * to accomodate the codec requirements.
             */
            numExpBufs = BufTab_chunk(hBufTab, numBufs, frameSize);

            if (numExpBufs < 0) {
                ERR("Failed to chunk %d bufs size %ld to %d bufs size %ld\n",
                    BufTab_getNumBufs(hBufTab), Buffer_getSize(hBuf),
                    numBufs, frameSize);
                return FAILURE;
            }

            /*
             * Did the current BufTab fit the chunked buffers,
             * or do we need to expand the BufTab (numExpBufs > 0)?
             */
            if (BufTab_expand(hBufTab, numExpBufs) < 0) {
                ERR("Failed to expand BufTab with %d buffers numExpBufs\n",
                    numExpBufs);
                return FAILURE;
            }
        }
        else {
            /* Just expand the BufTab with more buffers */
            if (BufTab_expand(hBufTab, numCodecBuffers) < 0) {
                ERR("Failed to expand BufTab with %d buffers numCodecBuffers\n",
                    numCodecBuffers);
                return FAILURE;
            }
        }
    }

    return numBufs;
}

/******************************************************************************
 * handleCodecBufs
******************************************************************************/
static Int handleCodecBufs(Vdec2_Handle hVd2, Fifo_Handle hFifo, int outputbufid, int freebufid, Fifo_Handle hDisplayOutFifo, int prime)
{
    Buffer_Handle hOutBuf, hFreeBuf;
    Int numDisplayBufs = 0;
		int i = 0;
		int minoutbufs = 0;
		//static int first = 1;
		BufferGfx_Dimensions dim;
		BufTab_Handle hOutBufTab = Vdec2_getBufTab(hVd2);
		//VIDDEC2_Handle hDecode = Vdec2_getVisaHandle(hVd2);
    /* Get a buffer for display from the codec */
    hOutBuf = Vdec2_getDisplayBuf(hVd2);
		
    while (hOutBuf) {
        /* Send buffer to display thread */
        if (Fifo_put(hFifo, hOutBuf) < 0) {
            ERR("Failed to send buffer to display thread\n");
            return FAILURE;
        }

        numDisplayBufs++;
        
        /* Get another buffer for display from the codec */
        hOutBuf = Vdec2_getDisplayBuf(hVd2);
    }

    /* Get a buffer to free from the codec */
    hFreeBuf = Vdec2_getFreeBuf(hVd2);
    while (hFreeBuf) {
        /* The codec is no longer using the buffer */
        Buffer_freeUseMask(hFreeBuf, CODEC_FREE);
#ifdef CorruptFrm
				if (outputbufid==0) {
					i = Buffer_getId(hFreeBuf);		
					if (i == (freebufid-1)) {
						//printf("#######################Freebuf got from decoder after critical error is %d\n",i);		
						BufferGfx_setFrameType(hFreeBuf,Buffer_Type_GRAPHICS);
						BufferGfx_resetDimensions(hFreeBuf);   	
						Buffer_print(hFreeBuf);
						if (prime) {
							Buffer_freeUseMask(hFreeBuf, DISPLAY_FREE);
						}
						else {      
						//	BufTab_print(hOutBufTab);	
							if (Fifo_put(hDisplayOutFifo, hFreeBuf) < 0) {
									ERR("Failed to send buffer to display thread\n");
									return FAILURE;
							}
						}
					}
				}
#endif
        hFreeBuf = Vdec2_getFreeBuf(hVd2);
    }

    return numDisplayBufs;
}

/******************************************************************************
 * videoThrFxn
 ******************************************************************************/
Void *videoThrFxn(Void *arg)
{
    VideoEnv               *envp                = (VideoEnv *) arg;
    Void                   *status              = THREAD_SUCCESS;
    VIDDEC2_Params          defaultParams       = Vdec2_Params_DEFAULT;
    VIDDEC2_DynamicParams   defaultDynParams    = Vdec2_DynamicParams_DEFAULT;
    BufferGfx_Attrs         gfxAttrs            = BufferGfx_Attrs_DEFAULT;
    Loader_Attrs            lAttrs              = Loader_Attrs_DEFAULT;
    Vdec2_Handle            hVd2                = NULL;
    Loader_Handle           hLoader             = NULL;
    BufTab_Handle           hBufTab             = NULL;
    Engine_Handle           hEngine             = NULL;
    Buffer_Handle           hDstBuf, hInBuf, hDispBuf;
    Int                     bufIdx, bufsSent, numDisplayBufs, numBufs;
    Int                     fifoRet, ret, frameNbr;
    IH264VDEC_Params        extnParams;
    VIDDEC2_Params         *params;
    VIDDEC2_DynamicParams  *dynParams;
    Int32                   bufSize;
    ColorSpace_Type         colorSpace = ColorSpace_YUV420PSEMI;    
#if 1 // Chitat: disabled for the Platinum H.264 Codecs
	IH264VDEC_Params 		h264params;
	IH264VDEC_DynamicParams h264DynParams;
#endif
		int i = 0;
    VIDDEC2_OutArgs         outArgs;
		BufferGfx_Dimensions dim;
		int fcnt = 0;
		int priming;

    /* Open the codec engine */
    hEngine = Engine_open(envp->engineName, NULL, NULL);

    if (hEngine == NULL) {
        ERR("Failed to open codec engine %s\n", envp->engineName);
        cleanup(THREAD_FAILURE);
    }

    /* Use supplied params if any, otherwise use defaults */
    params = envp->params ? envp->params : &defaultParams;
    dynParams = envp->dynParams ? envp->dynParams : &defaultDynParams;

#if 0 // Chitat: disabled for the Platinum H.264 Codecs
    if (envp->videoStd == VideoStd_720P_60) {
        params->maxFrameRate = 30000;
        params->maxWidth     = VideoStd_720P_WIDTH;
        params->maxHeight    = VideoStd_720P_HEIGHT;                        
    } else if (envp->videoStd == VideoStd_D1_PAL) {
        params->maxFrameRate = 25000;
        params->maxWidth     = VideoStd_D1_WIDTH;
        params->maxHeight    = VideoStd_D1_PAL_HEIGHT;                
    } else {
        params->maxFrameRate = 30000;
        params->maxWidth     = VideoStd_D1_WIDTH;
        params->maxHeight    = VideoStd_D1_NTSC_HEIGHT;        
    }
    
    if (colorSpace ==  ColorSpace_YUV420PSEMI) { 
        params->forceChromaFormat = XDM_YUV_420SP;
    } else {
        params->forceChromaFormat = XDM_YUV_422ILE;
    }
   
   
    extnParams.displayDelay = 16;
    extnParams.hdvicpHandle = (void*) NULL;
    extnParams.resetHDVICPeveryFrame = 0;
    extnParams.disableHDVICPeveryFrame = 1;

    if(strcmp(envp->videoDecoder,"h264dec") == 0)
    params->size = sizeof(IH264VDEC_Params);
    
    extnParams.viddecParams = *params;
   
    /* Create the video decoder */
    hVd2 = Vdec2_create(hEngine, envp->videoDecoder, (VIDDEC2_Params*)&extnParams, dynParams);
#else
	h264params.viddecParams.size = sizeof(IH264VDEC_Params);
    if (envp->videoStd == VideoStd_720P_60) {
        h264params.viddecParams.maxFrameRate = 30000;
        h264params.viddecParams.maxWidth     = VideoStd_720P_WIDTH;
        h264params.viddecParams.maxHeight    = VideoStd_720P_HEIGHT;                        
    } else if (envp->videoStd == VideoStd_D1_PAL) {
        h264params.viddecParams.maxFrameRate = 25000;
        h264params.viddecParams.maxWidth     = VideoStd_D1_WIDTH;
        h264params.viddecParams.maxHeight    = VideoStd_D1_PAL_HEIGHT;                
    } else {
        h264params.viddecParams.maxFrameRate = 30000;
        h264params.viddecParams.maxWidth     = VideoStd_D1_WIDTH;
        h264params.viddecParams.maxHeight    = VideoStd_D1_NTSC_HEIGHT;        
    }
    if (colorSpace ==  ColorSpace_YUV420PSEMI) { 
        h264params.viddecParams.forceChromaFormat = XDM_YUV_420SP;
    } else {
        h264params.viddecParams.forceChromaFormat = XDM_YUV_422ILE;
    }
	h264params.viddecParams.maxBitRate = 10*1000*1000;
	h264params.viddecParams.dataEndianness = XDM_BYTE;
	h264params.hdvicpHandle = (void*) NULL;
	h264params.displayDelay = 0;
	h264params.levelLimit= 22;
	h264params.frame_closedloop_flag= 0;	//0 - Universal; 1 - closed loop
	h264params.inputDataMode= 1;  
	h264params.sliceFormat= 1;
	h264params.disableHDVICPeveryFrame = 0;//1;

    h264DynParams.viddecDynamicParams.size = sizeof(IH264VDEC_DynamicParams);
	h264DynParams.viddecDynamicParams.decodeHeader = XDM_DECODE_AU;
	h264DynParams.viddecDynamicParams.displayWidth = 0;
	h264DynParams.viddecDynamicParams.frameSkipMode = IVIDEO_NO_SKIP;
#if 0 // Chitat: disabled for Platinium H.264 Codecs
	h264DynParams.viddecDynamicParams.frameOrder = IVIDDEC2_DISPLAY_ORDER;
#else
	h264DynParams.viddecDynamicParams.frameOrder = IVIDDEC2_DECODE_ORDER;
#endif
	h264DynParams.viddecDynamicParams.newFrameFlag = 0;
	h264DynParams.viddecDynamicParams.mbDataFlag = 0;

    h264DynParams.resetHDVICPeveryFrame = 1;
    h264DynParams.dataSyncHandle = NULL;
    h264DynParams.getDataFxn = NULL; 

	hVd2=Vdec2_create(hEngine, envp->videoDecoder,(VIDDEC2_Params*)&h264params,(VIDDEC2_DynamicParams *)&h264DynParams);
#endif

    if (hVd2 == NULL) {
        ERR("Failed to create video decoder: %s\n", envp->videoDecoder);
        cleanup(THREAD_FAILURE);
    }

    /* Which output buffer size does the codec require? */
    bufSize = Vdec2_getOutBufSize(hVd2);

    /* Both the codec and the display thread can own a buffer */
    gfxAttrs.bAttrs.useMask = CODEC_FREE | DISPLAY_FREE;

    /* Color space */
    gfxAttrs.colorSpace = colorSpace;

    /* Set the original dimensions of the Buffers to the max */
#if 0 // Chitat: disabled for the Platinum H.264 Codecs
    gfxAttrs.dim.width = params->maxWidth;
    gfxAttrs.dim.height = params->maxHeight;
#else
    gfxAttrs.dim.width = h264params.viddecParams.maxWidth;
    gfxAttrs.dim.height = h264params.viddecParams.maxHeight;
#endif
    gfxAttrs.dim.lineLength = BufferGfx_calcLineLength(gfxAttrs.dim.width,
                                                       colorSpace);

    /* Create a table of buffers for decoded data */
    hBufTab = BufTab_create(DISPLAY_PIPE_SIZE, bufSize,
                            BufferGfx_getBufferAttrs(&gfxAttrs));

    if (hBufTab == NULL) {
        ERR("Failed to create BufTab for display pipe\n");
        cleanup(THREAD_FAILURE);
    }

    /* The codec is going to use this BufTab for output buffers */
    Vdec2_setBufTab(hVd2, hBufTab);

    /* Ask the codec how much input data it needs */
    lAttrs.readSize = Vdec2_getInBufSize(hVd2);

    /* Let the loader thread read 300000 bytes extra */
    lAttrs.readAhead = 300000;

    /* Make the total ring buffer larger */
    lAttrs.readBufSize = (lAttrs.readSize + lAttrs.readAhead) * 2;

    /* Use asynchronous mode since we have a separate loader thread */
    lAttrs.async = TRUE;

    /* Create the file loader for reading encoded data */
    hLoader = Loader_create(envp->videoFile, &lAttrs);

    if (hLoader == NULL) {
        ERR("Failed to create loader for file %s\n", envp->videoFile);
        cleanup(THREAD_FAILURE);
    }

    /* The environment copy will be shared with the loader thread */
    envp->hLoader = hLoader;

    /* Signal that the Loader is created */
    Rendezvous_meet(envp->hRendezvousLoader);

    /* Make sure the display thread is stopped when it's unlocked */
    Pause_on(envp->hPausePrime);

    /* Signal that initialization is done and wait for other threads */
    Rendezvous_meet(envp->hRendezvousInit);

prime: 
    /* Initialize the state of the decode */
    ret = Dmai_EOK;
    frameNbr = 0;
    numDisplayBufs = 0;
    numBufs = DISPLAY_PIPE_SIZE;
		priming = 1;

    /* Prime the file loader */
    if (Loader_prime(hLoader, &hInBuf) < 0) {
        ERR("Failed to prime loader for file %s\n", envp->videoFile);
        cleanup(THREAD_FAILURE);
    }

    /* Prime the display thread with video buffers */
    for (bufIdx=0; bufIdx < numBufs; bufIdx++) {
        if (ret != Dmai_EFIRSTFIELD) {
            /* Get a free buffer from the BufTab */
            hDstBuf = BufTab_getFreeBuf(hBufTab);

            if (hDstBuf == NULL) {
                ERR("Failed to get free buffer from display pipe BufTab\n");
                BufTab_print(hBufTab);
                cleanup(THREAD_FAILURE);
            }
        }

        /* Make sure the whole buffer is used for output */
        BufferGfx_resetDimensions(hDstBuf);

        /* Decode the video buffer */
        ret = Vdec2_process_sithara(hVd2, hInBuf, hDstBuf, &outArgs);				

        if (ret < 0) {
            ERR("Failed to decode video buffer\n");
            cleanup(THREAD_FAILURE);
        }

        /* If no encoded data was used we cannot find the next frame */
        if (ret == Dmai_EBITERROR && Buffer_getNumBytesUsed(hInBuf) == 0) {
            ERR("Fatal bit error\n");
            cleanup(THREAD_FAILURE);
        }
#ifdef CorruptFrm
				if (outArgs.outputID[0]==0) {
				//printf("outArgs.outputID[0] = %d outArgs.freeBufID[0] %d numbufs %d\n",outArgs.outputID[0], outArgs.freeBufID[0], BufTab_getNumBufs(hBufTab));
				//		printf("################################outArgs.outputID[0]==0\n");
						bufIdx--;			
				}
#endif
        /* Increment statistics for the user interface */
        gblIncVideoBytesProcessed(Buffer_getNumBytesUsed(hInBuf));

        /* Send frames to display thread */
        bufsSent = handleCodecBufs(hVd2, envp->hDisplayInFifo, outArgs.outputID[0], outArgs.freeBufID[0], envp->hDisplayOutFifo, priming);

        if (bufsSent < 0) {
            cleanup(THREAD_FAILURE);
        }

        /* Keep track of the number of buffers sent to the display thread */
        numDisplayBufs += bufsSent;

        if (frameNbr == 0) {
            /*
             * Resize the BufTab after the first frame has been processed.
             * This because the codec may not know it's buffer requirements
             * before the first frame has been decoded.
             */
            numBufs = resizeBufTab(hVd2, DISPLAY_PIPE_SIZE);
						//printf("\n############################### numbufs = %d\n",numBufs);

            if (numBufs < 0) {
                cleanup(THREAD_FAILURE);
            }
        }

        /* Load a new encoded frame from the file system */
        if (Loader_getFrame(hLoader, hInBuf) < 0) {
            ERR("Failed to get frame of encoded data\n");
            cleanup(THREAD_FAILURE);
        }

        /* End of clip? */
        if (Buffer_getUserPtr(hInBuf) == NULL) {
            printf("Clip ended, exiting demo..\n");
            cleanup(THREAD_SUCCESS);
        }

        frameNbr++;
    }

    /* Release the display thread, it is now fully primed */
    Pause_off(envp->hPausePrime);
		
		priming = 0;

    /* Main loop */
    while (!gblGetQuit()) {
        if (ret != Dmai_EFIRSTFIELD) {
            /* Get a displayed frame from the display thread */
            fifoRet = Fifo_get(envp->hDisplayOutFifo, &hDispBuf);

            if (fifoRet != Dmai_EOK) {
                cleanup(THREAD_FAILURE);
            }

            /* Did the display thread flush the fifo? */
            if (fifoRet == Dmai_EFLUSH) {
                cleanup(THREAD_SUCCESS);
            }

            /* The display thread is no longer using the buffer */
            hDstBuf = BufTab_getBuf(hBufTab, Buffer_getId(hDispBuf));
            Buffer_freeUseMask(hDstBuf, DISPLAY_FREE);

            /* Keep track of the number of buffers sent to the display thread */
            numDisplayBufs--;

            /* Get a free buffer from the BufTab to give to the codec */
            hDstBuf = BufTab_getFreeBuf(hBufTab);

            if (hDstBuf == NULL) {
                ERR("Failed to get free buffer from BufTab\n");
                BufTab_print(hBufTab);
                cleanup(THREAD_FAILURE);
            }
						
        }

        /* Make sure the whole buffer is used for output */
        BufferGfx_resetDimensions(hDstBuf);

        /* Decode the video buffer */
        ret = Vdec2_process_sithara(hVd2, hInBuf, hDstBuf, &outArgs);			

        if (ret < 0) {
            ERR("Failed to decode video buffer\n");
            cleanup(THREAD_FAILURE);
        }
				
        /* If no encoded data was used we cannot find the next frame */
        if (ret == Dmai_EBITERROR && Buffer_getNumBytesUsed(hInBuf) == 0) {
            ERR("Fatal bit error\n");
            cleanup(THREAD_FAILURE);
        }
        /* Increment statistics for the user interface */
        gblIncVideoBytesProcessed(Buffer_getNumBytesUsed(hInBuf));
 
        
        /* Send frames to display thread */
        bufsSent = handleCodecBufs(hVd2, envp->hDisplayInFifo, outArgs.outputID[0], outArgs.freeBufID[0], envp->hDisplayOutFifo, priming);
			//	printf("frameNbr %d envp->hDisplayInFifo %d envp->hDisplayOutFifo %d\n",frameNbr, Fifo_getNumEntries(envp->hDisplayInFifo), Fifo_getNumEntries(envp->hDisplayOutFifo));
				//for TI encoder : envp->hDisplayInFifo 34 envp->hDisplayOutFifo 0
				
        if (bufsSent < 0) {
            cleanup(THREAD_FAILURE);
        }

        /* Keep track of the number of buffers sent to the display thread */
        numDisplayBufs += bufsSent;

        /* Load a new encoded frame from the file system */
        if (Loader_getFrame(hLoader, hInBuf) < 0) {
            ERR("Failed to get frame of encoded data\n");
            cleanup(THREAD_FAILURE);
        }

        frameNbr++;

        /* End of clip? */
        if (Buffer_getUserPtr(hInBuf) == NULL) {
                
                /* Flush the codec for display frames */
                Vdec2_flush(hVd2);
                bufsSent = 0;
                do {
                    /*
                     * Temporarily create a temporary dummy buffer for the process call.
                     * After a flush the codec ignores the input buffer, but since Codec
                     * Engine still address translates the buffer, it needs to exist.
                     */
                    hInBuf = Buffer_create(1, BufferGfx_getBufferAttrs(&gfxAttrs));

                    if (hInBuf == NULL) {
                        ERR("Failed to allocate dummy buffer\n");
                        cleanup(THREAD_FAILURE);
                    }

                    Buffer_setNumBytesUsed(hInBuf, 1);

										ret = Vdec2_process_sithara(hVd2, hInBuf, hDstBuf, &outArgs);			

                    if (ret < 0) {
                        ERR("Failed to decode video buffer\n");
                        cleanup(THREAD_FAILURE);
                    }

                    Buffer_delete(hInBuf);
            
                    /* Keep track of the number of buffers sent to the display thread */
                    numDisplayBufs += bufsSent;

                    /* Send frames to display thread */
                    bufsSent = handleCodecBufs(hVd2, envp->hDisplayInFifo, outArgs.outputID[0], outArgs.freeBufID[0], envp->hDisplayOutFifo, priming);

                }while(bufsSent > 0);

            /* Drain the display thread making sure all frames are displayed */
            while (numDisplayBufs > 0) {
                /* Get a displayed frame from the display thread */
                fifoRet = Fifo_get(envp->hDisplayOutFifo, &hDispBuf);

                if (fifoRet != Dmai_EOK) {
                    cleanup(THREAD_FAILURE);
                }

                /* Did the display thread flush the fifo? */
                if (fifoRet == Dmai_EFLUSH) {
                    cleanup(THREAD_SUCCESS);
                }

                /* The display thread is no longer using the buffer */
                hDstBuf = BufTab_getBuf(hBufTab, Buffer_getId(hDispBuf));
                Buffer_freeUseMask(hDstBuf, DISPLAY_FREE);

                /* Keep track of number of buffers sent to the display thread */
                numDisplayBufs--;
            }

            /* Wait for audio to complete if applicable */
            Rendezvous_meet(envp->hRendezvousLoop);

            /* Loop the clip or quit? */
            if (envp->loop) {
                /* Make sure the decoder has no state by recreating it */
                Vdec2_delete(hVd2);

                /* Make sure any buffers kept by the codec are freed */
                BufTab_freeAll(hBufTab);

                /* Undo any previous chunking done */
                BufTab_collapse(Vdec2_getBufTab(hVd2));
#if 0 // Chitat: disabled for the Platinum H.264 Codecs
                hVd2 = Vdec2_create(hEngine, envp->videoDecoder,
                                    (VIDDEC2_Params*)&extnParams, dynParams);
#else
				hVd2=Vdec2_create(hEngine, envp->videoDecoder,(VIDDEC2_Params*)&h264params,(VIDDEC2_DynamicParams *)&h264DynParams);
#endif
                if (hVd2 == NULL) {
                    ERR("Failed to create video decoder: %s\n",
                        envp->videoDecoder);
                    cleanup(THREAD_FAILURE);
                }

                /* The codec is going to use this BufTab for output buffers */
                Vdec2_setBufTab(hVd2, hBufTab);

                /* Halt the display thread for priming */
                Pause_on(envp->hPausePrime);

                goto prime;
            }
            else {
                printf("Clip ended, exiting demo..\n");
                gblSetQuit();
            }
        } /* End of clip? */
    } /* Main loop */

cleanup:
    /* Make sure the other threads aren't waiting for us */
    Rendezvous_force(envp->hRendezvousInit);
    Rendezvous_force(envp->hRendezvousLoop);
    Rendezvous_force(envp->hRendezvousLoader);
    Pause_off(envp->hPauseProcess);
    Pause_off(envp->hPausePrime);
    Fifo_flush(envp->hDisplayInFifo);
    if (hLoader) Loader_flush(hLoader);

    /* Meet up with other threads before cleaning up */
    Rendezvous_meet(envp->hRendezvousCleanup);

    /* Clean up the thread before exiting */
    if (hLoader) {
        Loader_delete(hLoader);
    }

    if (hVd2) {
        Vdec2_delete(hVd2);
    }

    if (hEngine) {
        Engine_close(hEngine);
    }

    if (hBufTab) {
        BufTab_delete(hBufTab);
    }

    return status;
}

