/**  
 * @file pa_mgmt.c
 *
 * @brief 
 *  Packet accelerator subsystem management functions.
 *  
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <cpsw_singlecore.h>

/* PA LLD include */
#include <ti/drv/pa/pa.h>

/* PASS RL file */
#include <ti/csl/cslr_device.h>
#include <ti/csl/cslr_pa_ss.h>

/* Firmware images */
#include <ti/drv/pa/fw/pafw.h>

/* PA command response queue handle */
Qmss_QueueHnd                           gPaCfgCmdRespQHnd;

/* Number of PA internal buffers to allocate */
#define     PA_NUM_BUFFERS              3

/* PA definitions */
#define     MAX_NUM_L2_HANDLES          10
#define     MAX_NUM_L3_HANDLES          20
#define     MAX_NUM_L4_HANDLES          40

#define     BUFSIZE_PA_INST             256
#define     BUFSIZE_L2_TABLE            1000
#define     BUFSIZE_L3_TABLE            4000

/* PA instance */
#pragma DATA_ALIGN(gPAInst, 8)
UInt8                                   gPAInst[BUFSIZE_PA_INST];

/* PA Driver Handle */
Pa_Handle                               gPAInstHnd;

/* Memory used for PA handles */
#pragma DATA_ALIGN(gMemL2Ram, 8)
UInt8                                   gMemL2Ram[BUFSIZE_L2_TABLE];

#pragma DATA_ALIGN(gMemL3Ram, 8)
UInt8                                   gMemL3Ram[BUFSIZE_L3_TABLE];

paHandleL2L3_t                          gPaL2Handles[MAX_NUM_L2_HANDLES];
paHandleL2L3_t                          gPaL3Handles[MAX_NUM_L3_HANDLES];
paHandleL4_t                            gPaL4Handles[MAX_NUM_L4_HANDLES];

extern volatile unsigned int cregister TSCL;

extern  Qmss_QueueHnd                   gPaTxQHnd [NUM_PA_TX_QUEUES], gTxFreeQHnd, gRxQHnd;

/** ============================================================================
 *   @n@b CycleDelay
 *
 *   @b Description
 *   @n This API implements a clock delay logic using the Time Stamp Counter (TSC)
 *      of the DSP.
 *
 *   @param[in]  
 *   @n count               Number of delay cycles to wait.
 * 
 *   @return    
 *   @n None
 * =============================================================================
 */
Void CycleDelay (Int32 count)
{
    UInt32                  sat;

    if (count <= 0)
        return;

    /* Get the current TSC and add a delay */
    sat =   TSCL + (UInt32)count;

    while (TSCL < sat);
}

/** ============================================================================
 *   @n@b Download_PAFirmware
 *
 *   @b Description
 *   @n This API downloads the PA firmware required for PDSP operation.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
Int32 Download_PAFirmware (Void)
{
    Int32                   i;

    /* Hold the PA in reset state during download */
    Pa_resetControl (gPAInstHnd, pa_STATE_RESET);

    /* PDPSs 0-2 use image c1 */
    for (i = 0; i < 3; i++)
    {
        Pa_downloadImage (gPAInstHnd, i, (Ptr)c1, c1Size);
    }

    /* PDSP 3 uses image c2 */
    Pa_downloadImage (gPAInstHnd, 3, (Ptr)c2, c2Size);

    /* PDSPs 4-5 use image m */
    for (i = 4; i < 6; i++)
    {
        Pa_downloadImage (gPAInstHnd, i, (Ptr)m, mSize);
    }

    /* Enable the PA back */
    Pa_resetControl (gPAInstHnd, pa_STATE_ENABLE);

    return 0;
}        

/** ============================================================================
 *   @n@b Add_MACAddress
 *
 *   @b Description
 *   @n This API adds the switch MAC address to the PA PDSP Lookup table. This 
 *      ensures that all packets destined for this MAC address get processed
 *      for forwarding to the host.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
 
//static UInt8 srcMac[6] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
UInt8 srcMac[6] = {0x30, 0x31, 0x32, 0x33, 0x34, 0x35};
 
Int32 Add_MACAddress (Void)
{
    Int32                       j;
    UInt16                      cmdSize;
    Qmss_Queue                  cmdReplyQInfo;
    paEthInfo_t                 ethInfo     =  { { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },     /* Src mac = dont care */   
                                                 //{ 0x10, 0x11, 0x12, 0x13, 0x14, 0x15 },     /* Dest mac */
    		                                     { 0x50, 0x51, 0x52, 0x53, 0x54, 0x55 },     /* Dest mac */
    		                                     //{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05 },     /* Dest mac */
                                                    0,                                          /* vlan = dont care */      
                                                    0x0800,                             		/* ether type = IPv4 */     
                                                    0,                                          /* MPLS tag = don't care */
                                                    pa_EMAC_PORT_0                              /* Input EMAC port */
                                               }; 
  
                                                    
                                                    
    paRouteInfo_t               routeInfo =     {   pa_DEST_CONTINUE_PARSE_LUT1,                /* Continue parsing */             
                                                    0,                                          /* Flow Id = dont care */          
                                                    0,                                          /* queue = dont care */            
                                                    0,                                          /* multi route = dont care */      
                                                    0,                                          /* swinfo0 = dont care */          
                                                    0,                                          /* SwInfo 1 is dont care */
                                                    0,                                          /* customType = pa_CUSTOM_TYPE_NONE */         \
                                                    0,                                          /* customIndex: not used */        \
                                                    0,                                          /* pkyType: for SRIO only */       \
                                                    NULL                                        /* No commands */
                                                };
    paRouteInfo_t               nFailInfo =     {   pa_DEST_DISCARD,                            /* Toss the packet  */           
	                                                0,                                          /* Flow Id = dont care */        
                                                    0,                                          /* queue = dont care */          
                                                    0,                                          /* mutli route = dont care */    
                                                    0,                                          /* swinfo0 = dont care */        
                                                    0,                                          /* SwInfo 1 is dont care */
                                                    0,                                          /* customType = pa_CUSTOM_TYPE_NONE */         \
                                                    0,                                          /* customIndex: not used */        \
                                                    0,                                          /* pkyType: for SRIO only */       \
                                                    NULL                                        /* No commands */
                                                };
    paCmdReply_t                cmdReplyInfo =  {   pa_DEST_HOST,                               /* Replies go to the host */            
                                                    0,                                          /* User chosen ID to go to swinfo0 */     
                                                    0,                                          /* Destination queue */                   
                                                    0                                           /* Flow ID */  
                                                };
    paReturn_t        retVal;
    paEntryHandle_t   retHandle;
    Int32             handleType, cmdDest;
    UInt32            psCmd       =   ((UInt32)(4 << 5) << 24);  
    UInt32            myswinfo[]  =   {0x11112222, 0x33334444};
    UInt8*            pCmdDataBuffer;
    Cppi_HostDesc*    pHostDesc;

    /* Get a Tx free descriptor to send a command to the PA PDSP */
    if ((pHostDesc = Qmss_queuePop (gTxFreeQHnd)) == NULL)
    {
        System_printf ("Error obtaining a Tx free descriptor \n");            
        return -1;
    }

    /* The descriptor address returned from the hardware has the 
     * descriptor size appended to the address in the last 4 bits.
     *
     * To get the true descriptor size, always mask off the last 
     * 4 bits of the address.
     */
    pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);
    
    /* Allocate a Tx buffer and attach the command info to it. */
    if ((pCmdDataBuffer = (Ptr) Memory_alloc(NULL, 320, 0, NULL)) == NULL)
    {
        System_printf ("Error allocating memory for PA Command data buffer \n");
        return -1;
    }
        
    /* Populate the Rx free descriptor with the buffer we just allocated. */
    Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 320);

    /* Save original buffer information */
    Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 320);    

    cmdSize                 =   pHostDesc->buffLen;
    cmdReplyInfo.replyId    =   0x11111111;  /* unique for each add mac command */

    /* Get the PA response queue number and populate the destination queue number
     * in the PA response configuration.
     */
    cmdReplyQInfo           =   Qmss_getQueueNumber (gPaCfgCmdRespQHnd);
    cmdReplyInfo.queue      =   cmdReplyQInfo.qNum;
    
    /* Use Source MAC as destination MAC if non-loopback */
    if(cpswLpbkMode == CPSW_LOOPBACK_NONE)
        memcpy(ethInfo.dst, srcMac, sizeof(srcMac));

    retVal  =   Pa_addMac  (gPAInstHnd,
                            pa_LUT1_INDEX_NOT_SPECIFIED, 
                            &ethInfo,
                            &routeInfo,
                            &nFailInfo,
                            &gPaL2Handles[0],
                            (paCmd_t) pHostDesc->buffPtr,
                            &cmdSize,
                            &cmdReplyInfo,
                            &cmdDest);
    if (retVal != pa_OK)  
    {
        System_printf ("Pa_addMac returned error %d\n", retVal);
        return -1;
    }
    
    /* This sets the extended info for descriptors, and this is required so PS info
     * goes to the right spot 
     */                   
    Cppi_setSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)myswinfo);

    /* Set the buffer length to the size used. It will be restored when the descriptor
     * is returned 
     */
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, cmdSize);
    pHostDesc->buffLen  =   cmdSize;
    
    /* Mark the packet as a configuration packet */
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)&psCmd, 4);
       
    /* Send the command to the PA and wait for the return */
    Qmss_queuePush (gPaTxQHnd[cmdDest - pa_CMD_TX_DEST_0], 
                    pHostDesc, 
                    pHostDesc->buffLen, 
                    SIZE_HOST_DESC, 
                    Qmss_Location_TAIL
                   );
    
    /* Poll on the PA response queue to see if response from PA has come */
    for (j = 0; j < 100; j++)  
    {
        CycleDelay (1000);

        if (Qmss_getQueueEntryCount (gPaCfgCmdRespQHnd) > 0)   
        {
            /* We have a response from PA PDSP for the command we submitted earlier for 
             * MAC address addition.
             */
            pHostDesc = Qmss_queuePop (gPaCfgCmdRespQHnd);

            /* Clear the size bytes */
            pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);

            if (pHostDesc->softwareInfo0 != cmdReplyInfo.replyId)  
            {
                System_printf ("Found an entry in PA response queue with swinfo0 = 0x%08x, expected 0x%08x\n", 
                                pHostDesc->softwareInfo0, cmdReplyInfo.replyId);
                pHostDesc->buffLen  =   pHostDesc->origBufferLen;
                Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

                return -1;
            }

            retVal  =   Pa_forwardResult (gPAInstHnd, (Ptr)pHostDesc->buffPtr, &retHandle, &handleType, &cmdDest);
            if (retVal != pa_OK)  
            {
                System_printf ("PA sub-system rejected Pa_addMac command\n");
                return -1;
            }
        
            /* Reset the buffer lenght and put the descriptor back on the Tx free queue */
            pHostDesc->buffLen = pHostDesc->origBufferLen;
            Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

            break;
        }
    }

    if (j == 100)  
    {
        System_printf ("Timeout waiting for reply from PA to Pa_addMac command\n");
        return -1;
    }

    return 0;
}

/** ============================================================================
 *   @n@b Add_IPAddress
 *
 *   @b Description
 *   @n This API adds the IP Address the application's using to the PA PDSP 
 *      Lookup table. This ensures that all packets destined for this 
 *      IP address get forwarded up to the host.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
Int32 Add_IPAddress (Void)
{
    Int32                       j;
    UInt16                      cmdSize;
    Qmss_Queue                  cmdReplyQInfo;
    paIpInfo_t                  ipInfo      =    {  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },   /* IP source = dont care */   
                                                    //{ 0xc0, 0xa8, 0x01, 0xa, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },   /* IP dest */
    		                                        { 0xc0, 0xa8, 0x01, 0x8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },   /* IP dest */
                                                    0,         /* SPI = dont care */                                                
                                                    0,         /* flow = dont care */                                                   
                                                    pa_IPV4,   /* IP type */                                                            
                                                    0,         /* GRE protocol */                                                       
                                                    0,         /* Ip protocol = dont care (TCP or UDP or anything else) */              
                                                    0,         /* TOS */                                                                
                                                    FALSE,     /* TOS = dont care (seperate field since TOS=0 is valid */
                                                    0          /* SCTP destination port = dont care */  
                                                };
    Int32                       macLink     =   {0};  /* Link this with the first MAC address created */
    paRouteInfo_t               routeInfo   =   {   pa_DEST_CONTINUE_PARSE_LUT2,                /* Continue parsing */             
                                                    0,                                          /* Flow Id = dont care */          
                                                    0,                                          /* queue = dont care */            
                                                    0,                                          /* multi route = dont care */      
                                                    0,                                          /* swinfo0 = dont care */          
                                                    0,                                          /* SwInfo 1 is dont care */
                                                    0,                                          /* customType = pa_CUSTOM_TYPE_NONE */         \
                                                    0,                                          /* customIndex: not used */        \
                                                    0,                                          /* pkyType: for SRIO only */       \
                                                    NULL                                        /* No commands */
                                                };
    paRouteInfo_t               nFailInfo   =   {   pa_DEST_DISCARD,                            /* Toss the packet  */           
	                                                0,                                          /* Flow Id = dont care */        
                                                    0,                                          /* queue = dont care */          
                                                    0,                                          /* mutli route = dont care */    
                                                    0,                                          /* swinfo0 = dont care */        
                                                    0,                                          /* SwInfo 1 is dont care */
                                                    0,                                          /* customType = pa_CUSTOM_TYPE_NONE */         \
                                                    0,                                          /* customIndex: not used */        \
                                                    0,                                          /* pkyType: for SRIO only */       \
                                                    NULL                                        /* No commands */
                                                };
    paCmdReply_t                cmdReplyInfo =  {   pa_DEST_HOST,                               /* Replies go to the host */            
                                                    0,                                          /* User chosen ID to go to swinfo0 */     
                                                    0,                                          /* Destination queue */                   
                                                    0                                           /* Flow ID */  
                                                };
    paReturn_t         retVal;
    paEntryHandle_t    retHandle;
    Int32              handleType, cmdDest;
    UInt32             psCmd       =   ((UInt32)(4 << 5) << 24);  
    UInt32             myswinfo[]  =   {0x11112222, 0x33334444};
    UInt8*             pCmdDataBuffer;
    Cppi_HostDesc*     pHostDesc;

    /* Get a Tx free descriptor to send a command to the PA PDSP */
    if ((pHostDesc = Qmss_queuePop (gTxFreeQHnd)) == NULL)
    {
        System_printf ("Error obtaining a Tx free descriptor \n");            
        return -1;
    }

    /* The descriptor address returned from the hardware has the 
     * descriptor size appended to the address in the last 4 bits.
     *
     * To get the true descriptor size, always mask off the last 
     * 4 bits of the address.
     */
    pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);
    
    /* Allocate a Tx buffer and attach the command info to it. */
    if ((pCmdDataBuffer = (Ptr) Memory_alloc(NULL, 1518, 0, NULL)) == NULL)
    {
        System_printf ("Error allocating memory for PA Command data buffer \n");
        return -1;
    }
        
    /* Populate the Rx free descriptor with the buffer we just allocated. */
    Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 1518);

    /* Save original buffer information */
    Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 1518);    

    cmdSize                 =   pHostDesc->buffLen;
    cmdReplyInfo.replyId    =   0x11111111;  /* unique for each add mac command */

    /* Get the PA response queue number and populate the destination queue number
     * in the PA response configuration.
     */
    cmdReplyQInfo           =   Qmss_getQueueNumber (gPaCfgCmdRespQHnd);
    cmdReplyInfo.queue      =   cmdReplyQInfo.qNum;

    retVal  =   Pa_addIp    (gPAInstHnd,
                             pa_LUT1_INDEX_NOT_SPECIFIED,
                            &ipInfo,
                            gPaL2Handles [macLink],
                            &routeInfo,
                            &nFailInfo,
                            &gPaL3Handles[0],
                            (paCmd_t) pHostDesc->buffPtr,
                            &cmdSize,
                            &cmdReplyInfo,
                            &cmdDest);
    if (retVal != pa_OK)  
    {
        System_printf ("Pa_addIp returned error %d\n", retVal);
        return -1;
    }
    
    /* This sets the extended info for descriptors, and this is required so PS info
     * goes to the right spot 
     */                   
    Cppi_setSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)myswinfo);

    /* Set the buffer length to the size used. It will be restored when the descriptor
     * is returned 
     */
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, cmdSize);
    pHostDesc->buffLen  =   cmdSize;
    
    /* Mark the packet as a configuration packet */
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)&psCmd, 4);
       
    /* Send the command to the PA and wait for the return */
    Qmss_queuePush (gPaTxQHnd[cmdDest - pa_CMD_TX_DEST_0], 
                    (UInt32 *)Convert_CoreLocal2GlobalAddr((UInt32)pHostDesc), 
                    pHostDesc->buffLen, 
                    SIZE_HOST_DESC, 
                    Qmss_Location_TAIL
                    );

    /* Poll on the PA response queue to see if response from PA has come */
    for (j = 0; j < 100; j++)  
    {
        CycleDelay (1000);

        if (Qmss_getQueueEntryCount (gPaCfgCmdRespQHnd) > 0)   
        {
            /* We have a response from PA PDSP for the command we submitted earlier for 
             * MAC address addition.
             */
            pHostDesc = Qmss_queuePop (gPaCfgCmdRespQHnd);

            /* Clear the size bytes */
            pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);

            if (pHostDesc->softwareInfo0 != cmdReplyInfo.replyId)  
            {
                System_printf ("Found an entry in PA response queue with swinfo0 = 0x%08x, expected 0x%08x\n", 
                                pHostDesc->softwareInfo0, cmdReplyInfo.replyId);
                pHostDesc->buffLen  =   pHostDesc->origBufferLen;
                Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

                return -1;
            }

            retVal  =   Pa_forwardResult (gPAInstHnd, (Ptr)pHostDesc->buffPtr, &retHandle, &handleType, &cmdDest);
            if (retVal != pa_OK)  
            {
                System_printf ("PA sub-system rejected Pa_addIp command\n");
                return -1;
            }
        
            /* Reset the buffer lenght and put the descriptor back on the Tx free queue */
            pHostDesc->buffLen = pHostDesc->origBufferLen;
            Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

            break;
        }
    }

    if (j == 100)  
    {
        System_printf ("Timeout waiting for reply from PA to Pa_addMac command\n");
        return -1;
    }

    return 0;
}

/** ============================================================================
 *   @n@b Add_Port
 *
 *   @b Description
 *   @n This API adds the UDP port the application's using to the PA PDSP 
 *      Lookup table. This ensures that all packets destined for this 
 *      UDP port get forwarded up to the host.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
Int32 Add_Port (Void)
{
    Int32                       j;
    UInt16                      cmdSize;
    Qmss_Queue                  cmdReplyQInfo;
    Qmss_Queue                  rxQInfo;
    //UInt16                      ports       =   {0x5678};
    UInt16                      ports       =   {0x8000};

    Int32                       ipLink      =   {0};  /* Link this with the first IP address created */
    paRouteInfo_t               routeInfo   =   {   pa_DEST_HOST,           /* Route a match to the host */   
                                                    0,                      /* Flow ID 0 */                   
                                                    0,                      /* Destination queue */           
                                                    -1,                     /* Multi route disabled */        
                                                    0xaaaaaaaa,             /* SwInfo 0 */                    
                                                    0,                      /* SwInfo 1 is dont care */
                                                    0,                      /* customType = pa_CUSTOM_TYPE_NONE */         \
                                                    0,                      /* customIndex: not used */        \
                                                    0,                      /* pkyType: for SRIO only */       \
                                                    NULL                    /* No commands */
                                                };                      
    paCmdReply_t                cmdReplyInfo =  {   pa_DEST_HOST,           /* Replies go to the host */            
                                                    0,                      /* User chosen ID to go to swinfo0 */     
                                                    0,                      /* Destination queue */                   
                                                    0                       /* Flow ID */  
                                                };
    paReturn_t       retVal;
    paEntryHandle_t  retHandle;
    Int32            handleType, cmdDest;
    UInt32           psCmd       =   ((UInt32)(4 << 5) << 24);  
    UInt32           myswinfo[]  =   {0x11112222, 0x33334444};
    UInt8*           pCmdDataBuffer;
    Cppi_HostDesc*   pHostDesc;
    
    routeInfo.flowId = (uint8_t)Cppi_getFlowId(gRxFlowHnd);

    /* Get a Tx free descriptor to send a command to the PA PDSP */
    if ((pHostDesc = Qmss_queuePop (gTxFreeQHnd)) == NULL)
    {
        System_printf ("Error obtaining a Tx free descriptor \n");            
        return -1;
    }

    /* The descriptor address returned from the hardware has the 
     * descriptor size appended to the address in the last 4 bits.
     *
     * To get the true descriptor size, always mask off the last 
     * 4 bits of the address.
     */
    pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);
    
    /* Allocate a Tx buffer and attach the command info to it. */
    if ((pCmdDataBuffer = (Ptr) Memory_alloc(NULL, 1518, 0, NULL)) == NULL)
    {
        System_printf ("Error allocating memory for PA Command data buffer \n");
        return -1;
    }
        
    /* Populate the Rx free descriptor with the buffer we just allocated. */
    Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 1518);

    /* Save original buffer information */
    Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)Convert_CoreLocal2GlobalAddr((UInt32)pCmdDataBuffer), 1518);    

    cmdSize                 =   pHostDesc->buffLen;
    cmdReplyInfo.replyId    =   0x11111111;  /* unique for each add mac command */

    /* Get the PA response queue number and populate the destination queue number
     * in the PA response configuration.
     */
    cmdReplyQInfo           =   Qmss_getQueueNumber (gPaCfgCmdRespQHnd);
    cmdReplyInfo.queue      =   cmdReplyQInfo.qNum;

    /* Setup the Rx queue as destination for the packets */
    rxQInfo                 =   Qmss_getQueueNumber (gRxQHnd);
    routeInfo.queue      	=   rxQInfo.qNum;

    retVal  =   Pa_addPort  (gPAInstHnd,
                            pa_LUT2_PORT_SIZE_16,
                            ports,
                            gPaL3Handles [ipLink],
                            FALSE,                      /* New Entry required */
                            pa_PARAMS_NOT_SPECIFIED,  
                            &routeInfo,
                            gPaL4Handles[0],
                            (paCmd_t) pHostDesc->buffPtr,
                            &cmdSize,
                            &cmdReplyInfo,
                            &cmdDest);
    if (retVal != pa_OK)  
    {
        System_printf ("Pa_addPort returned error %d\n", retVal);
        return -1;
    }
    
    /* This sets the extended info for descriptors, and this is required so PS info
     * goes to the right spot 
     */                   
    Cppi_setSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)myswinfo);

    /* Set the buffer length to the size used. It will be restored when the descriptor
     * is returned 
     */
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, cmdSize);
    pHostDesc->buffLen  =   cmdSize;
    
    /* Mark the packet as a configuration packet */
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)pHostDesc, (UInt8 *)&psCmd, 4);
       
    /* Send the command to the PA and wait for the return */
    Qmss_queuePush (gPaTxQHnd[cmdDest - pa_CMD_TX_DEST_0], 
                    (UInt32 *)Convert_CoreLocal2GlobalAddr((UInt32)pHostDesc), 
                    pHostDesc->buffLen, 
                    SIZE_HOST_DESC, 
                    Qmss_Location_TAIL
                   );

    /* Poll on the PA response queue to see if response from PA has come */
    for (j = 0; j < 100; j++)  
    {
        CycleDelay (1000);

        if (Qmss_getQueueEntryCount (gPaCfgCmdRespQHnd) > 0)   
        {
            /* We have a response from PA PDSP for the command we submitted earlier for 
             * MAC address addition.
             */
            pHostDesc = Qmss_queuePop (gPaCfgCmdRespQHnd);

            /* Clear the size bytes */
            pHostDesc = (Ptr) ((UInt32) pHostDesc & 0xFFFFFFF0);

            if (pHostDesc->softwareInfo0 != cmdReplyInfo.replyId)  
            {
                System_printf ("Found an entry in PA response queue with swinfo0 = 0x%08x, expected 0x%08x\n", 
                                pHostDesc->softwareInfo0, cmdReplyInfo.replyId);
                pHostDesc->buffLen  =   pHostDesc->origBufferLen;
                Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

                return -1;
            }

            retVal  =   Pa_forwardResult (gPAInstHnd, (Ptr)pHostDesc->buffPtr, &retHandle, &handleType, &cmdDest);
            if (retVal != pa_OK)  
            {
                System_printf ("PA sub-system rejected Pa_addPort command\n");
                return -1;
            }
        
            /* Reset the buffer lenght and put the descriptor back on the Tx free queue */
            pHostDesc->buffLen = pHostDesc->origBufferLen;
            Qmss_queuePush (gTxFreeQHnd, pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

            break;
        }
    }

    if (j == 100)  
    {
        System_printf ("Timeout waiting for reply from PA to Pa_addMac command\n");
        return -1;
    }

    return 0;
}

/** ============================================================================
 *   @n@b Init_PASS
 *
 *   @b Description
 *   @n This API initializes the PASS/PDSP and opens a queue that the application
 *      can use to receive command responses from the PASS.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
Int32 Init_PASS (Void)
{
	UInt8						isAllocated;				
    paSizeInfo_t                paSize;
    paConfig_t                  paCfg;
    Int32                       retVal;    
    Int32                       sizes[pa_N_BUFS];
    Int32                       aligns[pa_N_BUFS];
    Void*                       bases[pa_N_BUFS];
    
    /* Allocate space for the PA LLD buffers. The buffers we need to 
     * allocate space are:
     *      (1) PA LLD Instance Info Handle
     *      (2) PA LLD L2 Handle database
     *      (3) PA LLD L3 Handle database
     */
    paSize.nMaxL2 = MAX_NUM_L2_HANDLES;
    paSize.nMaxL3 = MAX_NUM_L3_HANDLES;
    paSize.nUsrStats = 0;
    if ((retVal = Pa_getBufferReq(&paSize, sizes, aligns)) != pa_OK)
    {
        System_printf ("Pa_getBufferReq returned error %d\n", retVal);
        return -1;
    }

    /* Validate the buffer allocations */
    /* The first buffer is always the instance buffer */
    if ((UInt32)gPAInst & (aligns[0] - 1))  
    {
        System_printf ("Pa_getBufferReq requires %d alignment for instance buffer, but address is 0x%08x\n", aligns[0], (UInt32)gPAInst);
        return -1;
    }

    if (sizeof(gPAInst) < sizes[0])  
    {
        System_printf ("Pa_getBufferReq requires %d bytes for instance buffer, have only %d\n", sizes[0], sizeof(gPAInst));
        return -1;
    }

    bases[0]    =   (Void *)gPAInst;

    /* The second buffer is the L2 table */
    if ((UInt32)gMemL2Ram & (aligns[1] - 1))  
    {
        System_printf ("Pa_getBufferReq requires %d alignment for buffer 1, but address is 0x%08x\n", aligns[1], (UInt32)gMemL2Ram);
        return (-1);
    }

    if (sizeof(gMemL2Ram) < sizes[1])  
    {
        System_printf ("Pa_getBufferReq requires %d bytes for buffer 1, have only %d\n", sizes[1], sizeof(gMemL2Ram));
        return -1;
    }

    bases[1]    =   (Void *)gMemL2Ram;

    /* The third buffer is the L3 table */
    if ((UInt32)gMemL3Ram & (aligns[2] - 1))  
    {
        System_printf ("Pa_alloc requires %d alignment for buffer 1, but address is 0x%08x\n", aligns[2], (UInt32)gMemL3Ram);
        return (-1);
    }

    if (sizeof(gMemL3Ram) < sizes[2])  
    {
        System_printf ("Pa_alloc requires %d bytes for buffer 1, have only %d\n", sizes[2], sizeof(gMemL3Ram));
        return (-1);
    }

    bases[2]    =   (Void *)gMemL3Ram;
    
    bases[3]    =   0;

    /* Finally initialize the PA LLD */
    paCfg.initTable =   TRUE;
    paCfg.initDefaultRoute = TRUE;
    paCfg.baseAddr = CSL_PA_SS_CFG_REGS;
    paCfg.sizeCfg   =   &paSize;
    if ((retVal = Pa_create (&paCfg, bases, &gPAInstHnd)) != pa_OK)  
    {
        System_printf ("Pa_create returned with error code %d\n", retVal);
        return -1;
    }

    /* Download the PASS PDSP firmware */
    if (Download_PAFirmware ())
    {
        return -1;
    }

    /* Open a PA Command Response Queue.
     *
     * This queue will be used to hold responses from the PA PDSP for all the
     * commands issued by the example application.
     *
     * This queue is used only at configuration time to setup the PA PDSP.
     */
    if ((gPaCfgCmdRespQHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    {
        System_printf ("Error opening a PA Command Response queue \n");
        return -1;
    }            

    /* Init done. Return success. */
    return 0;
}

/** ============================================================================
 *   @n@b Setup_PASS
 *
 *   @b Description
 *   @n This API sets up the PA LLD/PDSP with MAC/IP/UDP configuration used by
 *      the example application.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    Int32
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
Int32 Setup_PASS (Void)
{
    /* Setup the PA PDSP to forward packets matching our switch MAC 
     * address up to the host onto the example application.
     */
    if (Add_MACAddress () != 0)
    {
        return -1;
    }

    /* Add the IP address the example uses */
    if (Add_IPAddress () != 0)
    {
        return -1;
    }

    /* Add the port number on which our application is going to listen on */
    if (Add_Port () != 0)
    {
        return -1;
    }

    /* Return success */
    return 0;
}

void mdebugHaltPdsp (Int pdspNum)
{
    CSL_Pa_ssRegs *passRegs = (CSL_Pa_ssRegs *)CSL_PA_SS_CFG_REGS; 
	passRegs->PDSP_CTLSTAT[pdspNum].PDSP_CONTROL &= ~(CSL_PA_SS_PDSP_CONTROL_PDSP_ENABLE_MASK);

}
