/**  
 * @file cpsw_singlecore.c
 *
 * @brief 
 *  Example to illustrate the usage of EMAC CPSW3G switch using CPPI, QMSS
 * 	low level drivers and CSL.
 *
 * 	This example application does the following:
 * 	    (1) Initializes:
 * 	            (a) Queue Manager (QM) Subsystem 
 * 	            (b) Packet Accelerator (PA) CPPI DMA 
 * 	            (c) Ethernet Subsystem (Ethernet switch + SGMII + MDIO)
 * 	            (d) PA Subsystem + PDSP
 *
 * 	    (2) Sets up the CPPI descriptors and Queues required for sending and
 * 	        receiving data using Ethernet.
 * 	            (a) Uses Host descriptors
 * 	            (b) Uses High Priority Accumulation interrupts
 *
 * 	    (3) Sets up the example application's configuration (MAC address
 * 	        it uses to send/recv data; IP address and port number it's listening
 * 	        on) in PA Subsystem so as to enable the PASS to forward all packets
 * 	        matching this configuration onto the application for processing.
 * 	            (a) Switch MAC address configured   =   0x10:0x11:0x12:0x13:0x14:0x15
 * 	            (b) Example's IP address            =   192.168.1.10
 * 	            (c) Example App's listening port    =   0x5678
 *
 * 	    (4) Sends packets onto wire 
 * 	        (constructed manually in code here with following settings):
 * 	            (a) Source MAC      =   0x00:0x01:0x02:0x03:0x04:0x05
 * 	                Destination MAC =   0x10:0x11:0x12:0x13:0x14:0x15
 *              (b) Source IP       =   192.168.1.1
 *                  Destination IP  =   192.168.1.10
 *              (c) Source Port     =   0x1234
 *                  Destination Port=   0x5678
 *              (d) Payload Data (80 bytes)
 *
 *          The packets sent by the application are sent onto wire and 
 *          since the destination MAC on the packet is the Ethernet Switch 
 *          MAC address, the packets are received by simulator and passed 
 *          back up to the example application for processing.
 *      
 *      (5) Application receives all packets using QM High priority interrupt
 *          registered; Validates received packet against data sent.
 *
 *  Example application Setup:
 *
 *          PC Running Simulator using CCS connected to a
 *          Switch/Hub. You could put another PC on the Hub to observe packets 
 *          being sent onto wire. 
 *
 *          Please consult the Readme.txt packaged with the example to 
 *          setup the CCS simulator configuration required to run this example 
 *          succesfully.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <cpsw_singlecore.h>
#include <stdio.h>
#include <ti/csl/cslr_device.h>
#include <ti/csl/csl_psc.h>
#include <ti/csl/csl_pscAux.h>


/**************************************************************
************************** DEFINITIONS ************************
***************************************************************/
/* Number of packets to be used for testing the example. */
#define                     MAX_NUM_PACKETS                         10u

/* Counters to track number of packets sent/received by this application */
extern volatile UInt32				gRxCounter, gTxCounter;

/*
 * Default test configuration for the silicon
 *
 * To run test at the CCS simulator
 *    cpswSimTest = 1
 *    cpswLpbkMode = CPSW_LOOPBACK_EXTERNAL
 */
#ifdef  SIMULATOR_SUPPORT
Int cpswSimTest = 1;
Int cpswLpbkMode = CPSW_LOOPBACK_EXTERNAL;
#else
Int cpswSimTest = 0;
//Int cpswLpbkMode = CPSW_LOOPBACK_INTERNAL;
Int cpswLpbkMode = CPSW_LOOPBACK_EXTERNAL;
#endif
//Int cpswEvm6678 = 0;
Int cpswEvm6678 = 1;

void mdebugHaltPdsp (Int pdspNum);
volatile Int mdebugWait = 0;

/**************************************************************
**************** EXAMPLE APP FUNCTIONS ************************
***************************************************************/

/** ============================================================================
 *   @n@b Cpsw_SingleCoreApp
 *
 *   @b Description
 *   @n Example application that sets up the application, sends, receives
 *      data.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return
 *   @n None
 *
 * =============================================================================
 */
Void Cpsw_SingleCoreApp (Void)
{
	extern void view_ale_table(void);
	Int32			i;
	int				ct_show_ale = 0;

    System_printf ("**************************************************\n");
    System_printf ("******* Ethernet Single Core Example Start *******\n");
    System_printf ("**************************************************\n");

    /* Init internal cycle counter */
    TSCL = 1;  

    /* Initialize the components required to run the example:
     *  (1) QMSS
     *  (2) CPPI
     *  (3) Ethernet switch subsystem + MDIO + SGMII
     */
    /* Initialize QMSS */
    if (Init_Qmss () != 0)
    {
        System_printf ("QMSS init failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("QMSS successfully initialized \n");            
    }

    /* Initialize CPPI */
    if (Init_Cppi () != 0)
    {
        System_printf ("CPPI init failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("CPPI successfully initialized \n");            
    }

    /* Init PA LLD */
    if (Init_PASS () != 0)
    {
        System_printf ("PASS init failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("PASS successfully initialized \n");            
    }

    /* Initialize the CPSW switch */
    if (Init_Cpsw () != 0)
    {
        System_printf ("Ethernet subsystem init failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("Ethernet subsystem successfully initialized \n");            
    }

    /* Setup Tx */
    if (Setup_Tx () != 0)
    {
        System_printf ("Tx setup failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("Tx setup successfully done \n");            
    }



    /* Setup PA */
    if (Setup_PASS () != 0)
    {
        System_printf ("PASS setup failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("PASS setup successfully done \n");            
    }
    
    if (!cpswSimTest)
    {
        System_printf("Following is the ALE table before transmits.\n");
        view_ale_table();	// Added by Atsushi
    }
    
    /* Run some data through and verify transfer worked */
    //mdebugHaltPdsp(0);
    System_printf ("Packet Transmission Start ... \n");
    for (i = 0; i < MAX_NUM_PACKETS; i ++)
    {
        if (SendPacket () != 0)
        {
            System_printf ("Packet %d send failed \n", i);
            BIOS_exit (-1);
        }
    }
    
    //mdebugWait = 1;     
		
	//while (mdebugWait);
    
    
    if (!cpswSimTest)
    {
	    System_printf("Following is the ALE table after transmits.\n");
        view_ale_table();	// Added by Atsushi
    }
    
    /* Wait until all packet reception is done */
   	System_printf ("Packet Transmission Done.\nWait for all packets to be Received ... \n");

    while (gRxCounter != gTxCounter) {
    	System_printf("Received %d packets so far...\n", gRxCounter);
        CycleDelay (10000);
        
        if (!cpswSimTest)
        {
    	    if (++ ct_show_ale >= 10) {
    		    view_ale_table();
    		    ct_show_ale = 0;
    	    }
        }
    }
    
    System_printf ("Packets Sent\t\t=\t%d \nPackets Received\t=\t%d \nExample Done! \n", gTxCounter,  gRxCounter);
  
    System_printf ("**************************************************\n");
    System_printf ("******** Ethernet Single Core Example End ********\n");
    System_printf ("**************************************************\n");

    /* Example application done. Return success */
    BIOS_exit (0);
    
}

/***************************************************************************************
 * FUNCTION PURPOSE: Power up PA subsystem
 ***************************************************************************************
 * DESCRIPTION: this function powers up the PA subsystem domains
 ***************************************************************************************/
void passPowerUp (void)
{

    /* PASS power domain is turned OFF by default. It needs to be turned on before doing any 
     * PASS device register access. This not required for the simulator. */

    /* Set PASS Power domain to ON */        
    CSL_PSC_enablePowerDomain (CSL_PSC_PD_PASS);

    /* Enable the clocks for PASS modules */
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_PKTPROC, PSC_MODSTATE_ENABLE);
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_CPGMAC,  PSC_MODSTATE_ENABLE);
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_Crypto,  PSC_MODSTATE_ENABLE);

    /* Start the state transition */
    CSL_PSC_startStateTransition (CSL_PSC_PD_PASS);

    /* Wait until the state transition process is completed. */
    while (!CSL_PSC_isStateTransitionDone (CSL_PSC_PD_PASS));
}

/** ============================================================================
 *   @n@b main
 *
 *   @b Description
 *   @n Entry point for single core example application.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return
 *   @n None
 * =============================================================================
 */
void InitMainPLL();
Int32 main (Void)
{
    Task_Params                	cpswTaskParams;
    
    //InitMainPLL();
    if (!cpswSimTest)
    {
      passPowerUp();
    }
    
    /* Initialize the task params */
    Task_Params_init(&cpswTaskParams);

    /* Create the CPSW single core example task */
    Task_create((Task_FuncPtr)&Cpsw_SingleCoreApp, &cpswTaskParams, NULL);

    /* Start the BIOS Task scheduler */
	BIOS_start ();

	return 0;	
}
