#ifndef SYSREG_H
#define SYSREG_H
/*=============================================================================*\
 * Module Name:     sysreg.h
 * Created By:      zzw378
 * Created Date:    Mon Sep 20 11:34:33 2010
 * %version:        tcb_css2#28 %
 * %cvtype:         incl %
 * %instance:       tcbtut64_1 %
 * %derived_by:     zxw378 %
 * %date_modified:  Thursday, March 08, 2012 9:17:06 AM %
\*=============================================================================*/
/*=============================================================================*\
 * Copyright 2008, Delphi Technologies, Inc., All Rights Reserved.
 * Delphi Confidential.
 *------------------------------------------------------------------------------
 *
 * Description:
 *   This module contains the implementation of the DBC7.8 Layer 2
 *   Hardware releated register definitions for SYSTEM Module.
 *
 * Traces to: TI System Module Document [SPNU480h.pdf]
 *
 *
 * Applicable Standards (in order of precedence: highest first):
 *     SW REF 264.15D "Delphi C Coding Standards" [12-Mar-2006]
 *
 * Deviations from Delphi C Coding standards:
 *   1. C46 - Abbreviations are not documented in this source file.
 *
 *   2. C58 - Function header blocks only list global variables that
 *      are not accessed by macros or functions.
 *
 *   3. C60 - The function is preemptible or reentrant is not
 *      applicable to this program.
 *
 *   4. N4 - All user type definitions shall be lower case, delimited
 *      with underscores, and end with _T suffix.
 *
\*=============================================================================*/
/******************************
* Includes
*******************************/

#include "systype.h"

/******************************
* Defines
*******************************/
//-------------------------------------------------
// FMzPLL Control Register bit definitions
//
//  PLLCTL1_BITS
#define PLLMUL_NF(val) ((((val >= 0x5C) && (val <= 0xB8))?(val-1):0x5BU) * 256U) /* Multiplier */
#define REFCLKDIV_NR(val) (((val-1) & 0x3FU) << 16)                              /* Prescale divider */
#define BPOS_ENABLE       (0x01U << 29)                                          /* Bypass on PLL Slip */
#define BPOS_DISABLE      (0x02U << 29)                                          /* Bypass on PLL Slip */
#define ROS_ENABLE        (0x01U << 31)                                          /* Reset on PLL Slip */
#define ROS_DISABLE       (0x00U << 31)                                          /* Reset on PLL Slip */
#define ROF_ENABLE        (0x01U << 23)                                          /* Reset on Oscillator Fail  */
#define ROF_DISABLE       (0x00U << 23)                                          /* Reset on Oscillator Fail  */
#define PLLDIV_OD_R(val)  (((val-1) & 0x1FU) << 24)                              /* Postscale divider */

//  PLLCTL2_BITS
#define SPR_AMOUNT(val)     (((val-1) & 0x1FFU) << 0)
#define ODPLL(val)          (((val-1) & 0x7U) << 9)                              /* Postscale divider */
#define BWADJ(val)          (((val-1) & 0x1FFU) << 12)
#define SPREADINGRATE(val)  (((val-1) & 0x1FFU) << 22)
#define FM_ENABLE           (0x01 << 31)
#define FM_DISABLE          (0x00 << 31)


/* maximum value of the RTI timer */
#define RTI_MODULUS             (0xFFFFFFFFU)


/* RTI bit fields */

#define RTI_TAP_ENA     0x0040U
#define RTI_TAP_FLAG    0x0080U
#define RTI_CMP1_FLAG   0x0080U
#define RTI_CMP2_FLAG   0x0040U
#define RTI_CMP1_ENA    0x0020U
#define RTI_CMP2_ENA    0x0010U
#define RTI_CNTEN       0x0000U

#define RTIM_CNTR_20    0x0000U
#define RTIM_CNTR_17    0x0800U
#define RTIM_CNTR_14    0x1000U
#define RTIM_CNTR_11    0x1800U
#define RTIM_CNTR_8     0x2000U
#define RTIM_CNTR_5     0x2800U
#define RTIM_CNTR_2     0x3000U
#define RTIM_CNTR_U     0x3800U
#define RTIM_CLEAR      0x07FFU

#define RTI_PRES_2      0x0001U
#define RTI_PRES_3      0x0002U
#define RTI_PRES_4      0x0003U
#define RTI_PRES_5      0x0004U

#define RCLKSRC1_MASK   (0x0000000FuL)
#define RCLKSRC1_SRC0   (0x00000000uL)
#define RCLKSRC1_SRC1   (0x00000001uL)
#define RCLKSRC1_SRC2   (0x00000002uL)
#define RCLKSRC1_SRC3   (0x00000003uL)
#define RCLKSRC1_SRC4   (0x00000004uL)
#define RCLKSRC1_SRC5   (0x00000005uL)
#define RCLKSRC1_SRC6   (0x00000006uL)
#define RCLKSRC1_SRC7   (0x00000007uL)
#define RCLKSRC1_SRC8   (0x00000008uL)
#define RCLKSRC1_VCLK   (0x00000009uL)

#define RCLKDIV1_MASK   (0x00000300uL)
#define RCLKDIV1_DIV1   (0x00000000uL)
#define RCLKDIV1_DIV2   (0x00000100uL)
#define RCLKDIV1_DIV4   (0x00000200uL)
#define RCLKDIV1_DIV8   (0x00000300uL)

#define RCLKSRC2_MASK   (0x00000F00uL)
#define RCLKSRC2_SRC1   (0x00000000uL)
#define RCLKSRC2_SRC2   (0x00000100uL)
#define RCLKSRC2_SRC3   (0x00000200uL)
#define RCLKSRC2_SRC4   (0x00000300uL)
#define RCLKSRC2_SRC5   (0x00000400uL)
#define RCLKSRC2_SRC6   (0x00000500uL)
#define RCLKSRC2_SRC7   (0x00000600uL)
#define RCLKSRC2_SRC8   (0x00000700uL)
#define RCLKSRC2_VCLK   (0x00090000uL)

#define RCLKDIV2_MASK   (0x03000000uL)
#define RCLKDIV2_DIV1   (0x00000000uL)
#define RCLKDIV2_DIV2   (0x01000000uL)
#define RCLKDIV2_DIV4   (0x02000000uL)
#define RCLKDIV2_DIV8   (0x03000000uL)

#define RTI_INT0        (0x00000001uL)
#define RTI_INT1        (0x00000002uL)
#define RTI_INT2        (0x00000004uL)
#define RTI_INT3        (0x00000008uL)

#define RTIGCTRL_COS    (0x00008000uL)          /* Counters running in debug mode. */
#define RTIGCTRL_NCOS   (0x00000000uL)          /* Counters stopped in debug mode. */
#define RTIGCTRL_CNT0EN (0x00000001uL)          /* Counter 0 enabled. */
#define RTIGCTRL_CNT0DI (0x00000000uL)          /* Counter 0 disabled. */
#define RTIGCTRL_CNT1EN (0x00000002uL)          /* Counter 1 enabled. */
#define RTIGCTRL_CNT1DI (0x00000000uL)          /* Counter 1 disabled. */

#define RTICOMP_0_FRC_0 (0x00000000uL)
#define RTICOMP_0_FRC_1 (0x00000001uL)

#define RTICOMP_1_FRC_0 (0x00000000uL)
#define RTICOMP_1_FRC_1 (0x00000010uL)

#define RTICOMP_2_FRC_0 (0x00000000uL)
#define RTICOMP_2_FRC_1 (0x00000100uL)

#define RTICOMP_3_FRC_0 (0x00000000uL)
#define RTICOMP_3_FRC_1 (0x00001000uL)

#define SYS_MSTDONE     (0x00000001uL)
#define SYS_MINIDONE    (0x00000100uL)

#define PBIST_FAIL_STATUS_MASK  (0x00000001uL)
#define PBIST_FAIL_COUNT_MASK   (0x000000FFuL)
#define PBIST_FAIL_ADDRESS_MASK   (0x0000FFFFuL)
#define PBIST_FAIL_RETURN_DATA_MASK (0xFFFFFFFFU)

#define CCM7_STC        (0x00000100uL)
#define CCM7_STE        (0x00000001uL)
#define CCM7_STET       (0x00000002uL)
#define CCM7_CMPE       (0x00010000uL)

#define CDIS_MASK       (0x0000000FuL)
#define CDIS_SRC0       (0x00000001uL)
#define CDIS_SRC1       (0x00000002uL)
#define CDIS_SRC2       (0x00000004uL)
#define CDIS_SRC3       (0x00000008uL)
#define CDIS_SRC4       (0x00000010uL)
#define CDIS_SRC5       (0x00000020uL)
#define CDIS_SRC6       (0x00000040uL)
#define CDIS_SRC7       (0x00000080uL)

#define GHVSRC_MASK     (0x0000000FuL)
#define GHVSRC_SRC0     (0x00000000uL)
#define GHVSRC_SRC1     (0x00000001uL)
#define GHVSRC_SRC2     (0x00000002uL)
#define GHVSRC_SRC3     (0x00000003uL)
#define GHVSRC_SRC4     (0x00000004uL)
#define GHVSRC_SRC5     (0x00000005uL)
#define GHVSRC_SRC6     (0x00000006uL)
#define GHVSRC_SRC7     (0x00000007uL)

#define CLOCK_SOURCE_OSC       GHVSRC_SRC0
#define CLOCK_SOURCE_FMZPLL    GHVSRC_SRC1
#define CLOCK_SOURCE_32KHZ     GHVSRC_SRC2
#define CLOCK_SOURCE_EXT_CLK   GHVSRC_SRC3
#define CLOCK_SOURCE_LPO_LOW   GHVSRC_SRC4
#define CLOCK_SOURCE_LPO_HIGH  GHVSRC_SRC5

/* GHVSRC Bit Definitions */
#define CLOCK_CURRENT_SOURCE(val)         ((val & 0x7U) << 0)
#define CLOCK_LOW_POWER_MODE_SOURCE(val)  ((val & 0x7U) << 16)
#define CLOCK_WAKEUP_SOURCE(val)          ((val & 0x7U) << 24)

#define PCR_ENABLE_ALL  (0xFFFFFFFFuL)
#define PCR_DISABLE_ALL (0x0uL)

#define VCLKR1_MASK     (0x000F0000uL)          /* VBUS source1 is is HCLK / this value. */
#define VCLKR1_DIV1     (0x00000000uL)
#define VCLKR1_DIV2     (0x00010000uL)
#define VCLKR1_DIV3     (0x00020000uL)
#define VCLKR1_DIV4     (0x00030000uL)
#define VCLKR1_DIV5     (0x00040000uL)
#define VCLKR1_DIV6     (0x00050000uL)
#define VCLKR1_DIV7     (0x00060000uL)
#define VCLKR1_DIV8     (0x00070000uL)

#define VCLKR2_MASK     (0x0F000000uL)          /* VBUS source2 is is HCLK / this value. */
#define VCLKR2_DIV1     (0x00000000uL)
#define VCLKR2_DIV2     (0x01000000uL)
#define VCLKR2_DIV3     (0x02000000uL)
#define VCLKR2_DIV4     (0x03000000uL)
#define VCLKR2_DIV5     (0x04000000uL)
#define VCLKR2_DIV6     (0x05000000uL)
#define VCLKR2_DIV7     (0x06000000uL)
#define VCLKR2_DIV8     (0x07000000uL)

#define PERIPHERAL_CLOCK_DIVIDE_RATIO(val)  ((val & 0xFU) << 16)
#define HET_CLOCK_DIVIDE_RATIO(val)         ((val & 0xFU) << 24)

#define PERIPHERAL_SET0  0
#define PERIPHERAL_SET1  1
#define PERIPHERAL_SET2  2
#define PERIPHERAL_SET3  3

#define PENABLE         (0x00000100uL)          /* Peripheral enable flag. */

#define VCLKASRC1_MASK  (0x0000000FuL)
#define VCLKASRC1_SRC0  (0x00000000uL)
#define VCLKASRC1_SRC1  (0x00000001uL)
#define VCLKASRC1_SRC2  (0x00000002uL)
#define VCLKASRC1_SRC3  (0x00000003uL)
#define VCLKASRC1_SRC4  (0x00000004uL)
#define VCLKASRC1_SRC5  (0x00000005uL)
#define VCLKASRC1_SRC6  (0x00000006uL)
#define VCLKASRC1_SRC7  (0x00000007uL)
#define VCLKASRC1_VCLK  (0x00000009uL)

#define VCLKASRC2_MASK  (0x00000F00uL)
#define VCLKASRC2_SRC0  (0x00000000uL)
#define VCLKASRC2_SRC1  (0x00000100uL)
#define VCLKASRC2_SRC2  (0x00000200uL)
#define VCLKASRC2_SRC3  (0x00000300uL)
#define VCLKASRC2_SRC4  (0x00000400uL)
#define VCLKASRC2_SRC5  (0x00000500uL)
#define VCLKASRC2_SRC6  (0x00000600uL)
#define VCLKASRC2_SRC7  (0x00000700uL)
#define VCLKASRC2_SRC8  (0x00000800uL)
#define VCLKASRC2_VCLK  (0x00000900uL)

/* VCLKASRC Bit Definitions */
#define PERIPHERAL_ASYNCHRONOUS_CLOCK_1_SOURCE(val)         ((val & 0x7U) << 0)
#define PERIPHERAL_ASYNCHRONOUS_CLOCK_2_SOURCE(val)         ((val & 0x7U) << 8)

#define DMA_PORTB       (0x7U)
#define DMA_PORT_ASSIGNMENT_REG_0   0   /* Name of DMA PAR0 */
#define DMA_PORT_ASSIGNMENT_REG_1   1   /* Name of DMA PAR1 */
#define DMA_PORT_ASSIGNMENT_REG_2   2   /* Name of DMA PAR2 */
#define DMA_PORT_ASSIGNMENT_REG_3   3   /* Name of DMA PAR3 */

#define DMA_RES_MASK    (0x1U << 0)
#define DMA_EN_MASK     (0x1U << 16)

/* DMA CHCTRL RAM */
#define DMA_READ_ELEMENT_SIZE_08_BIT  (0x0U << 14)
#define DMA_READ_ELEMENT_SIZE_16_BIT  (0x1U << 14)
#define DMA_READ_ELEMENT_SIZE_32_BIT  (0x2U << 14)
#define DMA_READ_ELEMENT_SIZE_64_BIT  (0x3U << 14)

#define DMA_WRITE_ELEMENT_SIZE_08_BIT  (0x0U << 12)
#define DMA_WRITE_ELEMENT_SIZE_16_BIT  (0x1U << 12)
#define DMA_WRITE_ELEMENT_SIZE_32_BIT  (0x2U << 12)
#define DMA_WRITE_ELEMENT_SIZE_64_BIT  (0x3U << 12)

#define DMA_TRANSFER_TYPE_FRAME        (0x0U << 8)
#define DMA_TRANSFER_TYPE_BLOCK        (0x1U << 8)

#define DMA_ADDRESS_MODE_READ_TYPE_CONSTANT  (0x0U << 3)
#define DMA_ADDRESS_MODE_READ_TYPE_POST_INC  (0x1U << 3)
#define DMA_ADDRESS_MODE_READ_TYPE_INDEXED   (0x3U << 3)

#define DMA_ADDRESS_MODE_WRITE_TYPE_CONSTANT  (0x0U << 1)
#define DMA_ADDRESS_MODE_WRITE_TYPE_POST_INC  (0x1U << 1)
#define DMA_ADDRESS_MODE_WRITE_TYPE_INDEXED   (0x3U << 1)

#define DMA_AUTO_INITIATION_MODE     (0x1U << 0)
#define DMA_NEXT_CHANNEL_TO_BE_TRIGGERRED     (0x000000U << 16)

#define DMA_COUNT_FIELD_MAX_VALUE   (0x1FFFU)  /* 13-Bit Limit for FRAME and ELEMENT Count Registers */
#define DMA_PARITY_ADDRESS  ((unsigned16_T)DMA_Read_Bits(PAR,ERRORADDRESS))
/* Bit 11 - Bit 0 are valid, all the other bits of DMAPAR register are reserved */
#define DMAPAR_MASK           0x0FFF

/* Bit 15 - Bit 13 are valid, all the other bits of ABRTESR register are reserved */
#define ABRTESR_MASK          0x0000
/* Bit 7 to Bit 4 and Bit 1 to Bit 0 are valid, all the other bits of GLBSTAT
   register are reserved */
#define GLBSTAT_MASK          0x0301U
/* Bit 15 to Bit 7 are valid, all the other bits of SYSESR register are reserved */
#define SYSESR_MASK           0xE038U

#define IMPFASTS_MASK         (0x00FF0701U)

/* SYSESR register Mask */
#define SYSESR_PORRST_MASK                  (0x1U << 15)
#define SYSESR_OSCRST_MASK                  (0x1U << 14)
#define SYSESR_WDRST_MASK                   (0x1U << 13)
#define SYSESR_CPURST_MASK                  (0x1U << 5)
#define SYSESR_SWRST_MASK                   (0x1U << 4)
#define SYSESR_EXTRST_MASK                  (0x1U << 3)


/* GLBSTAT register Mask */
#define GLBSTAT_FBSLIP_MASK                 (0x1U << 9)
#define GLBSTAT_RFSLIP_MASK                 (0x1U << 8)
#define GLBSTAT_OSCFAIL_MASK                (0x1U << 0)

/* ESM Group 1 Channel Masks */
#define ESM_GROUP_1_CCM7_SELFTEST           (0x1U << 31)

/* ESM Group 2 Channel Masks */
#define ESM_GROUP_2_CCM7_COMPARE_FAIL       (0x1U << 2)

/*to enable esm check for init memrory test*/
#define ESMSR1_AVAILABLE


/* VIM Register Masks */
#define ADDERR_MASK           0x1FC
#define ADDERR_OFFSET         2

/* RAMECCREG MASKS */
#define ERRTYPE_OFFSET        8
/* RAM Register Masks */
#define RAMDERRADDR_MASK      (0x7FFF)

/*VIM_PARCTL*/
#define VIM_PAR_EN          (0x0A << 0)
#define VIM_TEST_EN         (0x0 << 8)

/* RAMCTRL */
#define RAM_ECC_INIT        (0x05 << 16)
#define RAM_ECC_ENABLE      (0x0A << 0)
#define RAM_ECC_WRT_ENA     (0x0 << 8)
#define RAM_RMWCBYP         (0x0A << 9)
#define RAMCTRL_ECC_WRT_ENA (0x1 << 8)
#define RAMCTRL_ECC_WRT_DIS (0x0 << 8)

/*RAMTHRESHOLD*/
#define RAM_RAMTHRESHOLD     (0x01 <<0)
#define RAM_ERROR_STATUS_FLAG     (0x00000001)

/*RAMINTCTRL*/
#define RAM_SECINTEN         (0x1 << 0)

/* RAMERRSTATUS */
#define RAM_SEC_DED          (0x1U << 5)
/*ESMLTCPR*/
#define ESM_TLCP             (0x03 << 14)

/*ESMEKR*/
#define ESM_EKEY_NORMAL             (0x00 << 0)
#define ESM_EKEY_ERROR_FORCE        (0x0A << 0)
#define ESM_EKEY_ERROR_RESET        (0x05 << 0)

#define ESMEKR_ENABLE_ERROR_FORCING_MODE    (0x0AU)
#define ESMEKR_ENABLE_NORMAL_MODE           (0x00U)
#define ESM_ERROR_PIN_MASK                  (0x01U)

/* ESM Channel Assignments */
#define ESM_GROUP_1_ERROR_DMA_MPU_MASK                                                     (0x01U << 2)
#define ESM_GROUP_1_ERROR_DMA_PARITY_MASK                                                  (0x01U << 3)
#define ESM_GROUP_1_ERROR_DMA_PARITY_SYSTEM_DMA_TRANSACTIONS_NCNB_USER_DATA_ACCESS_MASK    (0x01U << 5)
#define ESM_GROUP_1_ERROR_NHET_PARITY_MASK                                                 (0x01U << 7)
#define ESM_GROUP_1_ERROR_NHET_TU_PARITY_MASK                                              (0x01U << 8)
#define ESM_GROUP_1_ERROR_NHET_TU_MPU_MASK                                                 (0x01U << 9)
#define ESM_GROUP_1_ERROR_PLL_SLIP_MASK                                                    (0x01U << 10)
#define ESM_GROUP_1_ERROR_CLOCK_MONITOR_INTERRUPT_MASK                                     (0x01U << 11)
#define ESM_GROUP_1_ERROR_SYSTEM_ILLEGAL_TRANSACTIONS_OK_RESPONSE_VBUS_WRITES_MASK         (0x01U << 13)
#define ESM_GROUP_1_ERROR_VIM_RAM_PARITY_MASK                                              (0x01U << 15)
#define ESM_GROUP_1_ERROR_MIBSPI_PARITY_MASK                                               (0x01U << 17)
#define ESM_GROUP_1_ERROR_MIBADC_PARITY_MASK                                               (0x01U << 19)
#define ESM_GROUP_1_ERROR_DCAN1_PARITY_MASK                                                (0x01U << 21)
#define ESM_GROUP_1_ERROR_DCAN2_PARITY_MASK                                                (0x01U << 23)
#define ESM_GROUP_1_ERROR_RAM_CORRECTABLE_ERROR_MASK                                       (0x01U << 26)
#define ESM_GROUP_1_ERROR_SYSTEM_PRECISE_CPU_TRANSACTIONS_NCNB_CPU_ACCESS_MASK             (0x01U << 30)
#define ESM_GROUP_1_ERROR_CCM7_SELFTEST_MASK                                               (0x01U << 31)

#define ESM_GROUP_2_ERROR_CCM7_COMPARE_FAIL_MASK                          (0x01U << 2)
#define ESM_GROUP_2_ERROR_FLASH_UNCORRECTABLE_ERROR_MASK                  (0x01U << 4)
#define ESM_GROUP_2_ERROR_RAM_UNCORRECTABLE_ERROR_MASK                    (0x01U << 6)
#define ESM_GROUP_2_ERROR_RAM_ADDRESS_WRITE_DATA_BUS_PARITY_ERROR_MASK    (0x01U << 10)
#define ESM_GROUP_2_ERROR_CPU_READ_DATA_BUS_PARITY_ERROR_MASK             (0x01U << 11)


/*RTCTRL*/
#define DMA_RTC             (0x0 << 0)

/* DCTRL */
#define DMA_DEBUG_ENABLE    (0x0 << 0)

/*DMAMPCTRL */
#define DMA_PARITY          (0x0A << 0)
#define DMA_TEST_EN         (0x0 << 8)
#define DMA_ERROR_ACTION    (0x1 << 16)

/*MPCTRL*/
#define DMA_INT0AB          (0x0 << 4)
#define DMA_INT0ENA         (0x1 << 3)
#define DMA_REG0AP          (0x01<< 1)
#define DMA_REG0ENA         (0x1 << 0)
#define DMA_INT1AB          (0x0 << 12)
#define DMA_INT1ENA         (0x0 << 11)
#define DMA_REG1AP          (0x00<< 9)
#define DMA_REG1ENA         (0x0 << 8)
#define DMA_INT2AB          (0x0 << 20)
#define DMA_INT2ENA         (0x0 << 19)
#define DMA_REG2AP          (0x00<< 17)
#define DMA_REG2ENA         (0x0 << 16)
#define DMA_INT3AB          (0x0 << 28)
#define DMA_INT3ENA         (0x0 << 27)
#define DMA_REG3AP          (0x00<< 25)
#define DMA_REG3ENA         (0x0 << 24)

#define MEM_HARDWARE_INIT                (0x0A << 0)
#define DIS_MEM_HARDWARE_INIT            (0x05 << 0)
#define RTI0UC                           (0x00 << 0)
#define ENABLE_ALL_RAM                   (0x000001F7UL)
#define ENABLE_MIBSPI_RAM                  (0x00000008UL)
//#define SYS_MINIDONE                     (0x00000100UL)
#define DMA_MP_REGION0_START_ADDRESS     (0x00000000UL)
#define DMA_MP_REGION0_END_ADDRESS       (0x000BFFFFUL)
#define DMA_MP_REGION1_START_ADDRESS     (0x00000000UL)
#define DMA_MP_REGION1_END_ADDRESS       (0x00000000UL)
#define DMA_MP_REGION2_START_ADDRESS     (0x00000000UL)
#define DMA_MP_REGION2_END_ADDRESS       (0x00000000UL)
#define DMA_MP_REGION3_START_ADDRESS     (0x00000000UL)
#define DMA_MP_REGION3_END_ADDRESS       (0x00000000UL)
/***************************************************************/
#define LONGREG(reg)    ((*(volatile unsigned long *) (reg)))

#define SSIR           LONGREG(0xFFFFFFFCUL)

#define Check_SYSESR_Bit(member) \
              (SYS.SYSESR.bits.member)

#define Check_GLBSTAT_Bit(member) \
              (SYS.GLBSTAT.bits.member)

#define Check_ABRTESR_Bit(member) \
               (SYS.ABRTESR.bits.member)
/******************************
* Macros
*******************************/
#define RAM_Read_Word(reg) (RAMECCREG.##reg##.word )
#define VIM_Write_Word_Array(reg, data, index ) \
do \
{ \
    VIM.##reg##[(index)].word = (data) ; \
}while(0)

#define VIM_PAR_Read_Word(reg)     (VIM_PAR.##reg##.word )

#define VIM_PAR_Write_Word(reg, data) \
do \
{ \
    VIM_PAR.##reg##.word = (data) ; \
}while(0)

#define SRAM_Write_Word(reg, data ) \
do \
{ \
    RAMECCREG.##reg##.word = (data); \
}while(0)

#define Set_VIM_PAR_Bits(reg, bit_mask ) \
do \
{ \
    VIM_PAR.##reg##.word |= bit_mask ; \
}while(0)

#define Reset_VIM_PAR_Bits(reg, bit_mask ) \
do \
{ \
    VIM_PAR.##reg##.word &= ~bit_mask ; \
}while(0)

#define Set_SRAM_Bits(reg, bit_mask ) \
do \
{ \
   RAMECCREG.##reg##.word |= bit_mask ; \
}while(0)
#define SRAM_Read_Word(reg) (RAMECCREG.##reg##.word )

#define SYS_Write_Word(reg, data ) \
do \
{ \
    SYS.##reg##.word = (data); \
}while(0)

#define Set_SYS_Bits(reg, bit_mask ) \
do \
{ \
    SYS.##reg##.word |= bit_mask ; \
}while(0)

#define Reset_SYS_Bits(reg, bit_mask ) \
do \
{ \
    SYS.##reg##.word &= ~bit_mask ; \
}while(0)

#define SYS_Read_Word(reg) (SYS.##reg##.word )

#define SYS_Read_Bit(reg, bit_mask) ( ( SYS.##reg##.word ) & ( bit_mask ) )

#define SYS_Read_BitName(reg, bit_name) ( ( SYS.##reg##.##bit_name## ) )

/* System MPU Module access macros */
#define SYS_MPU_Write_Word(reg, data ) \
do \
{ \
    SYS_MPU.##reg##.word = data ; \
}while(0)

#define Set_SYS_MPU_Bits(reg, bit_mask ) \
do \
{ \
    SYS_MPU.##reg##.word |= bit_mask ; \
}while(0)

#define Reset_SYS_MPU_Bits(reg, bit_mask ) \
do \
{ \
    SYS_MPU.##reg##.word &= ~bit_mask ; \
}while(0)

#define SYS_MPU_Read_Word(reg) (SYS_MPU.##reg##.word )

#define SYS_MPU_Read_Bit(reg, bit_mask) ( ( SYS_MPU.##reg##.word ) & ( bit_mask ) )

#define Clear_System_Software_Interrupt_Request_Flag(ssir_vector_status)  ((ssir_vector_status) = SYS.SSIVEC.word)

#define Is_Previous_SSIR_Request_Status_Incomplete()  (0)

#define Setup_HETRAM_Wait_State(wait_state)

#define Setup_VIMRAM_Interrupt_Priority(data, element)  VIM_Write_Word_Array(VIM_CHANCTRL, (data), (element))

#define GET_RTI_FREE_RUNNING_COUNTER( num )    RTI_Read_Word_Index( RTICNTSET, RTIFRC, num )


/* System VIM Module access macros */
#define VIM_Write_Word_Index(reg, data, index ) \
do \
{ \
    VIM.##reg##.word[(index)] = (data) ; \
}while(0)

#define Disable_All_Peripheral_Interrupts()  \
 \
VIM_Write_Word_Index(VIM_REQMSKCLR, 0xFFFFFFFFU, 0); \
VIM_Write_Word_Index(VIM_REQMSKCLR, 0xFFFFFFFFU, 1)

#define VIM_Read_Word(reg) (VIM_PAR.VIM_##reg##.word )
#define Enable_All_Category_2_Peripheral_Interrupts(os_cat2_mask)  \
 \
VIM_Write_Word_Index(VIM_REQMSKSET, ((os_cat2_mask) & 0xFFFFFFFFU), 0); \
VIM_Write_Word_Index(VIM_REQMSKSET, ((os_cat2_mask)>>32), 1)

#define SELECT_FIQ_IRQ_MODE_FOR_ALL_INTERRUPT_CHANNELS(os_fiq_irq_mask) \
{ \
  VIM_Write_Word_Index(VIM_FIRQPR, ((~(os_fiq_irq_mask)) & 0xFFFFFFFFU), 0); \
  VIM_Write_Word_Index(VIM_FIRQPR, ((~(os_fiq_irq_mask))>>32), 1); \
}

#define GET_ADDRESS_PARITY_ERROR()                             (GET_ADDRESS_PARITY_ERROR_WITH_OFFSET() >> ADDERR_OFFSET)
#define GET_ADDRESS_PARITY_ERROR_WITH_OFFSET()                 (VIM_Read_Word(ADDERR)& ADDERR_MASK)

/* System RAM Module access macros */
#define GET_RAM_DOUBLE_ERROR_ADDRESS()                 (RAM_Read_Word(RAMDERRADDR)& RAMDERRADDR_MASK)

#define ENABLE_IRQ_INTERRUPT()    os_clear_i_bit()
#define ENABLE_FIQ_INTERRUPT()    os_enable_fiq()
#define DISABLE_FIQ_INTERRUPT()   os_disable_fiq()
#define DISABLE_IRQ_INTERRUPT()   os_set_i_bit()

#define SET_CIM_PERIPHERAL_INTERRUPT_MASK_REGISTER(mask)
#define SETUP_CIM_PERIPHERAL_INTERRUPT_MASK_REGISTER(mask)
#define CLEAR_ALL_INTERRUPT_CONFIGURATION() \
{ \
  VIM_Write_Word_Index(VIM_FIRQPR, 0, 0); \
  VIM_Write_Word_Index(VIM_FIRQPR, 0, 1); \
}

#define ENABLE_RTI_CLK_SOURCE(val)    SYS_Write_Word(RCLKSRC, (val))

#define RTI_Write_Word(reg, data ) \
do \
{ \
    RTI.##reg##.word = (data); \
}while(0)

#define RTI_Read_Word(reg) (RTI.##reg##.word )

#define RTI_Read_Word_Index( reg, member, index )        ( RTI.##reg##[(index)].member.word )


/* System RTI Module access macros */
#define RTI_Write_Word_Index(reg, member, data, index ) \
do \
{ \
    RTI.##reg##[(index)].member.word = (data) ; \
}while(0)


#define DISABLE_RTI_PERIPHERAL_INTERRUPTS(val)  RTI_Write_Word(RTICLEARINT, (val))
#define ENABLE_RTI_PERIPHERAL_INTERRUPTS(val)  RTI_Write_Word(RTISETINT, (val))
#define CLEAR_RTI_PERIPHERAL_INTERRUPT_FLAGS(val) RTI_Write_Word(RTIINTFLAG, (val))
#define GET_RTI_PERIPHERAL_INTERRUPT_FLAGS(val) (RTI_Read_Word(RTIINTFLAG) & (val))
#define Boot_Timer_Get()        (GET_RTI_PERIPHERAL_INTERRUPT_FLAGS(RTI_INT0))
#define EOL_Timer_Get()        (GET_RTI_PERIPHERAL_INTERRUPT_FLAGS(RTI_INT1))

#define DISABLE_RTI_COUNTER_USED_AS_RTOS_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_CNT0DI)
#define DISABLE_RTI_COUNTER_USED_AS_SECONDARY_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_CNT1DI)

#define INITIALIZE_RTI_COUNTER_USED_AS_PBIST_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIFRC, 0, 0)
#define INITIALIZE_RTI_COUNTER_USED_AS_RTOS_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIFRC, 0, 0)
#define INITIALIZE_RTI_COUNTER_USED_AS_ECC_RAM_TEST_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIFRC, 0, 0)

#define INITIALIZE_RTI_COUNTER_USED_AS_SECONDARY_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIFRC, 0, 1)
#define INITIALIZE_RTI_PRESCALE_COUNTER_USED_AS_SECONDARY_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 1)
#define SETUP_RTI_PRESCALE_USED_AS_SECONDARY_TICK(val) RTI_Write_Word_Index(RTICNTSET, RTICPUC, (val), 1)
#define SETUP_RTI_COMPARE_COUNTER_USED_AS_SECONDARY_TICK_FOR_NEXT_PERIOD() RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, TIMER_2P5_MILLISECOND, 1)
#define SETUP_PERIOD_FOR_RTI_COUNTER_USED_AS_SECONDARY_TICK()  RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, TIMER_2P5_MILLISECOND, 1)
#define ENABLE_RTI_COUNTER_USED_AS_SECONDARY_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_NCOS | RTIGCTRL_CNT0EN | RTIGCTRL_CNT1EN)

#define SELECT_RTI_COMPARE_FREE_RUNNING_COUNTER_CHANNEL_USED_FOR_SECONDARY_TICK()  RTI_Write_Word(RTICOMPCTRL, RTICOMP_1_FRC_1)

#define INITIALIZE_RTI_PRESCALE_COUNTER_USED_AS_RTOS_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 0)
#define INITIALIZE_RTI_PRESCALE_COUNTER_USED_AS_PBIST_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 0)
#define INITIALIZE_RTI_PRESCALE_COUNTER_USED_AS_ECC_RAM_TEST_TICK()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 0)

#define INITIALIZE_RTI_UP_COUNTER1()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 1)
#define INITIALIZE_RTI_UP_COUNTER0()  RTI_Write_Word_Index(RTICNTSET, RTIUC, 0, 0)
#define SETUP_RTI_PRESCALE_USED_AS_RTOS_TICK(val) RTI_Write_Word_Index(RTICNTSET, RTICPUC, (val), 0)
#define INITIALIZE_RTI_PRESCALE_USED_FOR_PBIST() RTI_Write_Word_Index(RTICNTSET, RTICPUC, 0, 0)
#define SETUP_RTI_PRESCALE_USED_AS_ECC_RAM_TEST_TICK(val) RTI_Write_Word_Index(RTICNTSET, RTICPUC, (val), 0)
#define ENABLE_RTI_COUNTER_USED_AS_RTOS_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_NCOS | RTIGCTRL_CNT0EN)
#define SETUP_RTI_COMPARE_COUNTER_USED_AS_RTOS_TICK_FOR_NEXT_PERIOD() RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, TIMER_MILLISECOND, 0)

#define SETUP_RTI_COMPARE_COUNTER_USED_AS_PBIST_TICK() RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, PBIST_MAX_CLOCK_CYCLES, 0)
#define ENABLE_RTI_COUNTER_USED_AS_PBIST_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_NCOS | RTIGCTRL_CNT0EN)

#define ENABLE_RTI_COUNTER_USED_AS_ECC_RAM_TEST_TICK()  RTI_Write_Word(RTIGCTRL, RTIGCTRL_NCOS | RTIGCTRL_CNT0EN)
#define SETUP_RTI_COMPARE_COUNTER_USED_AS_ECC_RAM_TEST_TICK() RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, ECC_RAM_MAX_CLOCK_CYCLES, 0)

#define SETUP_PERIOD_FOR_RTI_COUNTER_USED_AS_RTOS_TICK()  RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, TIMER_MILLISECOND, 0)

#define Next_RTI_Prescale_ENUM(val)                   (val+1)
#define CONVERT_RTI_PRESCALE_FACTOR_TO_PHYSICAL(val)  (Next_RTI_Prescale_ENUM(val))

#define SHUTDOWN_RTI_TIMER(counter_num)  \
do { \
  RTI_Write_Word(RTIGCTRL, (0x1 << (counter_num)) ); \
} while(0)

#define SELECT_MBIST_RAM_GROUP_TO_BE_INIT(val)  SYS_Write_Word(MSINENA, (val))
#define INITIALIZE_RTI_CAPTURE_CONTROL()   RTI_Write_Word(RTICAPCTRL,0)
#define INITIALIZE_RTI_COMPARE_CONTROL()   RTI_Write_Word(RTICOMPCTRL,0)
#define INITIALIZE_RTI_FREE_RUNNING_COUNTER()  RTI_Write_Word_Index(RTICNTSET, RTIFRC, 0, 1)
#define INITIALIZE_RTI_COMPARE1()  RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, 0, 1)
#define INITIALIZE_RTI_COMPARE2()  RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, 0, 2)
#define INITIALIZE_RTI_COMPARE3()   RTI_Write_Word_Index(RTICNTCMPSET, RTICOMP, 0, 3)
#define INITIALIZE_RTI_UPDATE_COMPARE0()  RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, 0, 0)
#define INITIALIZE_RTI_UPDATE_COMPARE1()  RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, 0, 1)
#define INITIALIZE_RTI_UPDATE_COMPARE2()  RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, 0, 2)
#define INITIALIZE_RTI_UPDATE_COMPARE3()   RTI_Write_Word_Index(RTICNTCMPSET, RTIUDCP, 0, 3)


#define SELECT_MBIST_RAM_GROUP_TO_BE_INIT(val)  SYS_Write_Word(MSINENA, (val))
#define ENABLE_PBIST_CONTROLLER()  SYS_Write_Word(MSINENA, (0x1U))
#define ENABLE_PBIST_MEMORY_SELF_TEST_MODE() SYS_Write_Word(MSTGCR, (0x0A))
#define DISABLE_PBIST_MEMORY_SELF_TEST_MODE() SYS_Write_Word(MSTGCR, (0x05))

#define SELECT_HCLK_AS_ROM_CLOCK_SRC   (0<<8)
#define ENABLE_HCLK_AS_ROM_CLOCK_SRC_PBIST() Set_SYS_Bits(MSTGCR,SELECT_HCLK_AS_ROM_CLOCK_SRC)


#define CLEAR_MEMTEST_BIST_DONE_FLAG()  SYS_Write_Word(MSTCGSTAT, SYS_MSTDONE)
#define GET_MEMTEST_BIST_DONE_FLAG()  (SYS_Read_Word(MSTCGSTAT) & SYS_MSTDONE)
#define GET_MEMINIT_MBIST_DONE_FLAG()  (SYS_Read_Word(MSTCGSTAT) & SYS_MINIDONE)
#define GET_MEMORY_SELF_TEST_STATUS()  SYS_Read_Word(MSTFAIL)
#define GET_MEMORY_INIT_STATUS()  SYS_Read_Word(MINISTAT)
#define OVERWRITE_MEMORY_SELF_TEST_STATUS(val)  SYS_Write_Word(MSTFAIL, (val))
#define ENABLE_PBIST_MEMORY_INIT_MODE() SYS_Write_Word(MINITGCR, (0x0A))
#define DISABLE_PBIST_MEMORY_INIT_MODE() SYS_Write_Word(MINITGCR, (0x05))
#define ENABLE_HET_PARITY()   HET_Write_Word(HETPCR,0x0A)
#define SETUP_RTI_PRESCALE_USED_FOR_PBIST(val) RTI_Write_Word_Index(RTICNTSET, RTICPUC, (val), 0)

/* SYSESR_EXTRST_MASK always get set under any RESET condition */
#define IS_NOT_JTAG_DEBUGGER_OR_POR_OR_EXTRST_ACTIVE()       (SYS_Read_Word(SYSESR) & (SYSESR_OSCRST_MASK|SYSESR_CPURST_MASK|SYSESR_SWRST_MASK))
#define CHECK_IF_NOT_JTAG_DEBUGGER_OR_POR_OR_EXTRST_ACTIVE(register_value)       ((register_value) & (SYSESR_OSCRST_MASK|SYSESR_CPURST_MASK|SYSESR_SWRST_MASK))
#define CHECK_IF_NOT_POR_OR_EXTRST_OR_SWRST_ACTIVE(register_value)       ((register_value) & (SYSESR_WDRST_MASK|SYSESR_OSCRST_MASK|SYSESR_CPURST_MASK))

#define IS_JTAG_DEBUGGER_OR_POR_OR_EXTRST_ACTIVE()       (SYS_Read_Word(SYSESR) & (SYSESR_PORRST_MASK|SYSESR_WDRST_MASK|SYSESR_EXTRST_MASK))

#define IS_JTAG_DEBUGGER_ACTIVE()       IS_WATCH_DOG_RESET_EXCEPTION()

#define IS_POWER_ON_RESET_EXCEPTION()   (SYS_Read_Word(SYSESR) & SYSESR_PORRST_MASK)
#define IS_WATCH_DOG_RESET_EXCEPTION()  (SYS_Read_Word(SYSESR) & SYSESR_WDRST_MASK)
#define IS_SOFTWARE_RESET_EXCEPTION()   (SYS_Read_Word(SYSESR) & SYSESR_SWRST_MASK)
#define IS_EXTERNAL_RESET_EXCEPTION()   (SYS_Read_Word(SYSESR) & SYSESR_EXTRST_MASK)
#define IS_CPU_RESET_EXCEPTION()        (SYS_Read_Word(SYSESR) & SYSESR_CPURST_MASK)
#define IS_ADDRESS_GLOBAL_EXCEPTION()            (0) /*Check_GLBSTAT_Bit( SYSADDR )*/
#define IS_MPU_ADDRESS_GLOBAL_EXCEPTION()        (0) /*Check_GLBSTAT_Bit( MPUADDR ) */
#define IS_ILLEGAL_MAP_SYSTEM_EXCEPTION()        (0) /* Check_SYSESR_Bit(ILLMAP) */
#define IS_MPU_ACCESS_GLOBAL_EXCEPTION()         (0) /* Check_GLBSTAT_Bit( MPUACC ) */
#define IS_OSCILLATOR_FAIL_DETECTED_GLOBAL()     Check_GLBSTAT_Bit( OSCFAIL )
#define IS_OSCILLATOR_RESET_EXCEPTION()          Check_SYSESR_Bit( OSCRST )
#define IS_PERIPHERAL_ACCESS_SYSTEM_EXCEPTION()  (0) /* Check_SYSESR_Bit( PLLACC ) */
#define IS_PLL_OVERCYCLE_SLIP_DETECTED_GLOBAL()  Check_GLBSTAT_Bit( FBSLIP )
#define IS_PLL_UNDERCYCLE_SLIP_DETECTED_GLOBAL() Check_GLBSTAT_Bit( RFSLIP )
#define IS_SYSTEM_ACCESS_EXCEPTION_GLOBAL()      (0) /* Check_GLBSTAT_Bit( SYSACC ) */
#define IS_SYSTEM_ADDRESS_EXCEPTION_GLOBAL()     (0) /* Check_GLBSTAT_Bit( SYSADDR ) */
#define IS_SYSTEM_ILLEGAL_MODE_EXCEPTION()       (0) /* Check_SYSESR_Bit( ILLMODE ) */
#define IS_SYSTEM_ILLEGAL_ACCESS_EXCEPTION()     (0) /*  Check_SYSESR_Bit( ILLACC ) */
#define IS_SYSTEM_ILLEGAL_ADDRESS_EXCEPTION()    (0) /* Check_SYSESR_Bit( ILLADDR )  */

#define CLEAR_RESET_EXCEPTION_STATUS_REGISTER()   SYS_Write_Word(SYSESR, SYSESR_PORRST_MASK|SYSESR_OSCRST_MASK|SYSESR_WDRST_MASK|SYSESR_SWRST_MASK|SYSESR_EXTRST_MASK|SYSESR_CPURST_MASK)
#define CLEAR_GLOBAL_EXCEPTION_STATUS_REGISTER()  SYS_Write_Word(GLBSTAT, GLBSTAT_FBSLIP_MASK|GLBSTAT_RFSLIP_MASK|GLBSTAT_OSCFAIL_MASK)
#define CLEAR_GLOBAL_EXCEPTION_STATUS_REGISTER_SLIP_BITS()  SYS_Write_Word(GLBSTAT, GLBSTAT_FBSLIP_MASK|GLBSTAT_RFSLIP_MASK)
#define CLEAR_ABORT_EXCEPTION_STATUS_REGISTER()
#define GET_GLOBAL_EXCEPTION_STATUS_REGISTER()    SYS_Read_Bit( GLBSTAT, GLBSTAT_MASK )
#define GET_ABORT_EXCEPTION_STATUS_REGISTER()     (0)
#define GET_SYSTEM_EXCEPTION_STATUS_REGISTER()    SYS_Read_Bit( SYSESR, SYSESR_MASK )

#define PBIST_Write_Word(reg, data ) \
do \
{ \
    PBIST.##reg##.word = (data); \
}while(0)

#define PBIST_Read_Word(reg) \
         (PBIST.##reg##.word)

#define PBIST_Read_Bits(reg, member) \
         (PBIST.PBIST##reg.bits.member )

#define PBIST_Read_Failsafe_Index_Bit(reg, bit_mask, index) ( ( PBIST.PBISTFAILSF.##reg##[(index)].word ) & ( bit_mask ) )

#define PBIST_Read_Failsafe_Index_Member(reg,member,index) ( ( PBIST.PBISTFAILSF.##reg##[(index)].bits.##member ))

#define ENABLE_PBIST_INTERNAL_CLOCK()   PBIST_Write_Word(PBISTPACT, 3)
#define DISABLE_PBIST_INTERNAL_CLOCK()   PBIST_Write_Word(PBISTPACT, 0)

#define ENABLE_HET_PARITY()   HET_Write_Word(HETPCR,0x0A)


#define SELECT_ALGO_RAM_INFO_FROM_ROM() PBIST_Write_Word(PBISTROM, 3)
#define ENABLE_PBIST_APPLICATION_SELFTEST()  PBIST_Write_Word(PBISTOVER, 1)
#define ENABLE_PBIST_RAMGRP_SELFTEST()       PBIST_Write_Word(PBISTOVER, 0)
#define SELECT_RESPECTIVE_PBIST_MEMTEST_RAM_GROUP() \
do \
{ \
    PBIST_Write_Word(PBISTRINFOL, 0x07FFU); \
    PBIST_Write_Word(PBISTRINFOU, 0x0U); \
}while(0)

#define SELECT_PBIST_MEMTEST_RAM_GROUP(val) \
do \
{ \
    if ((unsigned long long)(val) >= 0x100000000ULL) \
    { \
    PBIST_Write_Word(PBISTRINFOL, 0x0U); \
    PBIST_Write_Word(PBISTRINFOU, ((unsigned long long)(val) >> 32)); \
    } \
    else \
    { \
    PBIST_Write_Word(PBISTRINFOL, (val)); \
    PBIST_Write_Word(PBISTRINFOU, 0x0U); \
    } \
}while(0)

#define SELECT_RESPECTIVE_PBIST_MEMTEST_ALGORITHM()  PBIST_Write_Word(PBISTALGO, 0xFFFFU)
#define SELECT_PBIST_MEMTEST_ALGORITHM_FOR_RAM_GRP(val) PBIST_Write_Word(PBISTALGO, (val))

#define START_ROM_BASED_CPU_DRIVEN_MEMTEST()   PBIST_Write_Word(PBISTDLR, 0x14U)
#define WAIT_TILL_MEMTEST_BIST_DONE(delay_value)  \
   timer_save = Get_Free_Timer(); \
do { \
       current_time = Get_Free_Timer(); \
       if ( current_time  >= (timer_save) ) \
       { \
          return_val = current_time - (timer_save); \
       } \
       else \
       { \
          return_val = ( RTI_MAX_COUNT_VALUE_OVERFLOW + current_time ) - timer_save; \
       } \
       current_time = return_val; \
       current_time = ( ( current_time * ( RTI_PRESCALE_FACTOR + 1 ) ) / (RTI_CLOCK/UNITS_TO_MILLI) ); \
       if (Boot_Timer_Get()) \
       { \
          Timer_Reload(); \
       } \
} while( (SYS_MSTDONE != SYS_Read_Bit(MSTCGSTAT, SYS_MSTDONE)) \
         && \
         (current_time < (delay_value)) \
         )

#define WAIT_TILL_MEMTEST_BIST_DONE_FLAG_IS_CLEARED(delay_value) \
   timer_save = Get_Free_Timer(); \
do { \
       current_time = Get_Free_Timer(); \
       if ( current_time  >= (timer_save) ) \
       { \
          return_val = current_time - (timer_save); \
       } \
       else \
       { \
          return_val = ( RTI_MAX_COUNT_VALUE_OVERFLOW + current_time ) - timer_save; \
       } \
       current_time = return_val; \
       current_time = ( ( current_time * ( RTI_PRESCALE_FACTOR + 1 ) ) / (RTI_CLOCK/UNITS_TO_MILLI) ); \
       if (Boot_Timer_Get()) \
       { \
          Timer_Reload(); \
       } \
} while( (SYS_MSTDONE == SYS_Read_Bit(MSTCGSTAT, SYS_MSTDONE)) \
         && \
         (current_time < (delay_value)) \
         )

#define WAIT_TILL_MEMINIT_MBIST_DONE_FLAG_IS_CLEARED(delay_value) \
   timer_save = Get_Free_Timer(); \
do { \
       current_time = Get_Free_Timer(); \
       if ( current_time  >= (timer_save) ) \
       { \
          return_val = current_time - (timer_save); \
       } \
       else \
       { \
          return_val = ( RTI_MAX_COUNT_VALUE_OVERFLOW + current_time ) - timer_save; \
       } \
       current_time = return_val; \
       current_time = ( ( current_time * ( RTI_PRESCALE_FACTOR + 1 ) ) / (RTI_CLOCK/UNITS_TO_MILLI) ); \
       if (Boot_Timer_Get()) \
       { \
          Timer_Reload(); \
       } \
} while( (SYS_MINIDONE == SYS_Read_Bit(MSTCGSTAT, SYS_MINIDONE)) \
         && \
         (current_time < (delay_value)) \
         )

#define GET_MEMORY_SELFTEST_RESULT()   (PBIST_Read_Failsafe_Index_Bit(PBISTFSRF, PBIST_FAIL_STATUS_MASK, 0) ||  PBIST_Read_Failsafe_Index_Bit(PBISTFSRF, PBIST_FAIL_STATUS_MASK, 1))
#define GET_MEMORY_SELFTEST_FAIL_INFO()   (PBIST_Read_Word(PBISTRAMT))
#define GET_MEMORY_SELFTEST_FAIL_COUNT()   (PBIST_Read_Failsafe_Index_Bit(PBISTFSRF, PBIST_FAIL_COUNT_MASK, 0))
#define GET_MEMORY_SELFTEST_FAIL_ADDRESS()   (PBIST_Read_Failsafe_Index_Bit(PBISTFSRF, PBIST_FAIL_ADDRESS_MASK, 0))
#define GET_MEMORY_SELFTEST_FAIL_RETURN_DATA()   (PBIST_Read_Failsafe_Index_Bit(PBISTFSRF, PBIST_FAIL_RETURN_DATA_MASK, 0))


#define CCM7_Write_Word(reg, data ) \
do \
{ \
    CCM7.##reg##.word = (data); \
}while(0)

#define CCM7_Read_Word(reg) (CCM7.##reg##.word )

#define ENTER_CCM7_SELF_TEST_ERROR_FORCE_MODE()  CCM7_Write_Word(CCMKEYR, 0x0FU)
#define ENTER_CCM7_ERROR_FORCE_MODE()  CCM7_Write_Word(CCMKEYR, 0x09U)
#define ENTER_CCM7_SELF_TEST_MODE()    CCM7_Write_Word(CCMKEYR, 0x06U)
#define WAIT_TILL_CCM7_SELF_TEST_COMPLETE()  while(CCM7_STC != (CCM7_Read_Word(CCMSR) & CCM7_STC))
#define GET_CCM7_SELF_TEST_COMPLETE_STATUS()    (CCM7_STC == (CCM7_Read_Word(CCMSR) & CCM7_STC))
#define IS_SELF_TEST_COMPLETE_FLAG_TRUE(var)     ((var) & CCM7_STC)
#define IS_SELF_TEST_COMPLETE_FLAG_TRUE_WITH_NO_ERROR(var)   ((CCM7_STC == (var & (CCM7_STC|CCM7_STE|CCM7_STET|CCM7_CMPE)))?TRUE:FALSE)
#define GET_CCM7_SELF_TEST_COMPLETE_STATUS_MASK()  (CCM7_STC)

#define ENTER_CCM7_LOCK_STEP_MODE()    CCM7_Write_Word(CCMKEYR, 0x00U)
#define GET_CCM7_STATUS()    (CCM7_Read_Word(CCMSR) & (CCM7_STC | CCM7_STE | CCM7_STET | CCM7_CMPE))
#define CLEAR_CCM7_CMPE_STATUS()    CCM7_Write_Word(CCMSR, CCM7_CMPE)
#define IS_CCM7_SELF_TEST_MODE()    ( (0x06U == (CCM7_Read_Word(CCMKEYR) & 0x0FU)) ? TRUE : FALSE)
#define IS_CCM7_LOCK_STEP_MODE()    ( (0x00U == (CCM7_Read_Word(CCMKEYR) & 0x0FU)) ? TRUE : FALSE)
#define GET_CCM7_MODE()    CCM7_Read_Word(CCMKEYR)
#define IS_CCM7_SELF_TEST_COMPLETE_WITHOUT_ERROR()    ((CCM7_STC == GET_CCM7_STATUS()) ? TRUE : FALSE)
#define IS_CCM7_SELF_TEST_STATUS_WITHOUT_ERROR(status)    ((CCM7_STC == (status)) ? TRUE : FALSE)
#define IS_CCM7_SELF_TEST_COMPARE_MATCH_ERROR(var)   (((CCM7_STC|CCM7_STE) == (var & (CCM7_STC|CCM7_STE|CCM7_STET)))?TRUE:FALSE)
#define IS_CCM7_SELF_TEST_COMPARE_MISMATCH_ERROR(var)   (((CCM7_STC|CCM7_STE|CCM7_STET) == (var & (CCM7_STC|CCM7_STE|CCM7_STET)))?TRUE:FALSE)

#define DMA_Write_Word(reg, data ) \
do \
{ \
    DMA.DMA##reg##.word = (data); \
}while(0)

#define DMA_Write(reg,member,value) \
         (DMA.DMA##reg.bits.member=(value))


#define DMA_Write_Word_Macro(reg, data )     DMA.DMA##reg##.word = (data)


#define DMA_Read_Word(reg) (DMA.DMA##reg##.word )

#define DMA_Read_Bits(reg,member) (DMA.DMA##reg##.bits.member)

#define DMA_PRIMARY_CONTROL_PACKET_Write_Word(index, reg, data ) \
do \
{ \
    DMA_PRIMARY_CONTROL_PACKET.DMAPRIMCHAN[(index)].DMA##reg##.word = (data); \
}while(0)

#define Set_DMA_Bits(reg, bit_mask ) \
do \
{ \
    DMA.DMA##reg##.word |= bit_mask ; \
}while(0)

#define Reset_DMA_Bits(reg, bit_mask ) \
do \
{ \
    DMA.DMA##reg##.word &= ~bit_mask ; \
}while(0)

#define DMA_PRIMARY_CONTROL_PACKET_Read_Word(index, reg) (DMA_PRIMARY_CONTROL_PACKET.DMAPRIMCHAN[(index)].DMA##reg##.word )
#define Setup_DMAREQ_Assignment_Channel(data, element)  DMA_Write_Word(DREQASIX[(element)], (data))

#define Setup_DMA_Channel_Port_Assignment(data, element)  Setup_DMA_Channel_Port_Assignment_Again(data, element)
#define Setup_DMA_Channel_Port_Assignment_Again(data, element)  DMA_Write_Word(PAR##element##, data )

#define SETUP_DMA_HW_REQ_CHANNEL_ENABLE(channel_mask)   DMA_Write_Word(HWCHENAS, (channel_mask))
#define SETUP_DMA_HW_REQ_CHANNEL_DISABLE(channel_mask)   DMA_Write_Word(HWCHENAR, (channel_mask))

#define SETUP_DMA_CHANNEL_TRANSFER_COUNT(channel, frame_count, element_count)   SETUP_DMA_CHANNEL_TRANSFER_COUNT_Again(channel, frame_count, element_count)
#define SETUP_DMA_CHANNEL_TRANSFER_COUNT_Again(DMA_CHANNEL, DMA_FRAME_COUNT, DMA_ELEMENT_COUNT)   DMA_PRIMARY_CONTROL_PACKET_Write_Word(DMA_CHANNEL, ITCOUNT, (((DMA_FRAME_COUNT)<<16) | (DMA_ELEMENT_COUNT)) )

#define DMA_CHANNEL_INTERRUPT_VECTOR_NUMBER(channel_index)   ((channel_index)+1)
#define GET_DMA_BTCA_INTERRUPT_VECTOR_NUMBER()                      (DMA_Read_Word(BTCAOFFSET))

#define GET_DMA_PARITY_ERROR_ADDR()                 (DMA_Read_Word(PAR) &  DMAPAR_MASK)
#define SETUP_DMA_FRAME_TRANSFER_COMPLETE_INTERRUPT(mask)  DMA_Write_Word(FTCINTENAS, mask )
#define SETUP_DMA_BLOCK_TRANSFER_COMPLETE_INTERRUPT(mask)  DMA_Write_Word(BTCINTENAS, mask )
#define SETUP_DMA_GLOBAL_TRANSFER_COMPLETE_INTERRUPT(mask)  DMA_Write_Word(GCHIENAS, mask )

#define ENABLE_DMA_RESET()   DMA_Write_Word(GCTRL, DMA_RES_MASK )
#define DISABLE_DMA_RESET()  Reset_DMA_Bits(GCTRL, DMA_RES_MASK )
#define GET_DMA_RESET()      (DMA_Read_Word(GCTRL) &  DMA_RES_MASK )
#define ENABLE_DMA()         DMA_Write_Word(GCTRL, DMA_EN_MASK )
#define GET_DMA_ENABLE()      (DMA_Read_Word(GCTRL) &  DMA_EN_MASK )

#define SETUP_DMA_HIGH_PRIORITY_QUEUE_SCHEDULING_SCHEME_FIXED_PRIORITY()      Reset_DMA_Bits(PTCRL, (0x1U<<17))
#define SETUP_DMA_HIGH_PRIORITY_QUEUE_SCHEDULING_SCHEME_ROTATING_PRIORITY()   Set_DMA_Bits(PTCRL, (0x1U<<17))
#define SETUP_DMA_LOW_PRIORITY_QUEUE_SCHEDULING_SCHEME_FIXED_PRIORITY()       Reset_DMA_Bits(PTCRL, (0x1U<<16))
#define SETUP_DMA_LOW_PRIORITY_QUEUE_SCHEDULING_SCHEME_ROTATING_PRIORITY()    Set_DMA_Bits(PTCRL, (0x1U<<16))
#define SETUP_DMA_LOW_PRIORITY_QUEUE_FOR_CHANNEL(channel_index)         DMA_Write_Word(CHPRIOR, (0x1U<<(channel_index)))
#define SETUP_DMA_HIGH_PRIORITY_QUEUE_FOR_CHANNEL(channel_index)        DMA_Write_Word(CHPRIOS, (0x1U<<(channel_index)))
#define SETUP_DMA_LOW_PRIORITY_QUEUE_FOR_ALL_CHANNEL()                  DMA_Write_Word(CHPRIOR, (0xFFFFU))
#define SETUP_DMA_HIGH_PRIORITY_QUEUE_FOR_SELECTED_CHANNELS(channel_mask)   DMA_Write_Word(CHPRIOS, (channel_mask))

#define GET_DMA_QUEUE_PRIORITY_FOR_ALL_CHANNEL()                  (DMA_Read_Word(CHPRIOR) & 0xFFFFU)
#define GET_DMA_QUEUE_PRIORITY_ALT_FOR_ALL_CHANNEL()              (DMA_Read_Word(CHPRIOS) & 0xFFFFU)

#define TRIGGER_DMA_SW_REQUEST(DMA_CHANNEL_MASK)    DMA_Write_Word(SWCHENAS, (DMA_CHANNEL_MASK) )
#define CLEAR_DMA_REQUEST_POLLING_FLAG_STATUS(DMA_CHANNEL_MASK)    DMA_Write_Word(BTCFLAG, (DMA_CHANNEL_MASK) )
#define GET_DMA_REQUEST_POLLING_FLAG_STATUS(DMA_CHANNEL_MASK)    (DMA_Read_Word(BTCFLAG) & (DMA_CHANNEL_MASK) )


#define GET_DMA_BUS_ERROR_POLLING_FLAG_STATUS(DMA_CHANNEL_MASK)    (DMA_Read_Word(BERFLAG) & (DMA_CHANNEL_MASK) )


#define INITIALIZE_PRIMARY_CONTROL_PACKET_SOURCE_ADDRESS(channel, address)  DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, ISADDR, address)
#define INITIALIZE_PRIMARY_CONTROL_PACKET_DESTINATION_ADDRESS(channel, address)  DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, IDADDR, address)
#define INITIALIZE_PRIMARY_CONTROL_PACKET_CONFIG(channel, config)      DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, CHCTRL, (config) )
#define DMA_ELEMENT_COUNT(val)   (val << 0)
#define DMA_FRAME_COUNT(val)   (val << 16)
#define DMA_ELEMENT_SRC_INDEX_OFFSET(val)   (val << 0)
#define DMA_ELEMENT_DST_INDEX_OFFSET(val)   (val << 16)
#define DMA_FRAME_SRC_INDEX_OFFSET(val)   (val << 0)
#define DMA_FRAME_DST_INDEX_OFFSET(val)   (val << 16)
#define INITIALIZE_PRIMARY_CONTROL_PACKET_TRANSFER_COUNT(channel, count_config)  DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, ITCOUNT, count_config)
#define INITIALIZE_PRIMARY_CONTROL_PACKET_ELEMENT_INDEXED_ADDRESSING_OFFSET(channel, offset_config)  DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, EIOFF, offset_config)
#define INITIALIZE_PRIMARY_CONTROL_PACKET_FRAME_INDEXED_ADDRESSING_OFFSET(channel, offset_config)  DMA_PRIMARY_CONTROL_PACKET_Write_Word(channel, FIOFF, offset_config)
#define SETUP_DMA_PARITY_CHECK_FUNCTIONALITY(val)  DMA_Write(PCR,PARITY_ENA,val);
#define SETUP_DMA_PARITY_ERROR_ACTION_FUNCTIONALITY(val)  DMA_Write(PCR,ERRA,val);
#define SETUP_INTERRUPT_ASSIGNMENT_FOR_DMA(module_num,val)   SETUP_INTERRUPT_ASSIGNMENT_FOR_DMA_AGAIN(module_num,val)
#define SETUP_INTERRUPT_ASSIGNMENT_FOR_DMA_AGAIN(PERIPHERAL_NUM,VALUE)   DMA_Write(MPCTRL,INT##PERIPHERAL_NUM##AB,VALUE);
#define SETUP_INTERRUPT_ENABLE_FOR_DMA(module_num,val)   SETUP_INTERRUPT_ENABLE_FOR_DMA_AGAIN(module_num,val)
#define SETUP_INTERRUPT_ENABLE_FOR_DMA_AGAIN(PERIPHERAL_NUM,VALUE)   DMA_Write(MPCTRL,INT##PERIPHERAL_NUM##ENA,VALUE);
#define SETUP_MEMORY_PROTECT_PERMISSION_FOR_DMA(module_num,val)  SETUP_MEMORY_PROTECT_PERMISSION_FOR_DMA_AGAIN(module_num,val)
#define SETUP_MEMORY_PROTECT_PERMISSION_FOR_DMA_AGAIN(PERIPHERAL_NUM,VALUE)   DMA_Write(MPCTRL,REG##PERIPHERAL_NUM##A,VALUE);
#define SETUP_REGION_ENABLE_FOR_DMA(module_num,val)   SETUP_REGION_ENABLE_FOR_DMA_AGAIN(module_num,val)
#define SETUP_REGION_ENABLE_FOR_DMA_AGAIN(PERIPHERAL_NUM,VALUE)   DMA_Write(MPCTRL,REG##PERIPHERAL_NUM##ENA,VALUE);

#define SETUP_DMA_REGION0_START_ADDRESS(val)   DMA_Write_Word(MPR0S,val)
#define SETUP_DMA_REGION0_END_ADDRESS(val)     DMA_Write_Word(MPR0E,val)
#define SETUP_DMA_REGION1_START_ADDRESS(val)   DMA_Write_Word(MPR1S,val)
#define SETUP_DMA_REGION1_END_ADDRESS(val)     DMA_Write_Word(MPR1E,val)
#define SETUP_DMA_REGION2_START_ADDRESS(val)   DMA_Write_Word(MPR2S,val)
#define SETUP_DMA_REGION2_END_ADDRESS(val)     DMA_Write_Word(MPR2E,val)
#define SETUP_DMA_REGION3_START_ADDRESS(val)   DMA_Write_Word(MPR3S,val)
#define SETUP_DMA_REGION3_END_ADDRESS(val)     DMA_Write_Word(MPR3E,val)

#define SETUP_PLL_1_VALUE(value)     SYS_Write_Word(PLLCTL1, (value) )
#define SETUP_PLL_2_VALUE(value)     SYS_Write_Word(PLLCTL2, (value) )
#define GET_PLL_1_VALUE()            SYS_Read_Word(PLLCTL1)
#define GET_PLL_2_VALUE()            SYS_Read_Word(PLLCTL2)

#define DISABLE_CLOCK_SOURCE_SINGLE(val)  SYS_Write_Word(CSDISSET, (val) )
#define ENABLE_CLOCK_SOURCE_SINGLE(val)  SYS_Write_Word(CSDISCLR, (val) )
#define ENABLE_CLOCK_SOURCE()        SYS_Write_Word(CSDIS, (~CDIS_SRC1) )
#define WAIT_TILL_PLL_LOCKED() \
{ \
  while (0 == (SYS_Read_Word(CSVSTAT) & CDIS_SRC1)); \
}

#define ENABLE_CLOCK_DOMAINS_SINGLE()       SYS_Write_Word(CDDISSET, (0xA0) )
#define ENABLE_CLOCK_DOMAINS()       SYS_Write_Word(CDDIS, (0) )
#define GET_CLOCK_DOMAINS()          SYS_Read_Word(CDDIS)

#define IS_CLOCK_ENABLED()           (0 == (SYS_Read_Word(CSDIS) & CDIS_SRC1))
#define IS_CLOCK_VALID()             (SYS_Read_Word(CSVSTAT) & CDIS_SRC1)
#define IS_CLOCK_VALID_SINGLE(val)             (SYS_Read_Word(CSVSTAT) & (val))
#define SELECT_CLOCK_SOURCE_FOR_CLOCK_TREE(val)   SYS_Write_Word(GHVSRC, (val) )
#define GET_CLOCK_SOURCE_FOR_CLOCK_TREE()   SYS_Read_Word(GHVSRC)

#define ENABLE_CLOCK_SOURCE_SINGLE_FMZPLL()  ENABLE_CLOCK_SOURCE_SINGLE(CDIS_SRC1)
#define DISABLE_CLOCK_SOURCE_SINGLE_LPO()    DISABLE_CLOCK_SOURCE_SINGLE(CDIS_SRC4)
#define ENABLE_CLOCK_SOURCE_DUAL_LPO()       ENABLE_CLOCK_SOURCE_SINGLE((CDIS_SRC4|CDIS_SRC5))
#define ENABLE_CLOCK_SOURCE_SINGLE_OSC()     ENABLE_CLOCK_SOURCE_SINGLE(CDIS_SRC0)
#define IS_OSC_CLOCK_VALID()                 IS_CLOCK_VALID_SINGLE(CDIS_SRC0)

#define PCR_Write_Word(reg, data ) \
do \
{ \
    PCR.##reg##.word = (data); \
}while(0)

#define PCR_Write_Word_Index(reg, index, data ) \
do \
{ \
    PCR.##reg##index##.word = (data); \
}while(0)

#define DISABLE_PERIPHERAL_CLOCK(index)   DISABLE_PERIPHERAL_CLOCK_Again(index)
#define DISABLE_PERIPHERAL_CLOCK_Again(INDEX)  PCR_Write_Word_Index(PSPWRDWNSET, INDEX, PCR_DISABLE_ALL)

#define ENABLE_PERIPHERAL_CLOCK(index)   ENABLE_PERIPHERAL_CLOCK_Again(index)
#define ENABLE_PERIPHERAL_CLOCK_Again(INDEX)  PCR_Write_Word_Index(PSPWRDWNCLR, INDEX, PCR_ENABLE_ALL)

#define ENABLE_PERIPHERAL_MEMORY_CLOCK(index)   ENABLE_PERIPHERAL_MEMORY_CLOCK_Again(index)
#define ENABLE_PERIPHERAL_MEMORY_CLOCK_Again(INDEX)  PCR_Write_Word_Index(PCSPWRDWNCLR, INDEX, PCR_ENABLE_ALL)

#define ENABLE_PERIPHERAL_MEMORY_PROTECTION(index)   ENABLE_PERIPHERAL_MEMORY_PROTECTION_Again(index)
#define ENABLE_PERIPHERAL_MEMORY_PROTECTION_Again(INDEX)  PCR_Write_Word_Index(PMPROTSET, INDEX, PCR_ENABLE_ALL)

#define DISABLE_PERIPHERAL_MEMORY_PROTECTION(index)   DISABLE_PERIPHERAL_MEMORY_PROTECTION_Again(index)
#define DISABLE_PERIPHERAL_MEMORY_PROTECTION_Again(INDEX)  PCR_Write_Word_Index(PMPROTCLR, INDEX, ~PCR_DISABLE_ALL)

#define ENABLE_PERIPHERAL_PROTECTION(index)   ENABLE_PERIPHERAL_PROTECTION_Again(index)
#define ENABLE_PERIPHERAL_PROTECTION_Again(INDEX)  PCR_Write_Word_Index(PPROTSET, INDEX, PCR_ENABLE_ALL)

#define DISABLE_PERIPHERAL_PROTECTION(index)   DISABLE_PERIPHERAL_PROTECTION_Again(index)
#define DISABLE_PERIPHERAL_PROTECTION_Again(INDEX)  PCR_Write_Word_Index(PPROTCLR, INDEX, ~PCR_DISABLE_ALL)

#define CLEAR_EXTERNAL_CLOCK_OUT_PIN()
#define SET_PERIPHERAL_CLOCK_RATIO(val)                SYS_Write_Word(CLKCNTL, val)
#define GET_PERIPHERAL_CLOCK_RATIO()                   SYS_Read_Word(CLKCNTL)
#define SET_PERIPHERAL_CLOCK_RATIO_VCLK2(current_register_contents, val)     SYS_Write_Word(CLKCNTL, ((current_register_contents & ~VCLKR2_MASK)|val))
#define SET_PERIPHERAL_CLOCK_RATIO_VCLK(current_register_contents, val)     SYS_Write_Word(CLKCNTL, ((current_register_contents & ~VCLKR1_MASK)|val))

#define SET_PERIPHERAL_ASYNCHRONOUS_CLOCK_SOURCE(val)  SYS_Write_Word(VCLKASRC, val)
#define ENABLE_PERIPHERAL()      Set_SYS_Bits(CLKCNTL, PENABLE )
#define DISABLE_PERIPHERAL()     Reset_SYS_Bits(CLKCNTL, PENABLE )

#define ENABLE_WATCHDOG_KEY_FOR_RESET(var)
#define FEED_WATCHDOG(var)

#define RTP_Write_Word(reg, data ) \
do \
{ \
    RTP.##reg.word = data ; \
}while(0)

#define Set_GIO_Bits_For_RTP(reg, bit_mask ) \
do \
{ \
    RTP.##reg.word |= bit_mask ; \
}while(0)

#define Reset_GIO_Bits_For_RTP(reg, bit_mask ) \
do \
{ \
    RTP.##reg.word &= ~bit_mask ; \
}while(0)

#define RTP_Read_Word(reg) (RTP.##reg.word )

#define SET_GIO_PORT_FUNCTION_FOR_RTPPORT(val)     RTP_Write_Word(RTPPC0, (val))
#define SET_GIO_PORT_OUPUT_LEVEL_FOR_RTPPORT(val)  RTP_Write_Word(RTPPC3, (val))
#define SET_GIO_PORT_OUPUT_DIR_FOR_RTPPORT(val)  RTP_Write_Word(RTPPC1, (val))
#define SET_GIO_PORT_INPUT_PULL_CONFIG_FOR_RTPPORT(val)     RTP_Write_Word(RTPPC7, (val))
#define SET_GIO_PORT_INPUT_PULL_SELECT_FOR_RTPPORT(val)     RTP_Write_Word(RTPPC8, (val))
#define SET_GIO_PORT_OUPUT_SET_FOR_RTPPORT(val)  RTP_Write_Word(RTPPC4, (val))
#define SET_GIO_PORT_OUPUT_CLR_FOR_RTPPORT(val)  RTP_Write_Word(RTPPC5, (val))
#define GET_GIO_PORT_STATE_FOR_RTPPORT()  RTP_Read_Word(RTPPC2)
#define SET_GIO_PIN_AS_OUTPUT_FOR_RTPPORT(val)  Set_GIO_Bits_For_RTP(RTPPC1, (val))
#define SET_GIO_PIN_AS_INPUT_FOR_RTPPORT(val)  Reset_GIO_Bits_For_RTP(RTPPC1, (val))
#define ENABLE_RTP_AS_GIO_PORT_FUNCTION()  RTP_Write_Word(RTPGLBCTRL, (0x05))
#define ENABLE_RTP_AS_TRACE_PORT_FUNCTION()  RTP_Write_Word(RTPGLBCTRL, (0x0A))

#define SET_GIO_PORT_OUTPUT_LEVEL_FOR_ECPPORT(val)  SYS_Write_Word(SYSPC4, (val))
#define SET_GIO_PORT_OUTPUT_DIR_FOR_ECPPORT(val)    SYS_Write_Word(SYSPC2, (val))
#define SET_GIO_PORT_OUTPUT_SET_FOR_ECPPORT(val)    SYS_Write_Word(SYSPC5, (val))
#define SET_GIO_PORT_OUTPUT_CLR_FOR_ECPPORT(val)    SYS_Write_Word(SYSPC6, (val))
#define GET_GIO_PORT_STATE_FOR_ECPPORT()            SYS_Read_Word(SYSPC3)
#define SET_GIO_PIN_AS_OUTPUT_FOR_ECPPORT(val)      Set_SYS_Bits(SYSPC2, (val))
#define SET_GIO_PIN_AS_INPUT_FOR_ECPPORT(val)       Reset_SYS_Bits(SYSPC2, (val))

#define ENABLE_ECP_AS_GIO_PORT_FUNCTION()           SYS_Write_Word(SYSPC1, (0x00))
#define ENABLE_ECP_AS_CLOCK_PORT_FUNCTION()         SYS_Write_Word(SYSPC1, (0x01))
#define IS_ECP_AS_GIO_PORT_FUNCTION()               ((0 == (SYS_Read_Word(SYSPC1) & 0x01)) ? TRUE : FALSE)
#define IS_GIO_PORT_AS_OUTPUT_DIR_FOR_ECPPORT()     (( 0x01 == (SYS_Read_Word(SYSPC2) & 0x01)) ? TRUE : FALSE)

#define ESM_Write_Word(reg, data ) \
do \
{ \
    ESM.##reg##.word = (data); \
}while(0)

#define ESM_Read_Word(reg) (ESM.##reg##.word )

#define ESM_Read_Word_Index(reg, index) (ESM.##reg##[(index)].word )

#define ESM_Write_Word_Index(reg, index, data) (ESM.##reg##[(index)].word = (data) )

#define GET_ESM_GROUP1_STATUS()   ESM_Read_Word_Index(ESMSRx, 0)
//#define CLEAR_ESM_GROUP1_STATUS(mask)   ESM_Write_Word(ESMSRx[0], (mask))
#define GET_ESM_GROUP2_STATUS()   ESM_Read_Word_Index(ESMSRx, 1)
#define GET_ESM_GROUP3_STATUS()   ESM_Read_Word_Index(ESMSRx, 2)

#define GET_ESM_INTERRUPT_OFFSET_HIGH_REGISTER()  ESM_Read_Word(ESMIOFFHR)
#define CLEAR_ESM_ERROR_INFLUENCE()   ESM_Write_Word(ESMIEPCR1, 0xFFFFFFFFU)

#define GET_ESM_INTERRUPT_OFFSET_LOW_REGISTER()  ESM_Read_Word(ESMIOFFLR)

#define IS_ENABLE_ESM_ERROR_PIN_CONTROL_FOR_CCM7_SELF_TEST()   ((ESM_Read_Word(ESMIEPSR1) & 0x80000000U)?TRUE:FALSE)
#define ENABLE_ESM_ERROR_PIN_CONTROL()   ESM_Write_Word(ESMIEPSR1, 0x4002AC2CU)
#define CLEAR_ESM_GROUP1_INTERRUPTS()   ESM_Write_Word(ESMIECR1, 0xFFFFFFFFU)
#define ENABLE_ESM_INTERRUPTS()   ESM_Write_Word(ESMIESR1, 0x44AAACACU)

#define GET_PSA_CONTENTS()
#define GET_ESM_GROUP2_SHADOW_STATUS()   ESM_Read_Word(ESMSSR2)
#define CLEAR_ALL_ESM_GROUP2_SHADOW_STATUS()   ESM_Write_Word(ESMSSR2, 0xFFFFFFFFU)
#define CLEAR_ESM_GROUP2_SHADOW_STATUS(mask)   ESM_Write_Word(ESMSSR2, (mask))
#define CLEAR_ESM_GROUP2_STATUS(mask)   ESM_Write_Word_Index(ESMSRx, 1, (mask))
#define GET_ESM_INTERRUPT_HIGH_LEVEL_OFFSET()   (ESM_Read_Word(ESMIOFFHR) & 0x7FU)
#define GET_ESM_INTERRUPT_LOW_LEVEL_OFFSET()   (ESM_Read_Word(ESMIOFFLR) & 0x7FU)
#define IS_ERROR_STATUS_MODULE_PRESENT() TRUE
#define GET_ESM_ERROR_PIN_REGISTER_STATUS()   (ESM_Read_Word(ESMEPSR) & ESM_ERROR_PIN_MASK)

#define ENABLE_ESMEKR_ERROR_FORCING_MODE() ESM_Write_Word(ESMEKR,ESMEKR_ENABLE_ERROR_FORCING_MODE)
#define ENABLE_ESMEKR_NORMAL_MODE() ESM_Write_Word(ESMEKR,ESMEKR_ENABLE_NORMAL_MODE)


#define CLEAR_ESM_GROUP1_STATUS(mask)   ESM_Write_Word_Index(ESMSRx, 0, (mask))
#define GET_ESM_ERROR_PIN_STATE()   (ESM_Read_Word(ESMEPSR) & 0x1U)
#define IS_ESM_ERROR_KEY_STATE_NORMAL()   ( (((ESM_Read_Word(ESMEKR) & 0xFU) != ESMEKR_ENABLE_ERROR_FORCING_MODE) && (((ESM_Read_Word(ESMEKR) & 0xFU)) != ESM_EKEY_ERROR_RESET)) ? TRUE : FALSE)
#define IS_ESM_ERROR_KEY_ERROR_FORCE_MODE()    ((ESMEKR_ENABLE_ERROR_FORCING_MODE == (ESM_Read_Word(ESMEKR) & 0xFU) ) ? TRUE : FALSE)
#define IS_ESM_GROUP_2_CPU_COMPARE_ERROR()  (GET_ESM_GROUP2_SHADOW_STATUS() & ESM_GROUP_2_ERROR_CCM7_COMPARE_FAIL_MASK)
#define IS_ESM_GROUP_1_CPU_SELF_TEST_ERROR() (GET_ESM_GROUP1_STATUS() & ESM_GROUP_1_ERROR_CCM7_SELFTEST_MASK)
#define GET_ESM_ERROR_KEY_STATE()   (ESM_Read_Word(ESMEKR) & 0xFU)


#define CLEAR_ESM_GROUP2_STATUS_FOR_CPU_COMPARE_ERROR()  CLEAR_ESM_GROUP2_STATUS(ESM_GROUP_2_ERROR_CCM7_COMPARE_FAIL_MASK)
#define CLEAR_ESM_GROUP2_SHADOW_STATUS_FOR_CPU_COMPARE_ERROR()  CLEAR_ESM_GROUP2_SHADOW_STATUS(ESM_GROUP_2_ERROR_CCM7_COMPARE_FAIL_MASK)
#define CLEAR_ESM_GROUP1_STATUS_FOR_CCM7_SELF_TEST_ERROR()   CLEAR_ESM_GROUP1_STATUS(ESM_GROUP_1_ERROR_CCM7_SELFTEST_MASK)


#define RAMECCREG_Write_Word(reg, data ) \
do \
{ \
    RAMECCREG.RAM##reg##.word = (data); \
}while(0)

#define RAMECCREG_Write_Word_Macro(reg, data )     RAMECCREG.RAM##reg##.word = (data)


#define RAMECCREG_Read_Word(reg) (RAMECCREG.RAM##reg##.word )

#define RAMECCREG_Read_Bits(reg,member) (RAMECCREG.RAM##reg##.bits.member)

#define Set_RAMECCREG_Bits(reg, bit_mask ) \
do \
{ \
    RAMECCREG.RAM##reg##.word |= bit_mask ; \
}while(0)

#define RAMECC_Read_Bits(reg, member)  (RAMECCREG.##reg.bits.member )

#define RAMECC_Write_Bits(reg, member, value ) \
do \
{ \
    RAMECCREG.##reg.bits.member = (value); \
}while(0)

#define Reset_RAMECCREG_Bits(reg, bit_mask ) \
do \
{ \
    RAMECCREG.RAM##reg##.word &= ~bit_mask ; \
}while(0)

#define SINGLE_ERROR_ADDRESS_REGISTER ((unsigned16_T)RAMECCREG_Read_Word(SERRADDR))
#define SINGLE_ERROR_POSITION         RAMECCREG_Read_Bits(ERRPOSITION,SERRPOSITION)
#define SINGLE_ERROR_TYPE             RAMECCREG_Read_Bits(ERRPOSITION,ERRTYPE)
#define SINGLE_ERROR_OCCURRENCE_COUNT RAMECCREG_Read_Bits(OCCUR,SEC_OCCUR)
#define RAM_ECC_DED_ADDRESS           ((unsigned16_T)RAMECCREG_Read_Bits(DERRADDR,DERRADDR))


#define ENABLE_ECC_MASK (0x0FU)
#define ENABLE_ECC (0x0AU)
#define DISABLE_ECC (0x05U)

#define ENABLE_ECC_RAM_WRITE_BY_CPU_MASK (0x0100U)

#define ENABLE_ECC_RMWCBYP_MASK (0x0FU << 9)
#define ENABLE_ECC_RMWCBYP      (0x05U << 9)
#define DISABLE_ECC_RMWCBYP     (0x0AU << 9)
//#define VIM_FALLBACK_ADDRESS_SET() VIM_PAR_Write_Word(VIM_FBPARERR, (unsigned32_T)&(VIM_RAM_Initialize) )


#define ENABLE_ERR_DETECT (0x0AU)
#define EANBLE_MIBSPI_PARITY (0x0AU)

#define ENABLE_ECC_DCAN1_PARITY (0x0AU)
#define ENABLE_ECC_DCAN2_PARITY (0x0AU)
#define ENABLE_ECC_ADC_PARITY (0x0AU)
#define ENABLE_ECC_HET_TU_PARITY (0x0AU)
#define ENABLE_ECC_DMA_PARITY (0x0AU)
#define DISABLE_ECC_DMA_PARITY (0x05U)
#define ENABLE_DMA_PARITY_ERROR_ACTION (0x01U << 16)
#define ENABLE_DMA_PARITY_TEST_MASK    (0x01U << 8)

#define ENABLE_ECC_VIM_PARITY (0x0AU)
#define ENABLE_TEST_VIM_PARITY_RAM_MASK (0x1 << 8)  /* TEST Bit of PARCTL Register of VIM */
#define ENABLE_MEMORY_INIT  (0x0AU)
#define DISABLE_MEMORY_INIT  (0x05U)

#define ECC_RAM_GROUP_EXCEPT_HET  (0x000001F7U)
#define ECC_HET_RAM  (0x00000008U)

#define ENABLE_ECC_ESRAM_MODULE()    \
do { \
    RAMECCREG_Write_Word(CTRL,  (RAMECCREG_Read_Word(CTRL) & ~ENABLE_ECC_MASK) | ENABLE_ECC); \
} while(0)

#define DISABLE_ECC_ESRAM_MODULE()    \
do { \
    RAMECCREG_Write_Word(CTRL,  (RAMECCREG_Read_Word(CTRL) & ~ENABLE_ECC_MASK) | DISABLE_ECC); \
} while(0)

#define ENABLE_ECC_RMWCBYP_ESRAM_MODULE()    \
do { \
    RAMECCREG_Write_Word(CTRL,  (RAMECCREG_Read_Word(CTRL) & ~ENABLE_ECC_RMWCBYP_MASK) | ENABLE_ECC_RMWCBYP); \
} while(0)

#define DISABLE_ECC_RMWCBYP_ESRAM_MODULE()    \
do { \
    RAMECCREG_Write_Word(CTRL,  (RAMECCREG_Read_Word(CTRL) & ~ENABLE_ECC_RMWCBYP_MASK) | DISABLE_ECC_RMWCBYP); \
} while(0)

#define ENABLE_ECC_ESRAM_WRITE_BY_CPU()   Set_RAMECCREG_Bits(CTRL, ENABLE_ECC_RAM_WRITE_BY_CPU_MASK )
#define DISABLE_ECC_ESRAM_WRITE_BY_CPU()  Reset_RAMECCREG_Bits(CTRL, ENABLE_ECC_RAM_WRITE_BY_CPU_MASK )

#define GET_ESRAM_RAMCTRL_ERRATA()   (RAMECCREG_Read_Word(CTRL))

#define CLEAR_ECC_ERROR_OCCURRENCE_COUNTER()  RAMECCREG_Write_Word(OCCUR, 0)
#define SET_ECC_ERROR_THRESHOLD_COUNTER(val)     RAMECCREG_Write_Word(THRESHOLD, (val))

#define GET_ECC_ERROR_STATUS()   (RAMECCREG_Read_Word(ERRSTATUS) & 0x73FU)
#define CLEAR_ECC_ERROR_STATUS_ADDRESS_DECODE_FAIL()   RAMECCREG_Write_Word(ERRSTATUS, 0x04)
#define CLEAR_ECC_ERROR_STATUS_SINGLE_ERROR_FAIL()   RAMECCREG_Write_Word(ERRSTATUS, 0x01)

#define GET_ECC_ERROR_OCCURRENCE_COUNTER()   (RAMECCREG_Read_Word(OCCUR) & 0xFFFFU)
#define GET_ECC_ERROR_INTERRUPT_ENABLE_STATUS()   (RAMECCREG_Read_Word(INTCTRL) & ENABLE_SINGLE_ERROR_INTERRUPT)

#define ENABLE_ECC_MIBSPI_PARITY_CHECK()   SPI_Write_Word(1,UERRCTRL,EANBLE_MIBSPI_PARITY)
#define ENABLE_ECC_DCAN1_PARITY_CHECK()   Set_DCAN_Reg_Bit(1,CTL,(ENABLE_ECC_DCAN1_PARITY<<10))
#define ENABLE_ECC_DCAN2_PARITY_CHECK()   Set_DCAN_Reg_Bit(2,CTL,(ENABLE_ECC_DCAN2_PARITY<<10))

#define ENABLE_ECC_ADC_PARITY_CHECK()     Set_ADC_Bits(PARCR,ENABLE_ECC_ADC_PARITY)
#define ENABLE_ADC_PARITY_CHECK()     ADC_Write_Word(PARCR,ENABLE_ECC_ADC_PARITY)

#define ENABLE_ECC_HET_TU_PARITY_CHECK()
#define ENABLE_ECC_DMA_PARITY_CHECK()     Set_DMA_Bits(PCR,ENABLE_ECC_DMA_PARITY)
#define IS_DMA_PARITY_CHECK_ENABLED()     ( (DISABLE_ECC_DMA_PARITY == (DMA_Read_Word(PCR) & 0xFU)) ? FALSE : TRUE)
#define ENABLE_DMA_PARITY_CHECK()     DMA_Write_Word(PCR, ENABLE_ECC_DMA_PARITY|ENABLE_DMA_PARITY_ERROR_ACTION)
#define DISABLE_DMA_PARITY_CHECK()     DMA_Write_Word(PCR, DISABLE_ECC_DMA_PARITY)
#define ENABLE_DMA_PARITY_TEST()     Set_DMA_Bits(PCR, ENABLE_DMA_PARITY_TEST_MASK)
#define DISABLE_DMA_PARITY_TEST()    Reset_DMA_Bits(PCR, ENABLE_DMA_PARITY_TEST_MASK)
#define ENABLE_ECC_VIM_PARITY_CHECK()   VIM_PAR_Write_Word(VIM_PARCTL,ENABLE_ECC_VIM_PARITY)

#define ENABLE_TEST_VIM_PARITY_RAM()   Set_VIM_PAR_Bits(VIM_PARCTL,ENABLE_TEST_VIM_PARITY_RAM_MASK)
#define DISABLE_TEST_VIM_PARITY_RAM()  Reset_VIM_PAR_Bits(VIM_PARCTL,ENABLE_TEST_VIM_PARITY_RAM_MASK)
#define GET_VIM_PARITY_ERROR_STATUS()  (VIM_PAR_Read_Word(VIM_PARFLG) & 0x01)
#define CLEAR_VIM_PARITY_ERROR_STATUS()  VIM_PAR_Write_Word(VIM_PARFLG, 0x01)

#define ENABLE_GLOBAL_MEMORY_INITIALIZATION()   Set_SYS_Bits(MINITGCR,ENABLE_MEMORY_INIT)
#define DISABLE_GLOBAL_MEMORY_INITIALIZATION()  Set_SYS_Bits(MINITGCR,DISABLE_MEMORY_INIT)

#define SELECT_OTHER_RAM_GROUP()   Set_SYS_Bits(MSINENA,ECC_RAM_GROUP_EXCEPT_HET)
#define SELECT_HET_RAM()   Set_SYS_Bits(MSINENA,ECC_HET_RAM)
#define WAIT_TILL_MEMTEST_ECC_DONE()   (SYS_MINIDONE != SYS_Read_Bit(MSTCGSTAT, SYS_MINIDONE))


#define ENABLE_SINGLE_ERROR_INTERRUPT (0x01U)
#define SEC_THRESHOLD_VALUE (0x0001U)
#define CLEAR_ESM_INF_ERR_PIN (0xFFFFFFFFU)
#define ENABLE_PIN_CTRL_DMA (0x00000008U)
#define CLEAR_ESM_INTERRUPT_ALL (0xFFFFFFFFU)

//#define BOOT_ESM_GROUP1_ERR_ENA       (0x042A00A8UL)
//#define BOOT_ESM_GROUP_2_FLG_MASK     (0x00000C54UL)


#define DISABLE_ECC_SEC_INT()           RAMECCREG_Write_Word(INTCTRL,0x0)
#define ENABLE_ECC_SEC_INT()            RAMECCREG_Write_Word(INTCTRL,ENABLE_SINGLE_ERROR_INTERRUPT)
//#define GET_ECC_SEC_INT_ENABLE_STATUS()            (RAMECCREG_Read_Word(INTCTRL) & ENABLE_SINGLE_ERROR_INTERRUPT)
#define INIT_SEC_THRESHOLD()            RAMECCREG_Write_Word(THRESHOLD,SEC_THRESHOLD_VALUE)
#define CLEAR_ESM_INF_ERR_PIN_REG()     ESM_Write_Word(ESMIEPCR1,CLEAR_ESM_INF_ERR_PIN)
#define SET_ESM_REG_FOR_DMA_GROUP1()    ESM_Write_Word(ESMIEPSR1,ENABLE_PIN_CTRL_DMA)
#define CLEAR_ESM_INT_REG()             ESM_Write_Word(ESMIECR1,CLEAR_ESM_INTERRUPT_ALL)
#define ENABLE_ESM_INT_REG_FOR_GROUP1() ESM_Write_Word(ESMIESR1,BOOT_ESM_GROUP1_ERR_ENA)
#define ENABLE_ESM_INT_REG_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMIESR1,(val_mask))

#define SELECT_ESM_INT_LEVEL_LOW_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMILCR1,(val_mask))
#define SELECT_ESM_INT_LEVEL_HIGH_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMILSR1,(val_mask))
#define SELECT_ESM_INT_ENABLE_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMIESR1,(val_mask))
#define SELECT_ESM_INT_DISABLE_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMIECR1,(val_mask))
#define DISABLE_ESM_INT_ALL_FOR_GROUP1_MASK() ESM_Write_Word(ESMIECR1,(0xFFFFFFFFU))
#define SELECT_ESM_INFLUENCE_ERROR_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMIEPSR1,(val_mask))
#define SELECT_ESM_INFLUENCE_ERROR_NOT_FOR_GROUP1_MASK(val_mask) ESM_Write_Word(ESMIEPCR1,(val_mask))
#define DISABLE_ESM_INFLUENCE_ERROR_ALL_FOR_GROUP1_MASK() ESM_Write_Word(ESMIEPCR1,(0xFFFFFFFFU))

#define ESM_FLASH_PARITY_MASK 0x00000010UL
//#define CLEAR_ESM_GROUP2_STATUS(mask)   ESM_Write_Word_Index(ESMSRx, 1, mask)
#define CLEAR_ESM_GROUP2_FLASH_PARITY_ERROR()  CLEAR_ESM_GROUP2_STATUS(ESM_FLASH_PARITY_MASK)
#define CLEAR_ESM_GROUP2_STATUS_CCM_COMPARE_ERROR()   CLEAR_ESM_GROUP2_STATUS(0x04)


#define GET_PARITY_SYSTEM_WIDE()          SYS_Read_Word(DEVCR1)
/* Parity Selection */
#define SELECT_ODD_PARITY_SYSTEM_WIDE()   SYS_Write_Word(DEVCR1, (0x0A))  /* Decault at Power ON */
#define SELECT_EVEN_PARITY_SYSTEM_WIDE()   SYS_Write_Word(DEVCR1, (0x05))
#define IS_ODD_PARITY_SYSTEM_WIDE()      ( (0x0A == (SYS_Read_Word(DEVCR1) & 0x0F)) ? TRUE : FALSE )
#define GET_RAM_SINGLE_ERROR_ADDRESS()      RAMECC_Read_Bits(RAMSERRADDR, SERRADDR)
#define GET_RAM_SINGLE_ERROR_POSITION()     RAMECC_Read_Bits(RAMERRPOSITION, SERRPOSITION)
#define GET_RAM_SINGLE_ERROR_TYPE()         RAMECC_Read_Bits(RAMERRPOSITION, ERRTYPE)


#define GET_RAM_SINGLE_ERROR_OCCURRENCE_COUNTER()       RAMECC_Read_Bits(RAMOCCUR, SEC_OCCUR)

#define CLEAR_SINGLE_ERROR_CORRECT_INT_FLAG()       RAMECCREG_Write_Word(ERRSTATUS, 1)

#define GET_MEMINIT_MBIST_DONE_FLAG()  (SYS_Read_Word(MSTCGSTAT) & SYS_MINIDONE)
#define GET_MEMORY_INIT_STATUS()  SYS_Read_Word(MINISTAT)
#define VIM_FALLBACK_ADDRESS_SET(func_address) VIM_PAR_Write_Word(VIM_FBPARERR, (unsigned32_T)func_address )
#define VIM_ENABLE_PARITY_CHECK()      VIM_PAR_Write_Word(VIM_PARCTL, (VIM_PAR_EN | VIM_TEST_EN ))
#define ENABLE_SRAM_ECC()         SRAM_Write_Word(RAMCTRL,( ( SRAM_Read_Word(RAMCTRL) & 0xFFFFFFF0) | RAM_ECC_ENABLE ) )
#define INIT_SRAM_ECC()           Set_SRAM_Bits(RAMCTRL, ( RAM_ECC_INIT | RAM_ECC_WRT_ENA | RAM_RMWCBYP ) )
#define ENABLE_SRAM_SEC_ERROR_THRESHOLD()    SRAM_Write_Word(RAMTHRESHOLD,  RAM_RAMTHRESHOLD)
#define ENABLE_SRAM_SEC_INTERRUPT()      SRAM_Write_Word(RAMINTCTRL,  RAM_SECINTEN)
#define ENABLE_SRAM_ECC_WRITE()          Set_SRAM_Bits(RAMCTRL, RAMCTRL_ECC_WRT_ENA )
#define DISABLE_SRAM_ECC_WRITE()         Set_SRAM_Bits(RAMCTRL, RAMCTRL_ECC_WRT_DIS )
#define CLEAR_DOUBLE_ERROR()             SRAM_Write_Word(RAMERRSTATUS, RAM_SEC_DED )
#define GET_MEMORY_FAULT_DETECT_STATUS() RAM_Read_Word(RAMERRSTATUS)
#define ESM_LOW_TIME_COUNTER_PRELOAD()   ESM_Write_Word(ESMLTCPR, ESM_TLCP)
#define ESM_ERROR_KEY()                 ESM_Write_Word(ESMEKR, ESM_EKEY)
#define ESM_ERROR_KEY_NORMAL_MODE()                 ESM_Write_Word(ESMEKR, ESM_EKEY_NORMAL)
#define ESM_ERROR_KEY_ERROR_FORCE_MODE()            ESM_Write_Word(ESMEKR, ESM_EKEY_ERROR_FORCE)
#define ESM_ERROR_KEY_ERROR_RESET_MODE()            ESM_Write_Word(ESMEKR, ESM_EKEY_ERROR_RESET)

#define ENABLE_MEM_HARDWARE_INIT()        SYS_Write_Word( MINITGCR, MEM_HARDWARE_INIT )
#define DISABLE_MEM_HARDWARE_INIT()        SYS_Write_Word( MINITGCR, DIS_MEM_HARDWARE_INIT )
#define RTI_UP_COUNTER   RTI_Write_Word(RTIUC0, RTI0UC)
#define ENABLE_ALL_RAM_EXCEPT_HET()        SYS_Write_Word( MSINENA, ENABLE_ALL_RAM )
#define ENABLE_NHET_RAM_INIT()      SYS_Write_Word( MSINENA, ENABLE_NHET_RAM  )
#define CLEAR_ESM_ERROR_INFLUENCE()   ESM_Write_Word(ESMIEPCR1, 0xFFFFFFFFU)
#define ENABLE_ESM_ERROR_PIN_CONTROL()   ESM_Write_Word(ESMIEPSR1, 0x4002AC2CU)
#define CLEAR_ESM_GROUP1_INTERRUPTS()   ESM_Write_Word(ESMIECR1, 0xFFFFFFFFU)
#define ENABLE_ESM_INTERRUPTS()   ESM_Write_Word(ESMIESR1, 0x44AAACACU)
#define CLEAR_MEMTEST_HARDWARE_DONE_FLAG()  SYS_Write_Word(MSTCGSTAT, SYS_MINIDONE)

#define ENABLE_DMA_TEST_CONTROL()       DMA_Write_Word(RTCRL, DMA_RTC)
#define ENABLE_DMA_DEBUG()       DMA_Write_Word(DCTRL, DMA_DEBUG_ENABLE)
#define ENABLE_DMA_PARITY_ENABLE()   DMA_Write_Word(PCR, (DMA_PARITY  | DMA_TEST_EN | DMA_ERROR_ACTION ) )
#define ENABLE_DMA_MP_REGION0()   DMA_Write_Word(MPCTRL, (DMA_INT0AB   | DMA_INT0ENA  | DMA_REG0AP  | DMA_REG0ENA ))
#define WRITE_DMA_MP_REGION0_START_ADDRESS()   DMA_Write_Word(MPR0S, DMA_MP_REGION0_START_ADDRESS )
#define WRITE_DMA_MP_REGION0_END_ADDRESS() DMA_Write_Word(MPR0E, DMA_MP_REGION0_END_ADDRESS )
#define ENABLE_DMA_MP_REGION1()   DMA_Write_Word(MPCTRL, (DMA_INT1AB   | DMA_INT1ENA  | DMA_REG1AP  | DMA_REG1ENA ))
#define WRITE_DMA_MP_REGION1_START_ADDRESS()  DMA_Write_Word(MPR1S, DMA_MP_REGION1_START_ADDRESS )
#define WRITE_DMA_MP_REGION1_END_ADDRESS()  DMA_Write_Word(MPR1E, DMA_MP_REGION1_END_ADDRESS )
#define ENABLE_DMA_MP_REGION2()   DMA_Write_Word(MPCTRL, (DMA_INT2AB   | DMA_INT2ENA  | DMA_REG2AP  | DMA_REG2ENA ))
#define WRITE_DMA_MP_REGION2_START_ADDRESS()   DMA_Write_Word(MPR2S, DMA_MP_REGION2_START_ADDRESS )
#define WRITE_DMA_MP_REGION2_END_ADDRESS() DMA_Write_Word(MPR2E, DMA_MP_REGION2_END_ADDRESS)
#define ENABLE_DMA_MP_REGION3()   DMA_Write_Word(MPCTRL, (DMA_INT3AB   | DMA_INT3ENA  | DMA_REG3AP  | DMA_REG3ENA ))
#define WRITE_DMA_MP_REGION3_START_ADDRESS()   DMA_Write_Word(MPR3S, DMA_MP_REGION3_START_ADDRESS)
#define WRITE_DMA_MP_REGION3_END_ADDRESS()   DMA_Write_Word(MPR3E, DMA_MP_REGION3_END_ADDRESS)

#define SELECT_MBIST_RAM_GROUP_TO_BE_INIT(val)  SYS_Write_Word(MSINENA, (val))
#define ENABLE_PBIST_MEMORY_INIT_MODE() SYS_Write_Word(MINITGCR, (0x0A))
#define DISABLE_PBIST_MEMORY_INIT_MODE() SYS_Write_Word(MINITGCR, (0x05))


/******************************
* Types
*******************************/



typedef enum
{
   CLOCK_SOURCE_1,     /* 0x0000 Oscillator */
   CLOCK_SOURCE_2,     /* 0x0001 PLL        */
   CLOCK_SOURCE_3,     /* 0x0002 32KHZ      */
   CLOCK_SOURCE_4,     /* reserved         */
   CLOCK_SOURCE_5,     /* reserved         */
   CLOCK_SOURCE_6,     /* reserved         */
   CLOCK_SOURCE_7,     /* reserved         */
   CLOCK_SOURCE_8,     /* reserved         */
   CLOCK_SOURCE_9,     /* reserved         */
   CLOCK_SOURCE_10,    /* reserved         */
   CLOCK_SOURCE_11,    /* reserved         */
   CLOCK_SOURCE_12,    /* reserved         */
   CLOCK_SOURCE_13,    /* reserved         */
   CLOCK_SOURCE_14,    /* reserved         */
   CLOCK_SOURCE_15,    /* reserved         */
   CLOCK_SOURCE_16     /* reserved         */
}  SYS_CLOCK_SOURCE_T;

/* BMMCR1 (Bus Matrix Module Control Register1)  */
typedef union
{
   struct
   {
      bit_field_T             :28;  /* Reserved bit 31 - 4             */
      bit_field_T MEMSW       :4;   /* Memory Swap bit                 */
   } bits;
   unsigned32_T  word;
} BMMCR1_T;

/* BMMCR2 (Bus Matrix Module Control Register2)  */
typedef union
{
   struct
   {
      bit_field_T             :25;  /* Reserved bit 31 - 7              */
      bit_field_T PRTY_HPI    :1;   /* Arbitration Priority to HPI      */
      bit_field_T PRTY_RAM3   :1;   /* Arbitration Priority to eSRAM3   */
      bit_field_T PRTY_RAM2   :1;   /* Arbitration Priority to eSRAM2   */
      bit_field_T PRTY_CRC    :1;   /* Arbitration Priority to CRC      */
      bit_field_T PRTY_PBRG   :1;   /* Arbitration Pri Peripheral Bridge*/
      bit_field_T PRTY_FLASH  :1;  /* Arbitration Priority to FLASH     */
      bit_field_T PRTY_RAM0   :1;   /* Arbitration Priority to eSRAM0   */
   } bits;
   unsigned32_T  word;
} BMMCR2_T;

/* CDDIS (Clock Domain Disable Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T RTICLK2OFF   :1;  /* RTICLK 2 Domain OFF             */
      bit_field_T RTICLK1OFF   :1;  /* RTICLK 1 Domain OFF             */
      bit_field_T VCLKA2OFF    :1;  /* VCLKA 2 Domain OFF              */
      bit_field_T VCLKA1OFF    :1;  /* VCLKA Domain OFF                */
      bit_field_T VCLK2OFF     :1;  /* VCLK 2 Domain OFF               */
      bit_field_T VCLKPOFF     :1;  /* Peripheral VBUS domain OFF      */
      bit_field_T HCLKOFF      :1;  /* System clock domain OFF         */
      bit_field_T GCLKOFF      :1;  /* GCLK Domain OFF                 */
   } bits;
   unsigned32_T  word;
} CDDIS_T;

/* CDDISCLR (Clock Domain Disable Clear Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T RTICLK2OFFCLR:1;  /* CLR RTICLK 2 Domain OFF         */
      bit_field_T RTICLK1OFFCLR:1;  /* CLR RTICLK 1 Domain OFF         */
      bit_field_T VCLKA2OFFCLR :1;  /* CLR VCLKA 2 Domain OFF          */
      bit_field_T VCLKA1OFFCLR :1;  /* CLR VCLKA Domain OFF            */
      bit_field_T VCLK2OFFCLR  :1;  /* CLR VCLK 2 Domain OFF           */
      bit_field_T VCLKPOFFCLR  :1;  /* CLR Peripheral VBUS domain OFF  */
      bit_field_T HCLKOFFCLR   :1;  /* CLR System clock domain OFF     */
      bit_field_T GCLKOFFCLR   :1;  /* CLR GCLK Domain OFF             */
   } bits;
   unsigned32_T  word;
} CDDISCLR_T;

/* CDDISSET (Clock Domain Disable Set Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T RTICLK2OFFSET:1;  /* Set RTICLK 2 Domain OFF         */
      bit_field_T RTICLK1OFFSET:1;  /* Set RTICLK 1 Domain OFF         */
      bit_field_T VCLKA2OFFSET :1;  /* Set VCLKA 2 Domain OFF          */
      bit_field_T VCLKA1OFFSET :1;  /* Set VCLKA Domain OFF            */
      bit_field_T VCLK2OFFSET  :1;  /* Set VCLK 2 Domain OFF           */
      bit_field_T VCLKPOFFSET  :1;  /* Set Peripheral VBUS domain OFF  */
      bit_field_T HCLKOFFSET   :1;  /* Set System clock domain OFF     */
      bit_field_T GCLKOFFSET   :1;  /* Set GCLK Domain OFF             */
   } bits;
   unsigned32_T  word;
} CDDISSET_T;


/* CLKCNTL (Clock Control Register)  */
typedef union
{
   struct
   {
      bit_field_T             :4;   /* Reserved bit 31 - 28            */
      bit_field_T VCLK2R      :4;   /* VBUS clock2 ratio               */
      bit_field_T             :4;   /* Reserved bit 23 - 20            */
      bit_field_T VCLKR       :4;   /* VBUS clock ratio                */
      bit_field_T             :7;   /* Reserved bit 15 - 9             */
      bit_field_T PENA        :1;   /* Peripheral Enable               */
      bit_field_T             :8;   /* Reserved bit 7 - 0              */
   } bits;
   unsigned32_T  word;
} CLKCNTL_T;


/* CSDIS (Clock Source Disable Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T CLKSR7OFF    :1;  /* Clock Source 8 OFF              */
      bit_field_T CLKSR6OFF    :1;  /* Clock Source 7 OFF              */
      bit_field_T CLKSR5OFF    :1;  /* Clock Source 6 OFF              */
      bit_field_T CLKSR4OFF    :1;  /* Clock Source 5 OFF              */
      bit_field_T CLKSR3OFF    :1;  /* Clock Source 4 OFF              */
      bit_field_T CLKSR2OFF    :1;  /* Clock Source 3 OFF              */
      bit_field_T CLKSR1OFF    :1;  /* Clock Source 2 OFF              */
      bit_field_T CLKSR0OFF    :1;  /* Clock Source 1 OFF              */
   } bits;
   unsigned32_T  word;
} CSDIS_T;

/* CSDISCLR (Clock Source Disable Clear Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8              */
      bit_field_T CLRCLKSR7OFF :1;  /* Clock Source 8 ENABLE           */
      bit_field_T CLRCLKSR6OFF :1;  /* Clock Source 7 ENABLE           */
      bit_field_T CLRCLKSR5OFF :1;  /* Clock Source 6 ENABLE           */
      bit_field_T CLRCLKSR4OFF :1;  /* Clock Source 5 ENABLE           */
      bit_field_T CLRCLKSR3OFF :1;  /* Clock Source 4 ENABLE           */
      bit_field_T CLRCLKSR2OFF :1;  /* Clock Source 3 ENABLE           */
      bit_field_T CLRCLKSR1OFF :1;  /* Clock Source 2 ENABLE           */
      bit_field_T CLRCLKSR0OFF :1;  /* Clock Source 1 ENABLE           */
   } bits;
   unsigned32_T  word;
} CSDISCLR_T;

/* CSDISSET (Clock Source Disable Set Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T SETCLKSR7OFF :1;  /* Clock Source 8 DISABLE          */
      bit_field_T SETCLKSR6OFF :1;  /* Clock Source 7 DISABLE          */
      bit_field_T SETCLKSR5OFF :1;  /* Clock Source 6 DISABLE          */
      bit_field_T SETCLKSR4OFF :1;  /* Clock Source 5 DISABLE          */
      bit_field_T SETCLKSR3OFF :1;  /* Clock Source 4 DISABLE          */
      bit_field_T SETCLKSR2OFF :1;  /* Clock Source 3 DISABLE          */
      bit_field_T SETCLKSR1OFF :1;  /* Clock Source 2 DISABLE          */
      bit_field_T SETCLKSR0OFF :1;  /* Clock Source 1 DISABLE          */
   } bits;
   unsigned32_T  word;
} CSDISSET_T;

/* CSVSTAT (Clock Source Valid Status Register)  */
typedef union
{
   struct
   {
      bit_field_T              :24; /* Reserved bit 31 - 8             */
      bit_field_T CLKSR7V      :1;  /* Clock Source8 Valid             */
      bit_field_T CLKSR6V      :1;  /* Clock Source7 Valid             */
      bit_field_T CLKSR5V      :1;  /* Clock Source6 Valid             */
      bit_field_T CLKSR4V      :1;  /* Clock Source5 Valid             */
      bit_field_T CLKSR3V      :1;  /* Clock Source4 Valid             */
      bit_field_T CLKSR2V      :1;  /* Clock Source3 Valid             */
      bit_field_T CLKSR1V      :1;  /* Clock Source2 Valid             */
      bit_field_T CLKSR0V      :1;  /* Clock Source1 Valid             */
   } bits;
   unsigned32_T  word;
} CSVSTAT_T;

/* Memory Self-Test Global Control Register (MSTGCR)  */
typedef union
{
   struct
   {
      bit_field_T             :22;  /* Reserved bit 31 - 10                           */
      bit_field_T ROM_DIV     :2;   /* Prescaler divider bits for ROM clock source    */
      bit_field_T             :4;   /* Reserved bit 7 - 4                             */
      bit_field_T MSTGENA     :4;   /* Memory self-test controller global enable key  */
   } bits;
   unsigned32_T  word;
} MSTGCR_T;

/* Memory Hardware Initialization Global Control Register (MINITGCR)  */
typedef union
{
   struct
   {
      bit_field_T             :28;  /* Reserved bit 31 - 4                                */
      bit_field_T MINITGENA   :4;   /* Memory hardware initialization global enable key   */
   } bits;
   unsigned32_T  word;
} MINITGCR_T;

/* MBIST Controller/ Memory Initialization Enable Register (MSINENA)  */
typedef union
{
   struct
   {
      bit_field_T MSINENA     :32;  /* MBIST controller/memory initialization enable register   */
   } bits;
   unsigned32_T  word;
} MSINENA_T;

/* Memory Self-Test Fail Status Register (MSTFAIL)  */
typedef union
{
   struct
   {
      bit_field_T MSTFAIL     :32;  /* Memory self-test fail status  */
   } bits;
   unsigned32_T  word;
} MSTFAIL_T;

/* MSTC Global Status Register (MSTCGSTAT)  */
typedef union
{
   struct
   {
      bit_field_T             :23;  /* Reserved bit 31 - 9                              */
      bit_field_T MINIDONE    :1;   /* Mem h/W initialization test run complete status  */
      bit_field_T             :7;   /* Reserved bit 7 - 1                               */
      bit_field_T MSTDONE     :1;   /* Mem self-test run complete status                */
   } bits;
   unsigned32_T  word;
} MSTCGSTAT_T;

/* Memory Hardware Initialization Status Register (MINISTAT)  */
typedef union
{
   struct
   {
      bit_field_T MIDONE      :32;  /* Memory hardware initialization status bit  */
   } bits;
   unsigned32_T  word;
} MINISTAT_T;

/* ECPCNTL (ECP Control Register)  */
typedef union
{
   struct
   {
      bit_field_T             :7;   /* Reserved bit 31 - 25            */
      bit_field_T ECPSSEL     :1;   /* Selection bit of VCLK/OSCIN     */
      bit_field_T ECPCOS      :1;   /* ECP continue on suspend         */
      bit_field_T             :7;   /* Reserved bit 22 - 16            */
      bit_field_T ECPDIV      :16;  /* ECP divider value               */
   } bits;
   unsigned32_T  word;
} ECPCNTL_T;

/* DSP Master Global Control Register (DSPGCR)  */
typedef union
{
   struct
   {
      bit_field_T             :31;  /* Reserved bit 31 - 1 */
      bit_field_T DSPSHENA    :1;   /* DSP share enable    */
   } bits;
   unsigned32_T  word;
} DSPGCR_T;

/* DEV Parity Control Register1 (DEVCR1)  */
typedef union
{
   struct
   {
      bit_field_T             :28;  /* Reserved bit 31 - 4             */
      bit_field_T DEVPARSEL   :4;   /* Device parity select bit key    */
   } bits;
   unsigned32_T  word;
} DEVCR1_T;

/* GHVSRC (GCLK,HCLK,VCLK,VCLK2 Source Register)  */
typedef union
{
   struct
   {
      bit_field_T             :4;   /* Reserved bit 31 - 28            */
      bit_field_T GHVWAKE     :4;   /* GCLK,HCLK,VCLK,VCLK2 src wakeup */
      bit_field_T             :4;   /* Reserved bit 23 - 20            */
      bit_field_T HVLPM       :4;   /* HCLK,VCLK,VCLK2 src when GCLKoff*/
      bit_field_T             :12;  /* Reserved bit 15 - 4             */
      bit_field_T GHVSRC      :4;   /* GCLK,HCLK,VCLK,VCLK2 current src*/
   } bits;
   unsigned32_T  word;
} GHVSRC_T;

/* GLBSTAT (Global Status Register)  */
typedef union
{
   struct
   {
      bit_field_T             :22; /* Reserved bit 31 - 10           */
      bit_field_T FBSLIP      :1;  /* PLL over cycle slip detection  */
      bit_field_T RFSLIP      :1;  /* PLL under cycle slip detection */
      bit_field_T             :7;  /* Reserved bit 7 - 1             */
      bit_field_T OSCFAIL     :1;  /* Oscillator Fail Flag bit       */
   } bits;
   unsigned32_T  word;
} GLBSTAT_T;

/* ITIFLAG (Illegal transaction interrupt flag Register)  */
typedef union
{
   struct
   {
      bit_field_T             :4;  /* Reserved bits 15-12             */
      bit_field_T ILLTFLAG    :1;  /* AHB inst bus ill trans intr flag*/
      bit_field_T DILTFLAG    :1;  /* AHB data bus ill trans intr flag*/
      bit_field_T PILTFLAG    :1;  /* CPU VBUS WR ill trans intr flag */
      bit_field_T DPILTFLAG   :1;  /* DMA VBUS WR ill trans intr flag */
      bit_field_T ETFLAG      :1;  /* CPU EMIF WR timeout err intr flg*/
      bit_field_T DETFLAG     :1;  /* DMA EMIF WR timeout err intr flg*/
      bit_field_T UCEFLAG6    :1;  /* Uncorrectable error intr flag 6 */
      bit_field_T UCEFLAG5    :1;  /* Uncorrectable error intr flag 5 */
      bit_field_T UCEFLAG4    :1;  /* Uncorrectable error intr flag 4 */
      bit_field_T UCEFLAG3    :1;  /* Uncorrectable error intr flag 3 */
      bit_field_T UCEFLAG2    :1;  /* Uncorrectable error intr flag 2 */
      bit_field_T UCEFLAG1    :1;  /* Uncorrectable error intr flag 1 */
   } bits;
   unsigned16_T  word;
} ITIFLAG_T;

/* MMUGCR ( Register)  */
typedef union
{
   struct
   {
      bit_field_T             :31;  /* Reserved bit 31 - 1             */
      bit_field_T MPMENA      :1;   /* Memory protection mode Enable   */
   } bits;
   unsigned32_T  word;
} MMUGCR_T;

/* PLL Control 1 Register (PLLCTL1) */
/* TEST bit is not difined as it is "reserved" and should be left at default and never written.*/
typedef union
{
   struct
   {
      bit_field_T ROS            :1;  /* Reset on PLL Slip           */
      bit_field_T BPOS           :2;   /* Bypass on PLL Slip         */
      bit_field_T PLLDIV         :5;   /* PLL Output Clock Divider   */
      bit_field_T ROF            :1;   /* Reset On Oscillator Fail   */
      bit_field_T                :1;   /* Reserved bit 22            */
      bit_field_T REFCLKDIV      :6;   /* Reference Clock Divider    */
      bit_field_T PLLMUL         :16;   /* PLL Multiplication Factor */
   } bits;
   unsigned32_T  word;
} PLLCTL1_T;

/* PLL Control 2 Register (PLLCTL2) */
typedef union
{
   struct
   {
      bit_field_T FMENA           :1;  /* Frequency Modulation Enable */
      bit_field_T SPREADINGRATE   :9;   /* Modulation Spreading Rate  */
      bit_field_T                 :1;   /* Reserved bit 21            */
      bit_field_T BWADJ           :9;   /* Bandwidth Adjustment       */
      bit_field_T ODPLL           :3;   /* Internal PLL Output Divider*/
      bit_field_T SPR_AMOUNT      :9;   /* Spreading Amount           */
   } bits;
   unsigned32_T  word;
} PLLCTL2_T;

/* Die Identification Register Lower Word (DIEIDL)  */
typedef union
{
   struct
   {
      bit_field_T DIEIDL     :32;   /* Lower word (310) of die ID information    */
   } bits;
   unsigned32_T  word;
} DIEIDL_T;

/* Die Identification Register Upper Word (DIEIDH)  */
typedef union
{
   struct
   {
      bit_field_T DIEIDH     :32;   /* Higher word (63 - 32) of die ID information   */
   } bits;
   unsigned32_T  word;
} DIEIDH_T;

/* Voltage Regulator Control Register (VRCTL)  */
typedef union
{
   struct
   {
      bit_field_T               :24;  /* Reserved bit 31 - 8              */
      bit_field_T VLPMENA       :4;   /* Reserved bit 7 - 5              */
      bit_field_T VSLEEPENA     :4;   /* DSP master DMA access enable    */
   } bits;
   unsigned32_T  word;
} VRCTL_T;

/* LPO/Clock Monitor Control Register (LPOMONCTL)  */
typedef union
{
   struct
   {
      bit_field_T               :7;   /* Reserved bit 31 - 25              */
      bit_field_T BIASENABLE    :1;   /* Bias enable                       */
      bit_field_T               :12;  /* Reserved bit 23 - 12              */
      bit_field_T HFTRIM        :4;   /* High frequency oscillator trim value   */
      bit_field_T               :4;   /* Reserved bit 7 - 4                */
      bit_field_T LFTRIM        :4;   /* Low frequency oscillator trim value    */
   } bits;
   unsigned32_T  word;
} LPOMONCTL_T;

/* Clock Test Register (CLKTEST)  */
typedef union
{
   struct
   {
      bit_field_T                     :5;   /* Reserved bit 7 - 5              */
      bit_field_T ALTLIMPCLOCKENABLE  :1;   /* Reserved bit 7 - 5              */
      bit_field_T RANGEDETCTRL        :1;   /* Reserved bit 7 - 5              */
      bit_field_T RANGEDETENSSEL      :1;   /* Reserved bit 7 - 5              */
      bit_field_T CLK_TEST_EN         :4;   /* Clock test enable            */
      bit_field_T                     :4;   /* Reserved bit 15 - 12         */
      bit_field_T SEL_GIO_PIN         :4;   /* Select the clock source valid signal or clock source at functional GIO pin  */
      bit_field_T                     :4;   /* Reserved bit 7 - 4           */
      bit_field_T SEL_ECP_PIN         :4;   /* Select the clock at ECP Pin  */
   } bits;
   unsigned32_T  word;
} CLKTEST_T;

/* BOOT ROM Select Register (BTRMSEL)  */
typedef union
{
   struct
   {
      bit_field_T BTRMSLRST           :16;  /* Boot ROM select and reset */
      bit_field_T                     :12;  /* Reserved bit 15 - 4       */
      bit_field_T BTRMSEL             :4;   /* Boot ROM select           */
   } bits;
   unsigned32_T  word;
} BTRMSEL_T;

/* Imprecise Fault Status Register (IMPFASTS)  */
typedef union
{
   struct
   {
      bit_field_T                     :8;   /* Reserved bit 31 - 24              */
      bit_field_T MASTERID            :8;   /* Master ID                         */
      bit_field_T                     :5;   /* Reserved bit 15 - 11              */
      bit_field_T EMIFA               :1;   /* EMIF imprecise abort              */
      bit_field_T NCBA                :1;   /* Non-cacheable, bufferable abort   */
      bit_field_T VBUSA               :1;   /* VBUS abort                        */
      bit_field_T                     :7;   /* Reserved bit 7 - 1                */
      bit_field_T ATYPE               :1;   /* Abort type                        */
   } bits;
   unsigned32_T  word;
} IMPFASTS_T;

/* Imprecise Fault Address Register (IMPFTADD)  */
typedef union
{
   struct
   {
      bit_field_T IMPFTADD            :32;  /* Fault address during imprecise abort is captured here    */
   } bits;
   unsigned32_T  word;
} IMPFTADD_T;

/* RAMGCR (RAM Control Register)  */
typedef union
{
   struct
   {
      bit_field_T             :12;  /* Reserved bit 31 - 20                 */
      bit_field_T RAM_DFT_EN  :4;   /* Functional mode RAM DFT port enable key */
      bit_field_T             :1;   /* Reserved bit 15                 */
      bit_field_T WST_AENA3   :1;   /* eSRAM3 Addr Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit 13                 */
      bit_field_T WST_DENA3   :1;   /* eSRAM3 Data Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit 11                 */
      bit_field_T WST_AENA2   :1;   /* eSRAM2 Addr Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit  9                 */
      bit_field_T WST_DENA2   :1;   /* eSRAM2 Data Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit  7                 */
      bit_field_T WST_AENA1   :1;   /* eSRAM1 Addr Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit  5                 */
      bit_field_T WST_DENA1   :1;   /* eSRAM1 Data Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit  3                 */
      bit_field_T WST_AENA0   :1;   /* eSRAM0 Addr Phase Wait State En */
      bit_field_T             :1;   /* Reserved bit  1                 */
      bit_field_T WST_DENA0   :1;   /* eSRAM0 Data Phase Wait State En */
   } bits;
   unsigned32_T  word;
} RAMGCR_T;

/* RCLKSRC (RTIclock Source Register)  */
typedef union
{
   struct
   {
      bit_field_T             :4;   /* Reserved bit 15 - 12            */
      bit_field_T RTI2DIV     :4;   /* RTI clock2 divider              */
      bit_field_T             :4;   /* Reserved bit 15 - 12            */
      bit_field_T RTI2SRC     :4;   /* RTICLK2 source                  */
      bit_field_T             :4;   /* Reserved bit 15 - 12            */
      bit_field_T RTI1DIV     :4;   /* RTI clock1 divider              */
      bit_field_T             :4;   /* Reserved bit 7 - 4              */
      bit_field_T RTI1SRC     :4;   /* RTICLK1 source                  */
   } bits;
   unsigned32_T  word;
} RCLKSRC_T;


/* SSIF (System Software Interrrupt Flag Register)  */
typedef union
{
   struct
   {
      bit_field_T             :28; /* Reserved bit 31 - 4              */
      bit_field_T SSI_FLAG4   :1;  /* System software interrupt flag 4 */
      bit_field_T SSI_FLAG3   :1;  /* System software interrupt flag 3 */
      bit_field_T SSI_FLAG2   :1;  /* System software interrupt flag 2 */
      bit_field_T SSI_FLAG1   :1;  /* System software interrupt flag 1 */
   } bits;
   unsigned32_T  word;
} SSIF_T;

/* System Software Interrupt Request 1 Register (SSIR1)  */
typedef union
{
   struct
   {
      bit_field_T            :16; /* Reserved bit 31 - 16                  */
      bit_field_T SSKEY1     :8;  /* System Software interrupt Request Key */
      bit_field_T SSDATA1    :8;  /* System Software interrupt Data        */
   } bits;
   unsigned32_T  word;
} SSIR1_T;

/* System Software Interrupt Request 2 Register (SSIR2)  */
typedef union
{
   struct
   {
      bit_field_T            :16; /* Reserved bit 31 - 16              */
      bit_field_T SSKEY2     :8;  /* System Software interrupt Request Key */
      bit_field_T SSDATA2    :8;  /* System Software interrupt Data        */
   } bits;
   unsigned32_T  word;
} SSIR2_T;

/* System Software Interrupt Request 3 Register (SSIR3)  */
typedef union
{
   struct
   {
      bit_field_T            :16; /* Reserved bit 31 - 16                  */
      bit_field_T SSKEY3     :8;  /* System Software interrupt Request Key */
      bit_field_T SSDATA3    :8;  /* System Software interrupt Data        */
   } bits;
   unsigned32_T  word;
} SSIR3_T;

/* System Software Interrupt Request 4 Register (SSIR4)  */
typedef union
{
   struct
   {
      bit_field_T            :16; /* Reserved bit 31 - 16                  */
      bit_field_T SSKEY4     :8;  /* System Software interrupt Request Key */
      bit_field_T SSDATA4    :8;  /* System Software interrupt Data        */
   } bits;
   unsigned32_T  word;
} SSIR4_T;

/* SSIVEC (Software Interrupt Vector Register)  */
typedef union
{
   struct
   {
      bit_field_T             :16;  /* Reserved bit 31 - 16            */
      bit_field_T SSIDATA     :8;   /* System Software Intr Data key   */
      bit_field_T SSIVECT     :8;   /* System Software Intr Vector     */
   } bits;
   unsigned32_T  word;
} SSIVEC_T;

/* SYSECR (System Exception Control Register)  */
typedef union
{
   struct
   {
      bit_field_T             :16; /* Reserved bit 31 - 16            */
      bit_field_T RESET       :2;  /* Software reset bits             */
      bit_field_T             :14; /* Reserved bit 13 - 0             */
   } bits;
   unsigned32_T  word;
} SYSECR_T;

/* SYSESR (System Reset Exception Status Register)  */
typedef union
{
   struct
   {
      bit_field_T             :16; /* Reserved bit 31 - 16            */
      bit_field_T PORRST      :1;  /* Power-up Reset flag             */
      bit_field_T OSCRST      :1;  /* Oscillator reset flag           */
      bit_field_T WDRST       :1;  /* Watchdog reset flag             */
      bit_field_T             :7;  /* Reserved bit 12 - 6             */
      bit_field_T CPURST      :1;  /* CPU reset flag                  */
      bit_field_T SWRST       :1;  /* Software Reset Flag             */
      bit_field_T EXTRST      :1;  /* External rst flag               */
      bit_field_T VSWRST      :1;  /* Wakeup from Vreg sleep rst flag */
      bit_field_T             :1;  /* Reserved bit 1                  */
      bit_field_T MPMODE      :1;  /* Memory protection mode status   */
   } bits;
   unsigned32_T  word;
} SYSESR_T;


/* SYSPC1 (System Pin Control Register1)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKFUN    :1;  /* ECPCLK Functional               */
   } bits;
   unsigned32_T  word;
} SYSPC1_T;

/* SYSPC2 (System Pin Control Register2)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKDIR    :1;  /* ECPCLK Direction                */
   } bits;
   unsigned32_T  word;
} SYSPC2_T;

/* SYSPC3 (System Pin Control Register3)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKDIN    :1;  /* ECPCLK Data Input               */
   } bits;
   unsigned32_T  word;
} SYSPC3_T;

/* SYSPC4 (System Pin Control Register4)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKDOUT   :1;  /* ECPCLK Data Output              */
   } bits;
   unsigned32_T  word;
} SYSPC4_T;

/* SYSPC5 (System Pin Control Register5)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKSET    :1;  /* ECPCLK DOUT Set                 */
   } bits;
   unsigned32_T  word;
} SYSPC5_T;

/* SYSPC6 (System Pin Control Register6)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKCLR    :1;  /* ECPCLK DOUT Clear               */
   } bits;
   unsigned32_T  word;
} SYSPC6_T;

/* SYSPC7 (System Pin Control Register7)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKODE    :1;  /* ECPCLK Open Drain Enable        */
   } bits;
   unsigned32_T  word;
} SYSPC7_T;

/* SYSPC8 (System Pin Control Register8)  */
typedef union
{
   struct
   {
      bit_field_T               :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKPDIS    :1;  /* ECPCLK Pull Up Disable          */
   } bits;
   unsigned32_T  word;
} SYSPC8_T;

/* SYSPC9 (System Pin Control Register9)  */
typedef union
{
   struct
   {
      bit_field_T              :31; /* Reserved bit 31 - 1             */
      bit_field_T ECPCLKPSEL   :1;  /* ECPCLK Pullup Selected          */
   } bits;
   unsigned32_T  word;
} SYSPC9_T;

/* SSW PLL BIST Control Register 1 (SSWPLL1) */
typedef union
{
   struct
   {
      bit_field_T                     :16; /* Reserved bit 31 - 16    */
      bit_field_T MOD_PH_CAP_INDEX    :8;  /* Mod ph caputure index   */
      bit_field_T                     :1;  /* Reserved bit 7          */
      bit_field_T COUNTER_READY       :1;  /* Counter read ready      */
      bit_field_T COUNTER_RESET       :1;  /* Counter reset           */
      bit_field_T COUNTER_EN          :1;  /* Counter enable          */
      bit_field_T TAP_COUNTER_DIS     :3;  /* Tap counter             */
      bit_field_T EXT_COUNTER_EN      :1;  /* Enables Counting on rising edge for all counters   */
   } bits;
   unsigned32_T  word;
} SSWPLL1_T;

/* SSW PLL BIST Control Register 2 (SSWPLL2) */
typedef union
{
   struct
   {
      bit_field_T SSW_CAPTURE_COUNT   :32;  /* Capture count.  */
   } bits;
   unsigned32_T  word;
} SSWPLL2_T;

/* SSW PLL BIST Control Register3 (SSWPLL3) */
typedef union
{
   struct
   {
      bit_field_T SSW_CLKOUT_COUNT   :32;  /* Value of CLKout count register.   */
   } bits;
   unsigned32_T  word;
} SSWPLL3_T;


/* VCLKASRC (Peripheral Asynchronous Clock Source Register)  */
typedef union
{
   struct
   {
      bit_field_T             :20;  /* Reserved bit 31 - 12            */
      bit_field_T VCLKA2S     :4;   /* Peripheral Async CLK 2 source   */
      bit_field_T             :4;   /* Reserved bit 11-8               */
      bit_field_T VCLKA1S     :4;   /* Peripheral Async CLK 1 source   */
   } bits;
   unsigned32_T  word;
} VCLKASRC_T;

/* DEV (Static Memory Control Registers)  */
typedef union
{
   struct
   {
      bit_field_T IC_DEVICE_CP15            :1;   /* CP15 CPU                   */
      bit_field_T IC_DEVICE_ID              :14;  /* Device ID                  */
      bit_field_T IC_DEVICE_TECH            :4;   /* Process tech of manuf dev  */
      bit_field_T IC_DEVICE_IO              :1;   /* Input/output voltage       */
      bit_field_T IC_DEVICE_PPAR            :1;   /* Peripheral parity          */
      bit_field_T IC_DEVICE_PROGRAM_PARITY  :2;   /* Parity of program memory   */
      bit_field_T IC_DEVICE_RECC            :1;   /* RAM ECC                    */
      bit_field_T IC_DEVICE_VERSION         :5;   /* Version of dev             */
      bit_field_T IC_DEVICE_PLATFORM_ID     :3;   /* TMS470Px ID is always 101b */
   } bits;
   unsigned32_T  word;
} DEV_T;




typedef struct
{

    SYSPC1_T        SYSPC1;

    SYSPC2_T        SYSPC2;

    SYSPC3_T        SYSPC3;

    SYSPC4_T        SYSPC4;

    SYSPC5_T        SYSPC5;

    SYSPC6_T        SYSPC6;

    SYSPC7_T        SYSPC7;

    SYSPC8_T        SYSPC8;

    SYSPC9_T        SYSPC9;

    SSWPLL1_T       SSWPLL1;

    SSWPLL2_T       SSWPLL2;

    SSWPLL3_T       SSWPLL3;

    CSDIS_T         CSDIS;

    CSDISSET_T      CSDISSET;

    CSDISCLR_T      CSDISCLR;

    CDDIS_T         CDDIS;

    CDDISSET_T      CDDISSET;

    CDDISCLR_T      CDDISCLR;

    GHVSRC_T        GHVSRC;

    VCLKASRC_T      VCLKASRC;

    RCLKSRC_T       RCLKSRC;

    CSVSTAT_T       CSVSTAT;

    MSTGCR_T        MSTGCR;

    MINITGCR_T      MINITGCR;

    MSINENA_T       MSINENA;

    MSTFAIL_T       MSTFAIL;

    MSTCGSTAT_T     MSTCGSTAT;

    MINISTAT_T      MINISTAT;

    PLLCTL1_T       PLLCTL1;

    PLLCTL2_T       PLLCTL2;

    bit_field_T     :32;

    DIEIDL_T        DIEIDL;

    DIEIDH_T        DIEIDH;

    VRCTL_T         VRCTL;

    LPOMONCTL_T     LPOMONCTL;

    CLKTEST_T       CLKTEST;

    unsigned32_T    HOLES[0x14/sizeof(unsigned32_T)];

    BTRMSEL_T       BTRMSEL;

    IMPFASTS_T      IMPFASTS;

    IMPFTADD_T      IMPFTADD;

    SSIR1_T         SSIR1;

    SSIR2_T         SSIR2;

    SSIR3_T         SSIR3;

    SSIR4_T         SSIR4;

    RAMGCR_T        RAMGCR;

    BMMCR1_T        BMMCR1;

    BMMCR2_T        BMMCR2;

    MMUGCR_T        MMUGCR;

    CLKCNTL_T       CLKCNTL;

    ECPCNTL_T       ECPCNTL;

    DSPGCR_T        DSPGCR;

    DEVCR1_T        DEVCR1;

    SYSECR_T        SYSECR;

    SYSESR_T        SYSESR;

    bit_field_T     :32;

    GLBSTAT_T       GLBSTAT;

    DEV_T           DEV;

    SSIVEC_T        SSIVEC;

    SSIF_T          SSIF;

//    SSIR_T          SSIR;  /* Linker is creating error. Don't know why? May be it is at MAX address */

} SYSREG_T;



/*********************************************************************************/

/* Peripheral Memory Protection Set Register 0 (PMPROTSET0)  */
typedef union
{
   struct
   {
      bit_field_T PCS31PROTSET               :1;   /* Peripheral memory frame protection set 31 */
      bit_field_T PCS30PROTSET               :1;   /* Peripheral memory frame protection set 30 */
      bit_field_T PCS29PROTSET               :1;   /* Peripheral memory frame protection set 29 */
      bit_field_T PCS28PROTSET               :1;   /* Peripheral memory frame protection set 28 */
      bit_field_T PCS27PROTSET               :1;   /* Peripheral memory frame protection set 27 */
      bit_field_T PCS26PROTSET               :1;   /* Peripheral memory frame protection set 26 */
      bit_field_T PCS25PROTSET               :1;   /* Peripheral memory frame protection set 25 */
      bit_field_T PCS24PROTSET               :1;   /* Peripheral memory frame protection set 24 */
      bit_field_T PCS23PROTSET               :1;   /* Peripheral memory frame protection set 23 */
      bit_field_T PCS22PROTSET               :1;   /* Peripheral memory frame protection set 22 */
      bit_field_T PCS21PROTSET               :1;   /* Peripheral memory frame protection set 21 */
      bit_field_T PCS20PROTSET               :1;   /* Peripheral memory frame protection set 20 */
      bit_field_T PCS19PROTSET               :1;   /* Peripheral memory frame protection set 19 */
      bit_field_T PCS18PROTSET               :1;   /* Peripheral memory frame protection set 18 */
      bit_field_T PCS17PROTSET               :1;   /* Peripheral memory frame protection set 17 */
      bit_field_T PCS16PROTSET               :1;   /* Peripheral memory frame protection set 16 */
      bit_field_T PCS15PROTSET               :1;   /* Peripheral memory frame protection set 15 */
      bit_field_T PCS14PROTSET               :1;   /* Peripheral memory frame protection set 14 */
      bit_field_T PCS13PROTSET               :1;   /* Peripheral memory frame protection set 13 */
      bit_field_T PCS12PROTSET               :1;   /* Peripheral memory frame protection set 12 */
      bit_field_T PCS11PROTSET               :1;   /* Peripheral memory frame protection set 11 */
      bit_field_T PCS10PROTSET               :1;   /* Peripheral memory frame protection set 10 */
      bit_field_T PCS09PROTSET               :1;   /* Peripheral memory frame protection set 9  */
      bit_field_T PCS08PROTSET               :1;   /* Peripheral memory frame protection set 8  */
      bit_field_T PCS07PROTSET               :1;   /* Peripheral memory frame protection set 7  */
      bit_field_T PCS06PROTSET               :1;   /* Peripheral memory frame protection set 6  */
      bit_field_T PCS05PROTSET               :1;   /* Peripheral memory frame protection set 5  */
      bit_field_T PCS04PROTSET               :1;   /* Peripheral memory frame protection set 4  */
      bit_field_T PCS03PROTSET               :1;   /* Peripheral memory frame protection set 3  */
      bit_field_T PCS02PROTSET               :1;   /* Peripheral memory frame protection set 2  */
      bit_field_T PCS01PROTSET               :1;   /* Peripheral memory frame protection set 1  */
      bit_field_T PCS00PROTSET               :1;   /* Peripheral memory frame protection set 0  */
   } bits;
   unsigned32_T  word;
} PMPROTSET0_T;

/* Peripheral Memory Protection Set Register 1 (PMPROTSET1)  */
typedef union
{
   struct
   {
      bit_field_T PCS63PROTSET               :1;   /* Peripheral memory frame protection set 63 */
      bit_field_T PCS62PROTSET               :1;   /* Peripheral memory frame protection set 62 */
      bit_field_T PCS61PROTSET               :1;   /* Peripheral memory frame protection set 61 */
      bit_field_T PCS60PROTSET               :1;   /* Peripheral memory frame protection set 60 */
      bit_field_T PCS59PROTSET               :1;   /* Peripheral memory frame protection set 59 */
      bit_field_T PCS58PROTSET               :1;   /* Peripheral memory frame protection set 58 */
      bit_field_T PCS57PROTSET               :1;   /* Peripheral memory frame protection set 57 */
      bit_field_T PCS56PROTSET               :1;   /* Peripheral memory frame protection set 56 */
      bit_field_T PCS55PROTSET               :1;   /* Peripheral memory frame protection set 55 */
      bit_field_T PCS54PROTSET               :1;   /* Peripheral memory frame protection set 54 */
      bit_field_T PCS53PROTSET               :1;   /* Peripheral memory frame protection set 53 */
      bit_field_T PCS52PROTSET               :1;   /* Peripheral memory frame protection set 52 */
      bit_field_T PCS51PROTSET               :1;   /* Peripheral memory frame protection set 51 */
      bit_field_T PCS50PROTSET               :1;   /* Peripheral memory frame protection set 50 */
      bit_field_T PCS49PROTSET               :1;   /* Peripheral memory frame protection set 49 */
      bit_field_T PCS48PROTSET               :1;   /* Peripheral memory frame protection set 48 */
      bit_field_T PCS47PROTSET               :1;   /* Peripheral memory frame protection set 47 */
      bit_field_T PCS46PROTSET               :1;   /* Peripheral memory frame protection set 46 */
      bit_field_T PCS45PROTSET               :1;   /* Peripheral memory frame protection set 45 */
      bit_field_T PCS44PROTSET               :1;   /* Peripheral memory frame protection set 44 */
      bit_field_T PCS43PROTSET               :1;   /* Peripheral memory frame protection set 43 */
      bit_field_T PCS42PROTSET               :1;   /* Peripheral memory frame protection set 42 */
      bit_field_T PCS41PROTSET               :1;   /* Peripheral memory frame protection set 41 */
      bit_field_T PCS40PROTSET               :1;   /* Peripheral memory frame protection set 40 */
      bit_field_T PCS39PROTSET               :1;   /* Peripheral memory frame protection set 39 */
      bit_field_T PCS38PROTSET               :1;   /* Peripheral memory frame protection set 38 */
      bit_field_T PCS37PROTSET               :1;   /* Peripheral memory frame protection set 37 */
      bit_field_T PCS36PROTSET               :1;   /* Peripheral memory frame protection set 36 */
      bit_field_T PCS35PROTSET               :1;   /* Peripheral memory frame protection set 35 */
      bit_field_T PCS34PROTSET               :1;   /* Peripheral memory frame protection set 34 */
      bit_field_T PCS33PROTSET               :1;   /* Peripheral memory frame protection set 33 */
      bit_field_T PCS32PROTSET               :1;   /* Peripheral memory frame protection set 32 */
   } bits;
   unsigned32_T  word;
} PMPROTSET1_T;

/* Peripheral Memory Protection Clear Register 0 (PMPROTCLR0)  */
typedef union
{
   struct
   {
      bit_field_T PCS31PROTCLR               :1;   /* Peripheral memory frame protection clear 31 */
      bit_field_T PCS30PROTCLR               :1;   /* Peripheral memory frame protection clear 30 */
      bit_field_T PCS29PROTCLR               :1;   /* Peripheral memory frame protection clear 29 */
      bit_field_T PCS28PROTCLR               :1;   /* Peripheral memory frame protection clear 28 */
      bit_field_T PCS27PROTCLR               :1;   /* Peripheral memory frame protection clear 27 */
      bit_field_T PCS26PROTCLR               :1;   /* Peripheral memory frame protection clear 26 */
      bit_field_T PCS25PROTCLR               :1;   /* Peripheral memory frame protection clear 25 */
      bit_field_T PCS24PROTCLR               :1;   /* Peripheral memory frame protection clear 24 */
      bit_field_T PCS23PROTCLR               :1;   /* Peripheral memory frame protection clear 23 */
      bit_field_T PCS22PROTCLR               :1;   /* Peripheral memory frame protection clear 22 */
      bit_field_T PCS21PROTCLR               :1;   /* Peripheral memory frame protection clear 21 */
      bit_field_T PCS20PROTCLR               :1;   /* Peripheral memory frame protection clear 20 */
      bit_field_T PCS19PROTCLR               :1;   /* Peripheral memory frame protection clear 19 */
      bit_field_T PCS18PROTCLR               :1;   /* Peripheral memory frame protection clear 18 */
      bit_field_T PCS17PROTCLR               :1;   /* Peripheral memory frame protection clear 17 */
      bit_field_T PCS16PROTCLR               :1;   /* Peripheral memory frame protection clear 16 */
      bit_field_T PCS15PROTCLR               :1;   /* Peripheral memory frame protection clear 15 */
      bit_field_T PCS14PROTCLR               :1;   /* Peripheral memory frame protection clear 14 */
      bit_field_T PCS13PROTCLR               :1;   /* Peripheral memory frame protection clear 13 */
      bit_field_T PCS12PROTCLR               :1;   /* Peripheral memory frame protection clear 12 */
      bit_field_T PCS11PROTCLR               :1;   /* Peripheral memory frame protection clear 11 */
      bit_field_T PCS10PROTCLR               :1;   /* Peripheral memory frame protection clear 10 */
      bit_field_T PCS09PROTCLR               :1;   /* Peripheral memory frame protection clear 9  */
      bit_field_T PCS08PROTCLR               :1;   /* Peripheral memory frame protection clear 8  */
      bit_field_T PCS07PROTCLR               :1;   /* Peripheral memory frame protection clear 7  */
      bit_field_T PCS06PROTCLR               :1;   /* Peripheral memory frame protection clear 6  */
      bit_field_T PCS05PROTCLR               :1;   /* Peripheral memory frame protection clear 5  */
      bit_field_T PCS04PROTCLR               :1;   /* Peripheral memory frame protection clear 4  */
      bit_field_T PCS03PROTCLR               :1;   /* Peripheral memory frame protection clear 3  */
      bit_field_T PCS02PROTCLR               :1;   /* Peripheral memory frame protection clear 2  */
      bit_field_T PCS01PROTCLR               :1;   /* Peripheral memory frame protection clear 1  */
      bit_field_T PCS00PROTCLR               :1;   /* Peripheral memory frame protection clear 0  */
   } bits;
   unsigned32_T  word;
} PMPROTCLR0_T;

/* Peripheral Memory Protection Clear Register 1 (PMPROTCLR1)  */
typedef union
{
   struct
   {
      bit_field_T PCS63PROTCLR               :1;   /* Peripheral memory frame protection clear 63 */
      bit_field_T PCS62PROTCLR               :1;   /* Peripheral memory frame protection clear 62 */
      bit_field_T PCS61PROTCLR               :1;   /* Peripheral memory frame protection clear 61 */
      bit_field_T PCS60PROTCLR               :1;   /* Peripheral memory frame protection clear 60 */
      bit_field_T PCS59PROTCLR               :1;   /* Peripheral memory frame protection clear 59 */
      bit_field_T PCS58PROTCLR               :1;   /* Peripheral memory frame protection clear 58 */
      bit_field_T PCS57PROTCLR               :1;   /* Peripheral memory frame protection clear 57 */
      bit_field_T PCS56PROTCLR               :1;   /* Peripheral memory frame protection clear 56 */
      bit_field_T PCS55PROTCLR               :1;   /* Peripheral memory frame protection clear 55 */
      bit_field_T PCS54PROTCLR               :1;   /* Peripheral memory frame protection clear 54 */
      bit_field_T PCS53PROTCLR               :1;   /* Peripheral memory frame protection clear 53 */
      bit_field_T PCS52PROTCLR               :1;   /* Peripheral memory frame protection clear 52 */
      bit_field_T PCS51PROTCLR               :1;   /* Peripheral memory frame protection clear 51 */
      bit_field_T PCS50PROTCLR               :1;   /* Peripheral memory frame protection clear 50 */
      bit_field_T PCS49PROTCLR               :1;   /* Peripheral memory frame protection clear 49 */
      bit_field_T PCS48PROTCLR               :1;   /* Peripheral memory frame protection clear 48 */
      bit_field_T PCS47PROTCLR               :1;   /* Peripheral memory frame protection clear 47 */
      bit_field_T PCS46PROTCLR               :1;   /* Peripheral memory frame protection clear 46 */
      bit_field_T PCS45PROTCLR               :1;   /* Peripheral memory frame protection clear 45 */
      bit_field_T PCS44PROTCLR               :1;   /* Peripheral memory frame protection clear 44 */
      bit_field_T PCS43PROTCLR               :1;   /* Peripheral memory frame protection clear 43 */
      bit_field_T PCS42PROTCLR               :1;   /* Peripheral memory frame protection clear 42 */
      bit_field_T PCS41PROTCLR               :1;   /* Peripheral memory frame protection clear 41 */
      bit_field_T PCS40PROTCLR               :1;   /* Peripheral memory frame protection clear 40 */
      bit_field_T PCS39PROTCLR               :1;   /* Peripheral memory frame protection clear 39 */
      bit_field_T PCS38PROTCLR               :1;   /* Peripheral memory frame protection clear 38 */
      bit_field_T PCS37PROTCLR               :1;   /* Peripheral memory frame protection clear 37 */
      bit_field_T PCS36PROTCLR               :1;   /* Peripheral memory frame protection clear 36 */
      bit_field_T PCS35PROTCLR               :1;   /* Peripheral memory frame protection clear 35 */
      bit_field_T PCS34PROTCLR               :1;   /* Peripheral memory frame protection clear 34 */
      bit_field_T PCS33PROTCLR               :1;   /* Peripheral memory frame protection clear 33 */
      bit_field_T PCS32PROTCLR               :1;   /* Peripheral memory frame protection clear 32 */
   } bits;
   unsigned32_T  word;
} PMPROTCLR1_T;

/* Peripheral Protection Set Register 0 (PPROTSET0)  */
typedef union
{
   struct
   {
      bit_field_T PS07QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS07QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS07QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS07QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS06QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS06QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS06QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS06QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS05QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS05QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS05QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS05QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS04QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS04QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS04QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS04QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS03QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS03QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS03QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS03QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS02QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS02QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS02QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS02QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS01QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS01QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS01QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS01QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS00QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS00QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS00QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS00QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
   } bits;
   unsigned32_T  word;
} PPROTSET0_T;

/* Peripheral Protection Set Register 1 (PPROTSET1)  */
typedef union
{
   struct
   {
      bit_field_T PS15QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS15QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS15QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS15QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS14QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS14QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS14QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS14QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS13QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS13QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS13QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS13QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS12QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS12QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS12QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS12QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS11QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS11QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS11QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS11QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS10QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS10QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS10QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS10QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS09QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS09QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS09QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS09QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS08QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS08QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS08QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS08QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
   } bits;
   unsigned32_T  word;
} PPROTSET1_T;

/* Peripheral Protection Set Register 2 (PPROTSET2)  */
typedef union
{
   struct
   {
      bit_field_T PS23QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS23QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS23QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS23QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS22QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS22QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS22QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS22QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS21QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS21QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS21QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS21QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS20QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS20QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS20QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS20QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS19QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS19QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS19QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS19QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS18QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS18QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS18QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS18QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS17QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS17QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS17QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS17QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS16QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS16QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS16QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS16QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
   } bits;
   unsigned32_T  word;
} PPROTSET2_T;

/* Peripheral Protection Set Register 3 (PPROTSET3)  */
typedef union
{
   struct
   {
      bit_field_T PS31QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS31QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS31QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS31QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS30QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS30QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS30QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS30QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS29QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS29QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS29QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS29QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS28QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS28QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS28QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS28QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS27QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS27QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS27QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS27QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS26QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS26QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS26QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS26QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS25QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS25QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS25QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS25QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS24QUAD3PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS24QUAD2PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS24QUAD1PROTSET               :1;   /* Peripheral select quadrant protection set */
      bit_field_T PS24QUAD0PROTSET               :1;   /* Peripheral select quadrant protection set */
   } bits;
   unsigned32_T  word;
} PPROTSET3_T;

/* Peripheral Protection Clear Register 0 (PPROTCLR0)  */
typedef union
{
   struct
   {
      bit_field_T PS07QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS07QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS07QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS07QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS06QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS06QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS06QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS06QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS05QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS05QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS05QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS05QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS04QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS04QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS04QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS04QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS03QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS03QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS03QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS03QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS02QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS02QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS02QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS02QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS01QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS01QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS01QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS01QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS00QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS00QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS00QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS00QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
   } bits;
   unsigned32_T  word;
} PPROTCLR0_T;

/* Peripheral Protection Clear Register 1 (PPROTCLR1)  */
typedef union
{
   struct
   {
      bit_field_T PS15QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS15QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS15QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS15QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS14QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS14QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS14QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS14QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS13QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS13QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS13QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS13QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS12QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS12QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS12QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS12QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS11QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS11QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS11QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS11QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS10QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS10QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS10QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS10QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS09QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS09QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS09QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS09QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS08QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS08QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS08QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS08QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
   } bits;
   unsigned32_T  word;
} PPROTCLR1_T;

/* Peripheral Protection Clear Register 2 (PPROTCLR2)  */
typedef union
{
   struct
   {
      bit_field_T PS23QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS23QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS23QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS23QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS22QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS22QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS22QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS22QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS21QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS21QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS21QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS21QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS20QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS20QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS20QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS20QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS19QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS19QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS19QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS19QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS18QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS18QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS18QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS18QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS17QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS17QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS17QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS17QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS16QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS16QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS16QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS16QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
   } bits;
   unsigned32_T  word;
} PPROTCLR2_T;

/* Peripheral Protection Clear Register 3 (PPROTCLR3)  */
typedef union
{
   struct
   {
      bit_field_T PS31QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS31QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS31QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS31QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS30QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS30QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS30QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS30QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS29QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS29QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS29QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS29QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS28QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS28QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS28QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS28QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS27QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS27QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS27QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS27QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS26QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS26QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS26QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS26QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS25QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS25QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS25QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS25QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS24QUAD3PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS24QUAD2PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS24QUAD1PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
      bit_field_T PS24QUAD0PROTCLR               :1;   /* Peripheral select quadrant protection clear*/
   } bits;
   unsigned32_T  word;
} PPROTCLR3_T;

/* Peripheral Memory Power-Down Set Register 0 (PCSPWRDWNSET0)  */
typedef union
{
   struct
   {
      bit_field_T PCS31PWRDWNSET               :1;   /* Peripheral memory clock power-down set 31*/
      bit_field_T PCS30PWRDWNSET               :1;   /* Peripheral memory clock power-down set 30*/
      bit_field_T PCS29PWRDWNSET               :1;   /* Peripheral memory clock power-down set 29*/
      bit_field_T PCS28PWRDWNSET               :1;   /* Peripheral memory clock power-down set 28*/
      bit_field_T PCS27PWRDWNSET               :1;   /* Peripheral memory clock power-down set 27*/
      bit_field_T PCS26PWRDWNSET               :1;   /* Peripheral memory clock power-down set 26*/
      bit_field_T PCS25PWRDWNSET               :1;   /* Peripheral memory clock power-down set 25*/
      bit_field_T PCS24PWRDWNSET               :1;   /* Peripheral memory clock power-down set 24*/
      bit_field_T PCS23PWRDWNSET               :1;   /* Peripheral memory clock power-down set 23*/
      bit_field_T PCS22PWRDWNSET               :1;   /* Peripheral memory clock power-down set 22*/
      bit_field_T PCS21PWRDWNSET               :1;   /* Peripheral memory clock power-down set 21*/
      bit_field_T PCS20PWRDWNSET               :1;   /* Peripheral memory clock power-down set 20*/
      bit_field_T PCS19PWRDWNSET               :1;   /* Peripheral memory clock power-down set 19*/
      bit_field_T PCS18PWRDWNSET               :1;   /* Peripheral memory clock power-down set 18*/
      bit_field_T PCS17PWRDWNSET               :1;   /* Peripheral memory clock power-down set 17*/
      bit_field_T PCS16PWRDWNSET               :1;   /* Peripheral memory clock power-down set 16*/
      bit_field_T PCS15PWRDWNSET               :1;   /* Peripheral memory clock power-down set 15*/
      bit_field_T PCS14PWRDWNSET               :1;   /* Peripheral memory clock power-down set 14*/
      bit_field_T PCS13PWRDWNSET               :1;   /* Peripheral memory clock power-down set 13*/
      bit_field_T PCS12PWRDWNSET               :1;   /* Peripheral memory clock power-down set 12*/
      bit_field_T PCS11PWRDWNSET               :1;   /* Peripheral memory clock power-down set 11*/
      bit_field_T PCS10PWRDWNSET               :1;   /* Peripheral memory clock power-down set 10*/
      bit_field_T PCS09PWRDWNSET               :1;   /* Peripheral memory clock power-down set 9 */
      bit_field_T PCS08PWRDWNSET               :1;   /* Peripheral memory clock power-down set 8 */
      bit_field_T PCS07PWRDWNSET               :1;   /* Peripheral memory clock power-down set 7 */
      bit_field_T PCS06PWRDWNSET               :1;   /* Peripheral memory clock power-down set 6 */
      bit_field_T PCS05PWRDWNSET               :1;   /* Peripheral memory clock power-down set 5 */
      bit_field_T PCS04PWRDWNSET               :1;   /* Peripheral memory clock power-down set 4 */
      bit_field_T PCS03PWRDWNSET               :1;   /* Peripheral memory clock power-down set 3 */
      bit_field_T PCS02PWRDWNSET               :1;   /* Peripheral memory clock power-down set 2 */
      bit_field_T PCS01PWRDWNSET               :1;   /* Peripheral memory clock power-down set 1 */
      bit_field_T PCS00PWRDWNSET               :1;   /* Peripheral memory clock power-down set 0 */
   } bits;
   unsigned32_T  word;
} PCSPWRDWNSET0_T;

/* Peripheral Memory Power-Down Set Register 1 (PCSPWRDWNSET1)  */
typedef union
{
   struct
   {
      bit_field_T PCS63PWRDWNSET               :1;   /* Peripheral memory clock power-down set 63*/
      bit_field_T PCS62PWRDWNSET               :1;   /* Peripheral memory clock power-down set 62*/
      bit_field_T PCS61PWRDWNSET               :1;   /* Peripheral memory clock power-down set 61*/
      bit_field_T PCS60PWRDWNSET               :1;   /* Peripheral memory clock power-down set 60*/
      bit_field_T PCS59PWRDWNSET               :1;   /* Peripheral memory clock power-down set 59*/
      bit_field_T PCS58PWRDWNSET               :1;   /* Peripheral memory clock power-down set 58*/
      bit_field_T PCS57PWRDWNSET               :1;   /* Peripheral memory clock power-down set 57*/
      bit_field_T PCS56PWRDWNSET               :1;   /* Peripheral memory clock power-down set 56*/
      bit_field_T PCS55PWRDWNSET               :1;   /* Peripheral memory clock power-down set 55*/
      bit_field_T PCS54PWRDWNSET               :1;   /* Peripheral memory clock power-down set 54*/
      bit_field_T PCS53PWRDWNSET               :1;   /* Peripheral memory clock power-down set 53*/
      bit_field_T PCS52PWRDWNSET               :1;   /* Peripheral memory clock power-down set 52*/
      bit_field_T PCS51PWRDWNSET               :1;   /* Peripheral memory clock power-down set 51*/
      bit_field_T PCS50PWRDWNSET               :1;   /* Peripheral memory clock power-down set 50*/
      bit_field_T PCS49PWRDWNSET               :1;   /* Peripheral memory clock power-down set 49*/
      bit_field_T PCS48PWRDWNSET               :1;   /* Peripheral memory clock power-down set 48*/
      bit_field_T PCS47PWRDWNSET               :1;   /* Peripheral memory clock power-down set 47*/
      bit_field_T PCS46PWRDWNSET               :1;   /* Peripheral memory clock power-down set 46*/
      bit_field_T PCS45PWRDWNSET               :1;   /* Peripheral memory clock power-down set 45*/
      bit_field_T PCS44PWRDWNSET               :1;   /* Peripheral memory clock power-down set 44*/
      bit_field_T PCS43PWRDWNSET               :1;   /* Peripheral memory clock power-down set 43*/
      bit_field_T PCS42PWRDWNSET               :1;   /* Peripheral memory clock power-down set 42*/
      bit_field_T PCS41PWRDWNSET               :1;   /* Peripheral memory clock power-down set 41*/
      bit_field_T PCS40PWRDWNSET               :1;   /* Peripheral memory clock power-down set 40*/
      bit_field_T PCS39PWRDWNSET               :1;   /* Peripheral memory clock power-down set 39*/
      bit_field_T PCS38PWRDWNSET               :1;   /* Peripheral memory clock power-down set 38*/
      bit_field_T PCS37PWRDWNSET               :1;   /* Peripheral memory clock power-down set 37*/
      bit_field_T PCS36PWRDWNSET               :1;   /* Peripheral memory clock power-down set 36*/
      bit_field_T PCS35PWRDWNSET               :1;   /* Peripheral memory clock power-down set 35*/
      bit_field_T PCS34PWRDWNSET               :1;   /* Peripheral memory clock power-down set 34*/
      bit_field_T PCS33PWRDWNSET               :1;   /* Peripheral memory clock power-down set 33*/
      bit_field_T PCS32PWRDWNSET               :1;   /* Peripheral memory clock power-down set 32*/
   } bits;
   unsigned32_T  word;
} PCSPWRDWNSET1_T;

/* Peripheral Memory Power-Down Clear Register 0 (PCSPWRDWNCLR0)  */
typedef union
{
   struct
   {
      bit_field_T PCS31PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 31*/
      bit_field_T PCS30PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 30*/
      bit_field_T PCS29PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 29*/
      bit_field_T PCS28PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 28*/
      bit_field_T PCS27PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 27*/
      bit_field_T PCS26PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 26*/
      bit_field_T PCS25PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 25*/
      bit_field_T PCS24PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 24*/
      bit_field_T PCS23PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 23*/
      bit_field_T PCS22PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 22*/
      bit_field_T PCS21PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 21*/
      bit_field_T PCS20PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 20*/
      bit_field_T PCS19PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 19*/
      bit_field_T PCS18PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 18*/
      bit_field_T PCS17PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 17*/
      bit_field_T PCS16PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 16*/
      bit_field_T PCS15PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 15*/
      bit_field_T PCS14PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 14*/
      bit_field_T PCS13PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 13*/
      bit_field_T PCS12PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 12*/
      bit_field_T PCS11PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 11*/
      bit_field_T PCS10PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 10*/
      bit_field_T PCS09PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 9 */
      bit_field_T PCS08PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 8 */
      bit_field_T PCS07PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 7 */
      bit_field_T PCS06PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 6 */
      bit_field_T PCS05PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 5 */
      bit_field_T PCS04PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 4 */
      bit_field_T PCS03PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 3 */
      bit_field_T PCS02PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 2 */
      bit_field_T PCS01PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 1 */
      bit_field_T PCS00PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 0 */
   } bits;
   unsigned32_T  word;
} PCSPWRDWNCLR0_T;

/* Peripheral Memory Power-Down Clear Register 1 (PCSPWRDWNCLR1)  */
typedef union
{
   struct
   {
      bit_field_T PCS63PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 63*/
      bit_field_T PCS62PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 62*/
      bit_field_T PCS61PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 61*/
      bit_field_T PCS60PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 60*/
      bit_field_T PCS59PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 59*/
      bit_field_T PCS58PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 58*/
      bit_field_T PCS57PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 57*/
      bit_field_T PCS56PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 56*/
      bit_field_T PCS55PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 55*/
      bit_field_T PCS54PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 54*/
      bit_field_T PCS53PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 53*/
      bit_field_T PCS52PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 52*/
      bit_field_T PCS51PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 51*/
      bit_field_T PCS50PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 50*/
      bit_field_T PCS49PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 49*/
      bit_field_T PCS48PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 48*/
      bit_field_T PCS47PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 47*/
      bit_field_T PCS46PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 46*/
      bit_field_T PCS45PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 45*/
      bit_field_T PCS44PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 44*/
      bit_field_T PCS43PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 43*/
      bit_field_T PCS42PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 42*/
      bit_field_T PCS41PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 41*/
      bit_field_T PCS40PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 40*/
      bit_field_T PCS39PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 39*/
      bit_field_T PCS38PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 38*/
      bit_field_T PCS37PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 37*/
      bit_field_T PCS36PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 36*/
      bit_field_T PCS35PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 35*/
      bit_field_T PCS34PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 34*/
      bit_field_T PCS33PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 33*/
      bit_field_T PCS32PWRDWNCLR               :1;   /* Peripheral memory clock power-down clear 32*/
   } bits;
   unsigned32_T  word;
} PCSPWRDWNCLR1_T;

/* Peripheral Power-Down Set Register 0 (PSPWRDWNSET0)  */
typedef union
{
   struct
   {
      bit_field_T PS07QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS07QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS07QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS07QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS06QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS06QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS06QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS06QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS05QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS05QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS05QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS05QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS04QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS04QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS04QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS04QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS03QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS03QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS03QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS03QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS02QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS02QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS02QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS02QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS01QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS01QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS01QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS01QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS00QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS00QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS00QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS00QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNSET0_T;

/* Peripheral Power-Down Set Register 1 (PSPWRDWNSET1)  */
typedef union
{
   struct
   {
      bit_field_T PS15QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS15QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS15QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS15QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS14QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS14QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS14QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS14QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS13QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS13QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS13QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS13QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS12QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS12QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS12QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS12QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS11QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS11QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS11QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS11QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS10QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS10QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS10QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS10QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS09QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS09QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS09QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS09QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS08QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS08QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS08QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS08QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNSET1_T;

/* Peripheral Power-Down Set Register 2 (PSPWRDWNSET2)  */
typedef union
{
   struct
   {
      bit_field_T PS23QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS23QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS23QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS23QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS22QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS22QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS22QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS22QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS21QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS21QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS21QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS21QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS20QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS20QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS20QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS20QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS19QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS19QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS19QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS19QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS18QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS18QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS18QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS18QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS17QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS17QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS17QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS17QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS16QUAD3PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS16QUAD2PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS16QUAD1PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS16QUAD0PWRDWNSET            :1;   /* Peripheral select quadrant clock power-down set*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNSET2_T;

/* Peripheral Power-Down Set Register 3 (PSPWRDWNSET3)  */
typedef union
{
   struct
   {
      bit_field_T PS31QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS31QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS31QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS31QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS30QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS30QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS30QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS30QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS29QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS29QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS29QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS29QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS28QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS28QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS28QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS28QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS27QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS27QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS27QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS27QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS26QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS26QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS26QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS26QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS25QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS25QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS25QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS25QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS24QUAD3PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS24QUAD2PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS24QUAD1PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
      bit_field_T PS24QUAD0PWRDWNSET               :1;   /* Peripheral select quadrant clock power-down set*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNSET3_T;

/* Peripheral Power-Down Clear Register 0 (PSPWRDWNCLR0)  */
typedef union
{
   struct
   {
      bit_field_T PS07QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS07QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS07QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS07QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS06QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS06QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS06QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS06QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS05QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS05QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS05QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS05QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS04QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS04QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS04QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS04QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS03QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS03QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS03QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS03QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS02QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS02QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS02QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS02QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS01QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS01QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS01QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS01QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS00QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS00QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS00QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS00QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNCLR0_T;

/* Peripheral Power-Down Clear Register 1 (PSPWRDWNCLR1)  */
typedef union
{
   struct
   {
      bit_field_T PS15QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS15QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS15QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS15QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS14QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS14QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS14QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS14QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS13QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS13QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS13QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS13QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS12QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS12QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS12QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS12QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS11QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS11QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS11QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS11QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS10QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS10QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS10QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS10QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS09QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS09QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS09QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS09QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS08QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS08QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS08QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS08QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNCLR1_T;

/* Peripheral Power-Down Clear Register 2 (PSPWRDWNSET2)  */
typedef union
{
   struct
   {
      bit_field_T PS23QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS23QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS23QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS23QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS22QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS22QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS22QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS22QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS21QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS21QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS21QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS21QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS20QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS20QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS20QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS20QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS19QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS19QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS19QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS19QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS18QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS18QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS18QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS18QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS17QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS17QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS17QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS17QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS16QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS16QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS16QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS16QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNCLR2_T;

/* Peripheral Power-Down Clear Register 3 (PSPWRDWNSET3)  */
typedef union
{
   struct
   {
      bit_field_T PS31QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS31QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS31QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS31QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS30QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS30QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS30QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS30QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS29QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS29QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS29QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS29QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS28QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS28QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS28QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS28QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS27QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS27QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS27QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS27QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS26QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS26QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS26QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS26QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS25QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS25QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS25QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS25QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS24QUAD3PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS24QUAD2PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS24QUAD1PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
      bit_field_T PS24QUAD0PWRDWNCLR               :1;   /* Peripheral select quadrant clock power-down clear*/
   } bits;
   unsigned32_T  word;
} PSPWRDWNCLR3_T;

typedef struct
{
    PMPROTSET0_T      PMPROTSET0;
    PMPROTSET1_T      PMPROTSET1;

    bit_field_T       :32;
    bit_field_T       :32;

    PMPROTCLR0_T      PMPROTCLR0;
    PMPROTCLR1_T      PMPROTCLR1;

    bit_field_T       :32;
    bit_field_T       :32;


    PPROTSET0_T       PPROTSET0;
    PPROTSET1_T       PPROTSET1;
    PPROTSET2_T       PPROTSET2;
    PPROTSET3_T       PPROTSET3;

    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;

    PPROTCLR0_T       PPROTCLR0;
    PPROTCLR1_T       PPROTCLR1;
    PPROTCLR2_T       PPROTCLR2;
    PPROTCLR3_T       PPROTCLR3;

    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;

    PCSPWRDWNSET0_T   PCSPWRDWNSET0;
    PCSPWRDWNSET1_T   PCSPWRDWNSET1;

    bit_field_T       :32;
    bit_field_T       :32;

    PCSPWRDWNCLR0_T   PCSPWRDWNCLR0;
    PCSPWRDWNCLR1_T   PCSPWRDWNCLR1;

    bit_field_T       :32;
    bit_field_T       :32;

    PSPWRDWNSET0_T    PSPWRDWNSET0;
    PSPWRDWNSET1_T    PSPWRDWNSET1;
    PSPWRDWNSET2_T    PSPWRDWNSET2;
    PSPWRDWNSET3_T    PSPWRDWNSET3;

    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;
    bit_field_T       :32;

    PSPWRDWNCLR0_T    PSPWRDWNCLR0;
    PSPWRDWNCLR1_T    PSPWRDWNCLR1;
    PSPWRDWNCLR2_T    PSPWRDWNCLR2;
    PSPWRDWNCLR3_T    PSPWRDWNCLR3;

} PCRREG_T;


/*********************************************************************************/

/* VIM RAM Parity Flag Register (PARFLG) */
typedef union
{
   struct
   {
      bit_field_T                :31;   /* Reserved bit 31 - 1   */
      bit_field_T PARFLG         :1;    /* Parity error flag     */
   } bits;
   unsigned32_T  word;
} VIM_PARFLG_T;

/* VIM RAM Parity Control Register (PARCTL) */
typedef union
{
   struct
   {
      bit_field_T                   :23;  /* Reserved bit 31 - 9     */
      bit_field_T TEST              :1;   /* maps the parity bits into the VIM RAM    */
      bit_field_T                   :4;   /* Reserved bit 7 - 4      */
      bit_field_T PARENA            :4;   /* VIM parity enable       */
   } bits;
   unsigned32_T  word;
} VIM_PARCTL_T;

/* Address Parity Error Register (ADDERR) */
typedef union
{
   struct
   {

      bit_field_T VIM_RAM_OFFSET       :23;  /* VIM RAM offset                 */
      bit_field_T ADDERR               :7;   /* Address parity error register  */
      bit_field_T WORD_OFFSET          :2;   /* Word offset                    */
   } bits;
   unsigned32_T  word;
} VIM_ADDERR_T;

/* Fall-Back Address Parity Error Register (FBPARERR) */
typedef union
{
   struct
   {
      bit_field_T FBPARERR               :32;   /* Fall back address parity error  */
   } bits;
   unsigned32_T  word;
} VIM_FBPARERR_T;

/* IRQ Index Offset Vector Register (IRQIVEC) */
typedef union
{
   struct
   {
      bit_field_T                       :24;  /* Reserved bit 31 - 8         */
      bit_field_T IRQIVEC               :8;   /* IRQ index vector            */
   } bits;
   unsigned32_T  word;
} VIM_IRQIVEC_T;

/* FIQ Index Offset Vector Register (FIQIVEC) */
typedef union
{
   struct
   {
      bit_field_T                       :24;  /* Reserved bit 31 - 8         */
      bit_field_T FIQIVEC               :8;   /* FIQ index offset vector     */
   } bits;
   unsigned32_T  word;
} VIM_FIQIVEC_T;

/* FIQ/IRQ Program Control Registers[0:1] (FIRQPR[0:1]) */
typedef union
{
   struct
   {
      bit_field_T FIRQPR_31_00          :32;   /* FIQ/IRQ program control bits[31:0]    */
      bit_field_T                       :16;   /* Reserved bits 48 - 63                 */
      bit_field_T FIRQPR_47_32          :16;   /* FIQ/IRQ program control bits[47:32]   */
   } bits;
   unsigned32_T  word[2];
} VIM_FIRQPR_T;

/* Pending Interrupt Read Location Registers[0:1] (INTREQ[0:1]) */
typedef union
{
   struct
   {
      bit_field_T INTREQ_31_00          :32;   /* Pending interrupt bits[31:0]  */
      bit_field_T                       :16;   /* Reserved bits 48 - 63         */
      bit_field_T INTREQ_47_32          :16;   /* Pending interrupt bits[47:32] */
   } bits;
   unsigned32_T  word[2];
} VIM_INTREQ_T;

/* Interrupt Mask Set Registers[0:1] (REQMSKSET[0:1]) */
typedef union
{
   struct
   {
      bit_field_T REQMSKSET_31_00       :32;   /* Request mask set bits[31:0]   */
      bit_field_T                       :16;   /* Reserved bits 48 - 63         */
      bit_field_T REQMSKSET_47_32       :16;   /* Request mask set bits[47:32]  */
   } bits;
   unsigned32_T  word[2];
} VIM_REQMSKSET_T;

/* Interrupt Mask Clear Registers[0:1] (REQMSKCLR[0:1]) */
typedef union
{
   struct
   {
      bit_field_T REQMSKCLR_31_00       :32;   /* Request mask clear bits[31:0]  */
      bit_field_T                       :16;   /* Reserved bits 48 - 63          */
      bit_field_T REQMSKCLR_47_32       :16;   /* Request mask clear bits[47:32] */
   } bits;
   unsigned32_T  word[2];
} VIM_REQMSKCLR_T;

/* Wake-Up Mask Set Registers[0:1] (WAKMSKSET[0:1]) */
typedef union
{
   struct
   {
      bit_field_T WAKMSKSET_31_00       :32;   /* Wake-up mask set bits[31:0]   */
      bit_field_T                       :16;   /* Reserved bits 48 - 63         */
      bit_field_T WAKMSKSET_47_32       :16;   /* Wake-up mask set bits[47:32]  */
   } bits;
   unsigned32_T  word[2];
} VIM_WAKMSKSET_T;

/* Wake-Up Mask Clear Registers[0:1] (WAKMSKCLR[0:1]) */
typedef union
{
   struct
   {
      bit_field_T WAKMSKSET_31_00       :32;   /* Wake-up mask clear bits[31:0]  */
      bit_field_T                       :16;   /* Reserved bits 48 - 63          */
      bit_field_T WAKMSKSET_47_32       :16;   /* Wake-up mask clear bits[47:32] */
   } bits;
   unsigned32_T  word[2];
} VIM_WAKMSKCLR_T;

/* IRQ Interrupt Vector Register (IRQVECREG) */
typedef union
{
   struct
   {
      bit_field_T IRQVECREG          :32;   /* IRQ interrupt vector register  */
   } bits;
   unsigned32_T  word;
} VIM_IRQVECREG_T;

/* FIQ Interrupt Vector Register (FIQVECREG) */
typedef union
{
   struct
   {
      bit_field_T FIQVECREG          :32;   /* FIQ interrupt vector register  */
   } bits;
   unsigned32_T  word;
} VIM_FIQVECREG_T;

/* Capture Event Register (CAPEVT) */
typedef union
{
   struct
   {
      bit_field_T                    :9;   /* Reserved bit 31 - 23           */
      bit_field_T CAPEVTSRC1         :7;   /* Capture event source 1 mapping control  */
      bit_field_T                    :9;   /* Reserved bit 15 - 7            */
      bit_field_T CAPEVTSRC0         :7;   /* Capture event source 0 mapping control  */
   } bits;
   unsigned32_T  word;
} VIM_CAPEVT_T;

/* VIM Interrupt Control Register[0:31] (CHANCTRL[0:11])) */
typedef union
{
   struct
   {
      bit_field_T                    :1;   /* Reserved bit 31                     */
      bit_field_T CHANMAPww          :7;   /* Interrupt CHANx0 mapping control    */
      bit_field_T                    :1;   /* Reserved bit 23                     */
      bit_field_T CHANMAPxx          :7;   /* Interrupt CHANx1 mapping control    */
      bit_field_T                    :1;   /* Reserved bit 15                     */
      bit_field_T CHANMAPyy          :7;   /* Interrupt CHANx2 mapping control    */
      bit_field_T                    :1;   /* Reserved bit 7                      */
      bit_field_T CHANMAPzz          :7;   /* Interrupt CHANx3 mapping control    */
   } bits;
   unsigned32_T  word;
} VIM_CHANCTRL_T;

typedef struct
{
    VIM_PARFLG_T  VIM_PARFLG;
    VIM_PARCTL_T  VIM_PARCTL;
    VIM_ADDERR_T  VIM_ADDERR;
    VIM_FBPARERR_T VIM_FBPARERR;
} VIMPARREG_T;


/*********************************************************************************/

typedef struct
{
    VIM_IRQIVEC_T  VIM_IRQIVEC;
    VIM_FIQIVEC_T  VIM_FIQIVEC;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_FIRQPR_T   VIM_FIRQPR;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_INTREQ_T   VIM_INTREQ;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_REQMSKSET_T VIM_REQMSKSET;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_REQMSKCLR_T VIM_REQMSKCLR;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_WAKMSKSET_T VIM_WAKMSKSET;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_WAKMSKCLR_T VIM_WAKMSKCLR;
    bit_field_T    :32;
    bit_field_T    :32;

    VIM_IRQVECREG_T VIM_IRQVECREG;
    VIM_FIQVECREG_T VIM_FIQVECREG;
    VIM_CAPEVT_T    VIM_CAPEVT;
    bit_field_T    :32;

    VIM_CHANCTRL_T  VIM_CHANCTRL[12];
} VIMREG_T;


/*********************************************************************************/

typedef struct
{
   function_ptr_T VIM_ISR_VECTOR[49];
} VIMRAM_T;


/*********************************************************************************/

/* RAM Control Register (RAMCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :12;  /* Reserved bit 31 - 20       */
      bit_field_T ECC_INIT       :4;   /* ECC initialization enable  */
      bit_field_T                :3;   /* Reserved bit 15 - 13       */
      bit_field_T RMWCBYP        :4;   /* Read-modify-write correction bypass   */
      bit_field_T ECC_WRT_ENA    :1;   /* ECC write enable           */
      bit_field_T                :4;   /* Reserved bit 7 - 4         */
      bit_field_T ECC_ENABLE     :4;   /* ECC enable                 */
   } bits;
   unsigned32_T  word;
} RAMCTRL_T;

/* Threshold Register (RAMTHRESHOLD) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16               */
      bit_field_T SEC_THRESHOLD  :16;   /* Single error correction threshold  */
   } bits;
   unsigned32_T  word;
} RAMTHRESHOLD_T;

/* Occurrence Register (RAMOCCUR) */
typedef union
{
   struct
   {
      bit_field_T               :16;   /* Reserved bit 31 - 16             */
      bit_field_T SEC_OCCUR     :16;   /* Single error occurrence counter  */
   } bits;
   unsigned32_T  word;
} RAMOCCUR_T;

/* Interrupt Control Register (RAMINTCTRL) */
typedef union
{
   struct
   {
      bit_field_T              :31;   /* Reserved bit 31 - 1                    */
      bit_field_T SECINTEN     :1;    /* Single error correct interrupt enable  */
   } bits;
   unsigned32_T  word;
} RAMINTCTRL_T;

/* Memory Fault Detect Status Register (RAMERRSTATUS) */
typedef union
{
   struct
   {
      bit_field_T                :31;  /* Reserved bit 31 - 1                   */
      bit_field_T SECINTFLAG     :1;   /* Single error correct interrupt flag   */
   } bits;
   unsigned32_T  word;
} RAMERRSTATUS_T;

/* Single Error Address Register (RAMSERRADDR) */
typedef union
{
   struct
   {
      bit_field_T              :17;   /* Reserved bit 31 - 15            */
      bit_field_T SERRADDR     :15;   /* Single error address            */
   } bits;
   unsigned32_T  word;
} RAMSERRADDR_T;

/* RAM Error Position Register (RAMERRPOSITION) */
typedef union
{
   struct
   {
      bit_field_T                  :23;  /* Reserved bit 31 - 9   */
      bit_field_T ERRTYPE          :1;   /* Error Type            */
      bit_field_T SBZ              :2;   /* This bit is always 0  */
      bit_field_T SERRPOSITION     :6;   /* Single error position */
   } bits;
   unsigned32_T  word;
} RAMERRPOSITION_T;

/* Double Error Address Register (RAMDERRADDR) */
typedef union
{
   struct
   {
      bit_field_T                :17;   /* Reserved bit 31 - 15   */
      bit_field_T DERRADDR       :15;   /* Double error address   */
   } bits;
   unsigned32_T  word;
} RAMDERRADDR_T;

typedef struct
{
    RAMCTRL_T  RAMCTRL;
    RAMTHRESHOLD_T  RAMTHRESHOLD;
    RAMOCCUR_T    RAMOCCUR;
    RAMINTCTRL_T  RAMINTCTRL;
    RAMERRSTATUS_T RAMERRSTATUS;
    RAMSERRADDR_T  RAMSERRADDR;
    RAMERRPOSITION_T RAMERRPOSITION;
    RAMDERRADDR_T    RAMDERRADDR;
} RAMECCREG_T;


/*********************************************************************************/

/* RTI Global Control Register (RTIGCTRL) */
typedef union
{
   struct
   {
      bit_field_T            :16;  /* Reserved bit 31 - 16       */
      bit_field_T COS        :1;   /* Continue on suspend bit    */
      bit_field_T            :13;  /* Reserved bit 14 - 2        */
      bit_field_T CNT1EN     :1;   /* Counter 1 enable           */
      bit_field_T CNT0EN     :1;   /* Counter 0 enable           */
   } bits;
   unsigned32_T  word;
} RTIGCTRL_T;

/* RTI Timebase Control Register (RTITBCTRL) */
typedef union
{
   struct
   {
      bit_field_T            :30;  /* Reserved bit 31 - 16       */
      bit_field_T INC     :1;   /* Counter 1 enable           */
      bit_field_T TBEXT     :1;   /* Counter 0 enable           */
   } bits;
   unsigned32_T  word;
} RTITBCTRL_T;

/* RTI Capture Control Register (RTICAPCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 15 - 12            */
      bit_field_T COS        :1;   /* Reserved bit 15 - 12            */
      bit_field_T            :13;   /* Reserved bit 15 - 12            */
      bit_field_T CNT1EN     :1;   /* Reserved bit 15 - 12            */
      bit_field_T CAPCNTR0     :1;   /* Reserved bit 15 - 12            */
   } bits;
   unsigned32_T  word;
} RTICAPCTRL_T;

/* RTI Compare Control Register (RTICOMPCTRL) */
typedef union
{
   struct
   {
      bit_field_T                 :19;  /* Reserved bit 31 - 13    */
      bit_field_T COMPSEL3        :1;   /* Compare select 3        */
      bit_field_T                 :3;   /* Reserved bit 11 - 9     */
      bit_field_T COMPSEL2        :1;   /* Compare select 2        */
      bit_field_T                 :3;   /* Reserved bit 7 - 5      */
      bit_field_T COMPSEL1        :1;   /* Compare select 1        */
      bit_field_T                 :3;   /* Reserved bit 3 - 1      */
      bit_field_T COMPSEL0        :1;   /* Compare select 0        */
   } bits;
   unsigned32_T  word;
} RTICOMPCTRL_T;

/* RTI Free Running Counter 0 Register (RTIFRC0) */
typedef union
{
   struct
   {
      bit_field_T FRCx        :32;   /* Free running counter 0   */
   } bits;
   unsigned32_T  word;
} RTIFRCx_T;

/* RTI Up Counter 0 Register (RTIUC0) */
typedef union
{
   struct
   {
      bit_field_T UCx        :32;   /* Up counter 0   */
   } bits;
   unsigned32_T  word;
} RTIUCx_T;

/* RTI Compare Up Counter 0 Register (RTICPUC0) */
typedef union
{
   struct
   {
      bit_field_T CPUCx        :32;   /* Compare up counter 0  */
   } bits;
   unsigned32_T  word;
} RTICPUCx_T;

/* RTI Capture Free Running Counter 0 Register (RTICAFRC0) */
typedef union
{
   struct
   {
      bit_field_T CAFRCx        :32;   /* Capture free running counter 0   */
   } bits;
   unsigned32_T  word;
} RTICAFRCx_T;

/* RTI Capture Up Counter 0 Register (RTICAUC0) */
typedef union
{
   struct
   {
      bit_field_T CAUCx        :32;   /* Capture up counter 0    */
   } bits;
   unsigned32_T  word;
} RTICAUCx_T;

typedef struct
{
    RTIFRCx_T     RTIFRC;
    RTIUCx_T      RTIUC;
    RTICPUCx_T    RTICPUC;
    unsigned32_T  :32;
    RTICAFRCx_T   RTICAFRC;
    RTICAUCx_T    RTICAUC;
    unsigned32_T  :32;
    unsigned32_T  :32;
} RTICNTSET_T;

/* RTI Counter */
typedef enum
{
   RTI_COUNTER_ONE,
   RTI_COUNTER_TWO,
   MAX_RTI_COUNTERS
} RTI_Counter_Number_T;

/* RTI Compare 0 Register (RTICOMP0) */
typedef union
{
   struct
   {
      bit_field_T COMPx        :32;   /* Compare 0 bits  */
   } bits;
   unsigned32_T  word;
} RTICOMPx_T;

/* RTI Update Compare 0 Register (RTIUDCP0) */
typedef union
{
   struct
   {
      bit_field_T UDCPx        :32;   /* Update compare 0  */
   } bits;
   unsigned32_T  word;
} RTIUDCPx_T;

typedef struct
{
    RTICOMPx_T     RTICOMP;
    RTIUDCPx_T     RTIUDCP;
} RTICNTCMPSET_T;

/* RTI Set/Status Interrupt Register (RTISETINT) */
typedef union
{
   struct
   {
      bit_field_T                :13;  /* Reserved bit 31 - 19            */
      bit_field_T SETOVL1INT     :1;   /* Set free running counter 1 overflow interrupt   */
      bit_field_T SETOVL0INT     :1;   /* Set free running counter 0 overflow interrupt   */
      bit_field_T                :5;   /* Reserved bit 16 - 12            */
      bit_field_T SETDMA3        :1;   /* Set compare DMA request 3       */
      bit_field_T SETDMA2        :1;   /* Set compare DMA request 2       */
      bit_field_T SETDMA1        :1;   /* Set compare DMA request 1       */
      bit_field_T SETDMA0        :1;   /* Set compare DMA request 0       */
      bit_field_T                :4;   /* Reserved bit 7 - 4              */
      bit_field_T SETINT3        :1;   /* Set compare interrupt 3         */
      bit_field_T SETINT2        :1;   /* Set compare interrupt 2         */
      bit_field_T SETINT1        :1;   /* Set compare interrupt 1         */
      bit_field_T SETINT0        :1;   /* Set compare interrupt 0         */
   } bits;
   unsigned32_T  word;
} RTISETINT_T;

/* RTI Clear/Status Interrupt Register (RTICLEARINT) */
typedef union
{
   struct
   {
      bit_field_T                :13;  /* Reserved bit 31 - 19            */
      bit_field_T CLEAROVL1INT   :1;   /* Clear free running counter 1 overflow interrupt    */
      bit_field_T CLEAROVL0INT   :1;   /* Clear free running counter 0 overflow interrupt    */
      bit_field_T                :5;   /* Reserved bit 16 - 12            */
      bit_field_T CLEARDMA3      :1;   /* Clear compare DMA request 3     */
      bit_field_T CLEARDMA2      :1;   /* Clear compare DMA request 2     */
      bit_field_T CLEARDMA1      :1;   /* Clear compare DMA request 1     */
      bit_field_T CLEARDMA0      :1;   /* Clear compare DMA request 0     */
      bit_field_T                :4;   /* Reserved bit 7 - 4              */
      bit_field_T CLEARINT3      :1;   /* Clear compare interrupt 3       */
      bit_field_T CLEARINT2      :1;   /* Clear compare interrupt 2       */
      bit_field_T CLEARINT1      :1;   /* Clear compare interrupt 1       */
      bit_field_T CLEARINT0      :1;   /* Clear compare interrupt 0       */
   } bits;
   unsigned32_T  word;
} RTICLEARINT_T;

/* RTI Interrupt Flag Register (RTIINTFLAG) */
typedef union
{
   struct
   {
      bit_field_T                :13;  /* Reserved bit 31 - 19        */
      bit_field_T OVL1INT        :1;   /* Free running counter 1 overflow interrupt flag    */
      bit_field_T OVL0INT        :1;   /* Free running counter 0 overflow interrupt flag    */
      bit_field_T                :13;  /* Reserved bit 16 - 4         */
      bit_field_T INT3           :1;   /* Interrupt flag 3            */
      bit_field_T INT2           :1;   /* Interrupt flag 2            */
      bit_field_T INT1           :1;   /* Interrupt flag 1            */
      bit_field_T INT0           :1;   /* Interrupt flag 0            */
   } bits;
   unsigned32_T  word;
} RTIINTFLAG_T;

typedef struct
{
    RTIGCTRL_T  RTIGCTRL;
    RTITBCTRL_T RTITBCTRL; /* Not Present in TMS470PSF764 device */
    RTICAPCTRL_T  RTICAPCTRL;
    RTICOMPCTRL_T RTICOMPCTRL;
    RTICNTSET_T   RTICNTSET[MAX_RTI_COUNTERS];
    RTICNTCMPSET_T RTICNTCMPSET[4];
    unsigned32_T  :32;
    unsigned32_T  :32;
    unsigned32_T  :32;
    unsigned32_T  :32;
    RTISETINT_T   RTISETINT;
    RTICLEARINT_T RTICLEARINT;
    RTIINTFLAG_T  RTIINTFLAG;
} RTIREG_T;



/*********************************************************************************/

/* Variable Address Register (A0, A1, A2, A3) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16         */
      bit_field_T Ax             :16;   /* Variable Address 0, 1, 2, 3  */
   } bits;
   unsigned32_T  word;
} PBISTAx_T;

/* Variable Loop Count Register (L0, L1, L2, L3) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16               */
      bit_field_T Lx             :16;   /* Variable Loop Counter 0, 1, 2, 3   */
   } bits;
   unsigned32_T  word;
} PBISTLx_T;

/* DD0 Data Register (DD0) */
typedef union
{
   struct
   {
      bit_field_T DD1            :16;   /* DD1 Data Register Upper 16            */
      bit_field_T DD0            :16;   /* DD0 Data Register Lower 16            */
   } bits;
   unsigned32_T  word;
} PBISTDD_T;

/* DDE Data Register (DE0) */
typedef union
{
   struct
   {
      bit_field_T DE1            :16;   /* DE1 Data Register Upper 16   */
      bit_field_T DE0            :16;   /* DE0 Data Register Lower 16   */
   } bits;
   unsigned32_T  word;
} PBISTDE_T;

/* Constant Address Register (CA0, CA1, CA2, CA3) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16         */
      bit_field_T CAx            :16;   /* Constant Address 0, 1, 2, 3  */
   } bits;
   unsigned32_T  word;
} PBISTCAx_T;

/* Constant Loop Count Register (CL0, CL1, CL2, CL3) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16              */
      bit_field_T CLx            :16;   /* Constant Loop Counter 0, 1, 2, 3  */
   } bits;
   unsigned32_T  word;
} PBISTCLx_T;

/* Constant Increment Register (I0, I1, I2, I3) */
typedef union
{
   struct
   {
      bit_field_T                :16;   /* Reserved bit 31 - 16                   */
      bit_field_T Ix             :16;   /* Constant Increment Counter 0, 1, 2, 3  */
   } bits;
   unsigned32_T  word;
} PBISTIx_T;

/* RAM Configuration Register (RAMT) */
typedef union
{
   struct
   {
      bit_field_T RGS            :8;   /* Ram Group Select             */
      bit_field_T RDS            :8;   /* Return Data Select           */
      bit_field_T DWR            :8;   /* Data Width Register          */
      bit_field_T SMS            :2;   /* Sense Margin Select Register */
      bit_field_T PLS            :4;   /* Pipeline Latency Select      */
      bit_field_T RLS            :2;   /* RAM Latency Select           */
   } bits;
   unsigned32_T  word;
} PBISTRAMT_T;

/* Datalogger Register (DLR) */
typedef union
{
   struct
   {
      bit_field_T                 :21;  /* Reserved bit 31 - 11          */
      bit_field_T DLR10           :1;   /* Retention testing             */
      bit_field_T DLR9            :1;   /* GO / NO-GO testing            */
      bit_field_T DLR8            :1;   /* MISR testing                  */
      bit_field_T DLR7            :1;   /* Time Stamp                    */
      bit_field_T DLR6            :1;   /* Column Fail Masking           */
      bit_field_T DLR5            :1;   /* Emulation cache access        */
      bit_field_T DLR4            :1;   /* Config access                 */
      bit_field_T DLR3            :1;   /* TCK gated (Diag_clock gated)  */
      bit_field_T DLR2            :1;   /* ROM-based testing             */
      bit_field_T DLR1            :1;   /* IDDQ testing                  */
      bit_field_T DLR0            :1;   /* Distributed Compare mode      */
   } bits;
   unsigned32_T  word;
} PBISTDLR_T;

/* Clock-Mux Select Register (CMS) */
typedef union
{
   struct
   {
      bit_field_T             :28;   /* Reserved bit 31 - 4    */
      bit_field_T CMS         :4;    /* Clock input            */
   } bits;
   unsigned32_T  word;
} PBISTCMS_T;

/* Program Control Register (STR) */
typedef union
{
   struct
   {
      bit_field_T             :27;   /* Reserved bit 31 - 5    */
      bit_field_T STR         :5;    /* PBIST Controller Mode  */
   } bits;
   unsigned32_T  word;
} PBISTSTR_T;

/* Chip Select Register (CSR) */
typedef union
{
   struct
   {
      bit_field_T CSR3            :8;   /* Chip Select 3    */
      bit_field_T CSR2            :8;   /* Chip Select 2    */

      bit_field_T CSR1            :8;   /* Chip Select 1    */
      bit_field_T CSR0            :8;   /* Chip Select 0    */
   } bits;
   unsigned32_T  word;
} PBISTCSR_T;

/* Fail Delay Register (FDLY) */
typedef union
{
   struct
   {
      bit_field_T                 :24;  /* Reserved bit 31 - 8                 */
      bit_field_T FDLY            :8;   /* Delay in asserting the FAIL signal  */
   } bits;
   unsigned32_T  word;
} PBISTFDLY_T;

/* PBIST Activate/ROM Clock Enable Register (PACT) */
typedef union
{
   struct
   {
      bit_field_T                 :30;  /* Reserved bit 31 - 2       */

      bit_field_T PACT1           :1;   /* PBIST Activate            */
      bit_field_T PACT0           :1;   /* ROM Clock Enable Register */
   } bits;
   unsigned32_T  word;
} PBISTPACT_T;

/* PBIST ID Register */
typedef union
{
   struct
   {
      bit_field_T               :24;   /* Reserved bit 31 - 8 */
      bit_field_T ID            :8;    /* PBIST ID            */
   } bits;
   unsigned32_T  word;
} PBISTID_T;

/* Override Register (OVER) */
typedef union
{
   struct
   {
      bit_field_T                  :29;  /* Reserved bit 31 - 3     */
      bit_field_T OVER2            :1;   /* Multiple Memory Enable  */
      bit_field_T OVER1            :1;   /* READ Override Bit       */
      bit_field_T OVER0            :1;   /* RINFO Override Bit      */
   } bits;
   unsigned32_T  word;
} PBISTOVER_T;

/* Fail Status Fail Register 0 (FSRF0) */
typedef union
{
   struct
   {
      bit_field_T                 :31;  /* Reserved bit 31 - 1      */
      bit_field_T FSRF            :1;   /* Fail Status 0            */
   } bits;
   unsigned32_T  word;
} PBISTFSRFx_T;

/* Fail Status Count 0 Register (FSRC0) */
typedef union
{
   struct
   {
      bit_field_T                 :24;   /* Reserved bit 15 - 12   */
      bit_field_T FSRC            :8;   /* Reserved bit 15 - 12    */
   } bits;
   unsigned32_T  word;
} PBISTFSRCx_T;

/* Fail Status Address 0 Register (FSRA0) */
typedef union
{
   struct
   {
      bit_field_T                 :16;   /* Reserved bit 31 - 16    */
      bit_field_T FSRA            :16;   /* Fail Status Address 0   */
   } bits;
   unsigned32_T  word;
} PBISTFSRAx_T;

/* Fail Status Data Register 0 (FSRDL0) */
typedef union
{
   struct
   {
      bit_field_T FSRDL            :32;   /* Failure data on port 0  */
   } bits;
   unsigned32_T  word;
} PBISTFSRDLx_T;

typedef struct
{
   PBISTFSRFx_T  PBISTFSRF[2];
   PBISTFSRCx_T  PBISTFSRC[2];
   PBISTFSRAx_T  PBISTFSRA[2];
   PBISTFSRDLx_T  PBISTFSRDL[2];
} PBISTFAILSFx_T;

/* Fail Status Data Register 1 (FSRDL1) */
typedef union
{
   struct
   {
      bit_field_T FSRDL1            :32;   /* Failure data on port 1   */
   } bits;
   unsigned32_T  word;
} PBISTFSRDL1_T;

/* ROM Mask Register (ROM) */
typedef union
{
   struct
   {
      bit_field_T                :30;  /* Reserved bit 31 - 2  */
      bit_field_T PBIST_ROM      :2;   /* ROM Mask             */
   } bits;
   unsigned32_T  word;
} PBISTROM_T;

/* ROM Algorithm Mask Register (ALGO) */
typedef union
{
   struct
   {
      bit_field_T ALGOx            :32;   /* ALGOx select    */
   } bits;
   unsigned32_T  word;
} PBISTALGO_T;

/* RAM Info Mask Lower Register (RINFOL) */
typedef union
{
   struct
   {
      bit_field_T RINFOLx          :32;   /* RAM Group select[32:1]  */
   } bits;
   unsigned32_T  word;
} PBISTRINFOL_T;

/* RAM Info Mask Upper Register (RINFOU) */
typedef union
{
   struct
   {
      bit_field_T RINFOUx          :32;   /* RAM Group select[64:33]  */
   } bits;
   unsigned32_T  word;
} PBISTRINFOU_T;

typedef struct
{
    unsigned32_T RESERVED[0x100/sizeof(unsigned32_T)];

    PBISTAx_T  PBISTA[4];

    PBISTLx_T  PBISTL[4];

    PBISTDD_T  PBISTDD;
    PBISTDE_T  PBISTDE;
    unsigned32_T    :32;
    unsigned32_T    :32;
    PBISTCAx_T PBISTCA[4];

    PBISTCLx_T PBISTCL[4];

    PBISTIx_T  PBISTI[4];

    PBISTRAMT_T PBISTRAMT;
    PBISTDLR_T  PBISTDLR;
    PBISTCMS_T  PBISTCMS;
    PBISTSTR_T  PBISTSTR;
    unsigned32_T    :32;
    unsigned32_T    :32;
    PBISTCSR_T   PBISTCSR;
    PBISTFDLY_T  PBISTFDLY;

    PBISTPACT_T  PBISTPACT;
    PBISTID_T    PBISTID;
    PBISTOVER_T  PBISTOVER;
    unsigned32_T    :32;

    PBISTFAILSFx_T PBISTFAILSF;

    PBISTFSRDL1_T  PBISTFSRDL1;
    unsigned32_T    :32;
    unsigned32_T    :32;
    unsigned32_T    :32;
    PBISTROM_T   PBISTROM;
    PBISTALGO_T  PBISTALGO;
    PBISTRINFOL_T PBISTRINFOL;
    PBISTRINFOU_T PBISTRINFOU;

} PBISTREG_T;


/*********************************************************************************/

/* CCM-7 Status Register (CCMSR) */
typedef union
{
   struct
   {
      bit_field_T                :15;  /* Reserved bit 31 - 17   */
      bit_field_T CMPE           :1;   /* Compare Error          */
      bit_field_T                :7;   /* Reserved bit 15 - 9    */
      bit_field_T STC            :1;   /* Self Test Complete     */
      bit_field_T                :6;   /* Reserved bit 7 - 2     */
      bit_field_T STET           :1;   /* Self Test Error        */
      bit_field_T STE            :1;   /* Self Test Error        */
   } bits;
   unsigned32_T  word;
} CCMSR_T;

/* CCM-7 Key Register (CCMKEYR) */
typedef union
{
   struct
   {
      bit_field_T                :28;  /* Reserved bit 31 - 4   */
      bit_field_T MKEY           :4;   /* Mode Key              */
   } bits;
   unsigned32_T  word;
} CCMKEYR_T;

typedef struct
{
    CCMSR_T   CCMSR;
    CCMKEYR_T CCMKEYR;
} CCM7REG_T;


/*********************************************************************************/

/*********************************************************************************/

/* DMA Global Control Register (GCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :15;  /* Reserved bit 31 - 17      */
      bit_field_T DMA_EN         :1;   /* DMA enable bit            */
      bit_field_T                :6;   /* Reserved bit 15 - 10      */
      bit_field_T DEBUG_MODE     :2;   /* Debug Mode                */
      bit_field_T                :7;   /* Reserved bit 7 - 1        */
      bit_field_T DMA_RES        :1;   /* DMA software reset        */
   } bits;
   unsigned32_T  word;
} DMAGCTRL_T;

/* DMA Channel Pending Register (PEND) */
typedef union
{
   struct
   {
      bit_field_T PEND           :32;   /* Channel pending register  */
   } bits;
   unsigned32_T  word;
} DMAPEND_T;

/* DMA Status Register (DMASTAT) */
typedef union
{
   struct
   {
      bit_field_T STCH           :32;   /* Status of DMA channels   */
   } bits;
   unsigned32_T  word;
} DMASTAT_T;

/* DMA HW Channel Enable Set and Status Register (HWCHENAS) */
typedef union
{
   struct
   {
      bit_field_T HWCHENA        :32;   /* Hardware channel enable bit  */
   } bits;
   unsigned32_T  word;
} DMAHWCHENAS_T;

/* DMA HW Channel Enable Reset and Status Register (HWCHENAR) */
typedef union
{
   struct
   {
      bit_field_T HWCHDIS        :32;   /* HW channel disable bit   */
   } bits;
   unsigned32_T  word;
} DMAHWCHENAR_T;

/* DMA SW Channel Enable Set and Status Register (SWCHENAS) */
typedef union
{
   struct
   {
      bit_field_T SWCHENA        :32;   /* SW channel enable bit   */
   } bits;
   unsigned32_T  word;
} DMASWCHENAS_T;

/* DMA SW Channel Enable Reset and Status Register (SWCHENAR) */
typedef union
{
   struct
   {
      bit_field_T SWCHDIS        :32;   /* SW channel disable bit  */
   } bits;
   unsigned32_T  word;
} DMASWCHENAR_T;

/* DMA Channel Priority Set Register (CHPRIOS) */
typedef union
{
   struct
   {
      bit_field_T CPH           :32;   /* Channel priority set bit   */
   } bits;
   unsigned32_T  word;
} DMACHPRIOS_T;

/* DMA Channel Priority Reset Register (CHPRIOR) */
typedef union
{
   struct
   {
      bit_field_T CPR           :32;   /* Channel priority reset bit */
   } bits;
   unsigned32_T  word;
} DMACHPRIOR_T;

/* DMA Global Channel Interrupt Enable Set Register (GCHIENAS) */
typedef union
{
   struct
   {
      bit_field_T GCHIE         :32;   /* Global channel interrupt enable bit  */
   } bits;
   unsigned32_T  word;
} DMAGCHIENAS_T;

/* DMA Global Channel Interrupt Enable Reset Register (GCHIENAR) */
typedef union
{
   struct
   {
      bit_field_T GCHID         :32;   /* Global channel interrupt disable bit */
   } bits;
   unsigned32_T  word;
} DMAGCHIENAR_T;

/* DMA Request Assignment Register 0 (DREQASI0) */
typedef union
{
   struct
   {
      bit_field_T                  :2;   /* Reserved bit 31 - 30    */
      bit_field_T CH0ASI           :6;   /* Channel 0 assignment    */
      bit_field_T                  :2;   /* Reserved bit 23 - 22    */
      bit_field_T CH1ASI           :6;   /* Channel 1 assignment    */
      bit_field_T                  :2;   /* Reserved bit 15 - 14    */
      bit_field_T CH2ASI           :6;   /* Channel 2 assignment    */
      bit_field_T                  :2;   /* Reserved bit 7 - 6      */
      bit_field_T CH3ASI           :6;   /* Channel 3 assignment    */
   } bits;
   unsigned32_T  word;
} DREQASIx_T;

/* DMA Port Assignment Register 0 (PAR0) */
typedef union
{
   struct
   {
      bit_field_T                  :1;   /* Reserved bit 31          */
      bit_field_T CH0PA            :3;   /* Port channel 0           */
      bit_field_T                  :1;   /* Reserved bit 27          */
      bit_field_T CH1PA            :3;   /* Port channel 1           */
      bit_field_T                  :1;   /* Reserved bit 23          */
      bit_field_T CH2PA            :3;   /* Port channel 2           */
      bit_field_T                  :1;   /* Reserved bit 19          */
      bit_field_T CH3PA            :3;   /* Port channel 3           */
      bit_field_T                  :1;   /* Reserved bit 15          */
      bit_field_T CH4PA            :3;   /* Port channel 4           */
      bit_field_T                  :1;   /* Reserved bit 11          */
      bit_field_T CH5PA            :3;   /* Port channel 5           */
      bit_field_T                  :1;   /* Reserved bit 7           */
      bit_field_T CH6PA            :3;   /* Port channel 6           */
      bit_field_T                  :1;   /* Reserved bit 3           */
      bit_field_T CH7PA            :3;   /* Port channel 7           */
   } bits;
   unsigned32_T  word;
} DMAPAR0_T;

/* DMA Port Assignment Register 1 (PAR1) */
typedef union
{
   struct
   {
      bit_field_T                  :1;   /* Reserved bit 31           */
      bit_field_T CH8PA            :3;   /* Port channel 8            */
      bit_field_T                  :1;   /* Reserved bit 27           */
      bit_field_T CH9PA            :3;   /* Port channel 9            */
      bit_field_T                  :1;   /* Reserved bit 23           */
      bit_field_T CH10PA           :3;   /* Port channel 10           */
      bit_field_T                  :1;   /* Reserved bit 19           */
      bit_field_T CH11PA           :3;   /* Port channel 11           */
      bit_field_T                  :1;   /* Reserved bit 15           */
      bit_field_T CH12PA           :3;   /* Port channel 12           */
      bit_field_T                  :1;   /* Reserved bit 11           */
      bit_field_T CH13PA           :3;   /* Port channel 13           */
      bit_field_T                  :1;   /* Reserved bit 7            */
      bit_field_T CH14PA           :3;   /* Port channel 14           */
      bit_field_T                  :1;   /* Reserved bit 3            */
      bit_field_T CH15PA           :3;   /* Port channel 15           */
   } bits;
   unsigned32_T  word;
} DMAPAR1_T;

/* DMA Port Assignment Register 2 (PAR2) */
typedef union
{
   struct
   {
      bit_field_T                  :1;   /* Reserved bit 31            */
      bit_field_T CH16PA           :3;   /* Port channel 16            */
      bit_field_T                  :1;   /* Reserved bit 27            */
      bit_field_T CH17PA           :3;   /* Port channel 17            */
      bit_field_T                  :1;   /* Reserved bit 23            */
      bit_field_T CH18PA           :3;   /* Port channel 18            */
      bit_field_T                  :1;   /* Reserved bit 19            */
      bit_field_T CH19PA           :3;   /* Port channel 19            */
      bit_field_T                  :1;   /* Reserved bit 15            */
      bit_field_T CH20PA           :3;   /* Port channel 20            */
      bit_field_T                  :1;   /* Reserved bit 11            */
      bit_field_T CH21PA           :3;   /* Port channel 21            */
      bit_field_T                  :1;   /* Reserved bit 7             */
      bit_field_T CH22PA           :3;   /* Port channel 22            */
      bit_field_T                  :1;   /* Reserved bit 3             */
      bit_field_T CH23PA           :3;   /* Port channel 23            */
   } bits;
   unsigned32_T  word;
} DMAPAR2_T;

/* DMA Port Assignment Register 3 (PAR3) */
typedef union
{
   struct
   {
      bit_field_T                  :1;   /* Reserved bit 31            */
      bit_field_T CH24PA           :3;   /* Port channel 24            */
      bit_field_T                  :1;   /* Reserved bit 27            */
      bit_field_T CH25PA           :3;   /* Port channel 25            */
      bit_field_T                  :1;   /* Reserved bit 23            */
      bit_field_T CH26PA           :3;   /* Port channel 26            */
      bit_field_T                  :1;   /* Reserved bit 19            */
      bit_field_T CH27PA           :3;   /* Port channel 27            */
      bit_field_T                  :1;   /* Reserved bit 15            */
      bit_field_T CH28PA           :3;   /* Port channel 28            */
      bit_field_T                  :1;   /* Reserved bit 11            */
      bit_field_T CH29PA           :3;   /* Port channel 29            */
      bit_field_T                  :1;   /* Reserved bit 7             */
      bit_field_T CH30PA           :3;   /* Port channel 30            */
      bit_field_T                  :1;   /* Reserved bit 3             */
      bit_field_T CH31PA           :3;   /* Port channel 31            */
   } bits;
   unsigned32_T  word;
} DMAPAR3_T;

/* DMA FTC Interrupt Mapping Register (FTCMAP) */
typedef union
{
   struct
   {
      bit_field_T FTCAB           :32;   /* Frame transfer complete (FTC) interrupt to Group A or Group B */
   } bits;
   unsigned32_T  word;
} DMAFTCMAP_T;

/* DMA LFS Interrupt Mapping Register (LFSMAP) */
typedef union
{
   struct
   {
      bit_field_T LFSAB           :32;   /* Last frame started (LFS) interrupt to Group A or Group B */
   } bits;
   unsigned32_T  word;
} DMALFSMAP_T;

/* DMA HBC Interrupt Mapping Register (HBCMAP) */
typedef union
{
   struct
   {
      bit_field_T HBCAB           :32;   /* Half block complete (HBC) interrupt to Group A or Group B */
   } bits;
   unsigned32_T  word;
} DMAHBCMAP_T;

/* DMA BTC Interrupt Mapping Register (BTCMAP) */
typedef union
{
   struct
   {
      bit_field_T BTCAB           :32;   /* Block transfer complete (BTC) interrupt to Group A or Group B */
   } bits;
   unsigned32_T  word;
} DMABTCMAP_T;

/* DMA BER Interrupt Mapping Register (BERMAP) */
typedef union
{
   struct
   {
      bit_field_T BERAB           :32;   /* Bus error (BER) interrupt to Group A or Group B */
   } bits;
   unsigned32_T  word;
} DMABERMAP_T;

/* DMA FTC Interrupt Enable Set (FTCINTENAS) */
typedef union
{
   struct
   {
      bit_field_T FTCINTENA       :32;   /* Frame transfer complete (FTC) interrupt enable */
   } bits;
   unsigned32_T  word;
} DMAFTCINTENAS_T;

/* DMA FTC Interrupt Enable Reset (FTCINTENAR) */
typedef union
{
   struct
   {
      bit_field_T FTCINTDIS       :32;   /* Frame transfer complete (FTC) interrupt disable */
   } bits;
   unsigned32_T  word;
} DMAFTCINTENAR_T;

/* DMA LFS Interrupt Enable Set (LFSINTENAS) */
typedef union
{
   struct
   {
      bit_field_T LFSINTENA       :32;   /* Last frame started (LFS) interrupt enable */
   } bits;
   unsigned32_T  word;
} DMALFSINTENAS_T;

/* DMA LFS Interrupt Enable Reset (LFSINTENAR) */
typedef union
{
   struct
   {
      bit_field_T LFSINTDIS       :32;   /* Last frame started (LFS) interrupt disable */
   } bits;
   unsigned32_T  word;
} DMALFSINTENAR_T;

/* DMA HBC Interrupt Enable Reset (HBCINTENAS) */
typedef union
{
   struct
   {
      bit_field_T HBCINTENA       :32;   /* Half block complete (HBC) interrupt enable */
   } bits;
   unsigned32_T  word;
} DMAHBCINTENAS_T;

/* DMA HBC Interrupt Enable Reset (HBCINTENAR) */
typedef union
{
   struct
   {
      bit_field_T HBCINTDIS       :32;   /* Half block complete (HBC) interrupt disable */
   } bits;
   unsigned32_T  word;
} DMAHBCINTENAR_T;

/* DMA BTC Interrupt Enable Set (BTCINTENAS) */
typedef union
{
   struct
   {
      bit_field_T BTCINTENA       :32;   /* Block transfer complete (BTC) interrupt enable */
   } bits;
   unsigned32_T  word;
} DMABTCINTENAS_T;

/* DMA BTC Interrupt Enable Reset (BTCINTENAR) */
typedef union
{
   struct
   {
      bit_field_T BTCINTDIS       :32;   /* Block transfer complete (BTC) interrupt disable */
   } bits;
   unsigned32_T  word;
} DMABTCINTENAR_T;

/* DMA Global Interrupt Flag Register (GINTFLAG) */
typedef union
{
   struct
   {
      bit_field_T GINT           :32;   /* Global interrupt flags */
   } bits;
   unsigned32_T  word;
} DMAGINTFLAG_T;

/* DMA FTC Interrupt Flag Register (FTCFLAG) */
typedef union
{
   struct
   {
      bit_field_T FTC            :32;   /* Frame transfer complete (FTC) flags */
   } bits;
   unsigned32_T  word;
} DMAFTCFLAG_T;

/* DMA LFS Interrupt Flag Register (LFSFLAG) */
typedef union
{
   struct
   {
      bit_field_T LFSI           :32;   /* Last frame started (LFS) flags */
   } bits;
   unsigned32_T  word;
} DMALFSFLAG_T;

/* DMA HBC Interrupt Flag Register (HBCFLAG) */
typedef union
{
   struct
   {
      bit_field_T HBCI           :32;   /* Half block transfer (HBC) complete flags */
   } bits;
   unsigned32_T  word;
} DMAHBCFLAG_T;

/* DMA BTC Interrupt Flag Register (BTCFLAG) */
typedef union
{
   struct
   {
      bit_field_T BTCI           :32;   /* Block transfer complete (BTC) flags */
   } bits;
   unsigned32_T  word;
} DMABTCFLAG_T;

/* DMA BER Interrupt Flag Register (BERFLAG) */
typedef union
{
   struct
   {
      bit_field_T BERI           :32;   /* Bus error (BER) flags */
   } bits;
   unsigned32_T  word;
} DMABERFLAG_T;

/* DMA FTCA Interrupt Channel Offset Register (FTCAOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                     */
      bit_field_T sbz            :2;    /* These bits should always be zero        */
      bit_field_T FTCA           :6;    /* Channel causing FTC interrupt Group A   */
   } bits;
   unsigned32_T  word;
} DMAFTCAOFFSET_T;

/* DMA LFSA Interrupt Channel Offset Register (LFSAOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T LFSA           :6;    /* Channel causing LFS interrupt Group A          */
   } bits;
   unsigned32_T  word;
} DMALFSAOFFSET_T;

/* DMA HBCA Interrupt Channel Offset Register (HBCAOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T HBCA           :6;    /* Channel causing HBC interrupt Group A          */
   } bits;
   unsigned32_T  word;
} DMAHBCAOFFSET_T;

/* DMA BTCA Interrupt Channel Offset Register (BTCAOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T BTCA           :6;    /* Channel causing BTC interrupt Group A          */
   } bits;
   unsigned32_T  word;
} DMABTCAOFFSET_T;

/* DMA BERA Interrupt Channel Offset Register (BERAOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T BERA           :6;    /* Channel causing BER interrupt Group A          */
   } bits;
   unsigned32_T  word;
} DMABERAOFFSET_T;

/* DMA FTCB Interrupt Channel Offset Register (FTCBOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T FTCB           :6;    /* Channel causing FTC interrupt Group B          */
   } bits;
   unsigned32_T  word;
} DMAFTCBOFFSET_T;

/* DMA LFSB Interrupt Channel Offset Register (LFSBOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T LFSB           :6;    /* Channel causing LFS interrupt Group B          */
   } bits;
   unsigned32_T  word;
} DMALFSBOFFSET_T;

/* DMA HBCB Interrupt Channel Offset Register (HBCBOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T HBCB           :6;    /* Channel causing HBC interrupt Group B          */
   } bits;
   unsigned32_T  word;
} DMAHBCBOFFSET_T;

/* DMA BTCB Interrupt Channel Offset Register (BTCBOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T BTCB           :6;    /* Channel causing BTC interrupt Group B          */
   } bits;
   unsigned32_T  word;
} DMABTCBOFFSET_T;

/* DMA BERB Interrupt Channel Offset Register (BERBOFFSET) */
typedef union
{
   struct
   {
      bit_field_T                :24;   /* Reserved bit 31 - 8                            */
      bit_field_T sbz            :2;    /* These bits should always be programmed as zero */
      bit_field_T BERB           :6;    /* Channel causing BER interrupt Group B          */
   } bits;
   unsigned32_T  word;
} DMABERBOFFSET_T;

/* DMA Port Control Register (PTCRL) */
typedef union
{
   struct
   {
      bit_field_T                :7;   /* Reserved bit 31 - 25                              */
      bit_field_T PENDB          :1;   /* Transfers pending for port B                      */
      bit_field_T                :5;   /* Reserved bit 23 - 19                              */
      bit_field_T BYB            :1;   /* Bypass FIFO B                                     */
      bit_field_T PSFRHQPB       :1;   /* Priority scheme for high priority queue of Port B */
      bit_field_T PSFRLQPB       :1;   /* Priority scheme for low priority queue of Port B  */
      bit_field_T                :7;   /* Reserved bit 15 - 9                               */
      bit_field_T PENDA          :1;   /* Transfers pending for port A                      */
      bit_field_T                :5;   /* Reserved bit 7 - 3                                */
      bit_field_T BYA            :1;   /* Bypass FIFO A                                     */
      bit_field_T PSFRHQPA       :1;   /* Priority scheme for high priority queue of Port A */
      bit_field_T PSFRLQPA       :1;   /* Priority scheme for low priority queue of Port A  */
   } bits;
   unsigned32_T  word;
} DMAPTCRL_T;

/* DMA RAM Test Control (RTCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :31;   /* Reserved bit 31 - 1   */
      bit_field_T RTC            :1;    /* RAM test control      */
   } bits;
   unsigned32_T  word;
} DMARTCRL_T;

/* DMA Debug Control (DCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :3;   /* Reserved bit 31 - 29   */
      bit_field_T CHNUM          :5;   /* Channel number         */
      bit_field_T                :7;   /* Reserved bit 23 - 17   */
      bit_field_T DMADBGS        :1;   /* DMA debug status       */
      bit_field_T                :15;  /* Reserved bit 15 - 1    */
      bit_field_T DBGEN          :1;   /* Debug enable           */
   } bits;
   unsigned32_T  word;
} DMADCTRL_T;

/* DMA Watch Point Register (WPR) */
typedef union
{
   struct
   {
      bit_field_T WP           :32;   /* Watch point    */
   } bits;
   unsigned32_T  word;
} DMAWPR_T;

/* DMA Watch Mask Register (WMR) */
typedef union
{
   struct
   {
      bit_field_T WM           :32;   /* Watch mask     */
   } bits;
   unsigned32_T  word;
} DMAWMR_T;

/* DMA Port B Active Channel Source Address Register (PBACSADDR) */
typedef union
{
   struct
   {
      bit_field_T PBACSA       :32;   /* Port B active channel source address */
   } bits;
   unsigned32_T  word;
} DMAPBACSADDR_T;

/* DMA Port B Active Channel Destination Address Register (PBACDADDR) */
typedef union
{
   struct
   {
      bit_field_T PBACDA       :32;   /* Port B active channel destination address */
   } bits;
   unsigned32_T  word;
} DMAPBACDADDR_T;

/* DMA Port B Active Channel Transfer Count Register (PBACTC) */
typedef union
{
   struct
   {
      bit_field_T                     :3;    /* Reserved bit 31 - 29                */
      bit_field_T PBFTCOUNT           :13;   /* Port B active channel frame count   */
      bit_field_T                     :3;    /* Reserved bit 15 - 13                */
      bit_field_T PBETCOUNT           :13;   /* Port B active channel element count */
   } bits;
   unsigned32_T  word;
} DMAPBACTC_T;

/* DMA Parity Control Register (DMAPCR) */
typedef union
{
   struct
   {
      bit_field_T                     :15;  /* Reserved bit 31 - 17           */
      bit_field_T ERRA                :1;   /* Error action                   */
      bit_field_T                     :7;   /* Reserved bit 15 - 9            */
      bit_field_T TEST                :1;   /* Parity bits mapping            */
      bit_field_T                     :4;   /* Reserved bit 7 - 4             */
      bit_field_T PARITY_ENA          :4;   /* Parity error detection enable  */
   } bits;
   unsigned32_T  word;
} DMAPCR_T;

/* DMA Parity Error Address Register (DMAPAR) */
typedef union
{
   struct
   {
      bit_field_T                     :20;   /* Reserved bit 31 - 12     */
      bit_field_T ERRORADDRESS        :12;   /* Error address            */
   } bits;
   unsigned32_T  word;
} DMAPAR_T;

/* DMA Memory Protection Control Register (DMAMPCTRL) */
typedef union
{
   struct
   {
      bit_field_T                     :3;   /* Reserved bit 31 - 29         */
      bit_field_T INT3AB              :1;   /* Interrupt assignment of region 3 to Group A or B */
      bit_field_T INT3ENA             :1;   /* Interrupt enable of region 3 */
      bit_field_T REG3A               :2;   /* Region 3 access permission   */
      bit_field_T REG3ENA             :1;   /* Region 3 enable              */
      bit_field_T                     :3;   /* Reserved bit 23 - 21         */
      bit_field_T INT2AB              :1;   /* Interrupt assignment of region 2 to Group A or B */
      bit_field_T INT2ENA             :1;   /* Interrupt enable of region 2 */
      bit_field_T REG2A               :2;   /* Region 2 access permission   */
      bit_field_T REG2ENA             :1;   /* Region 2 enable              */
      bit_field_T                     :3;   /* Reserved bit 15 - 13         */
      bit_field_T INT1AB              :1;   /* Interrupt assignment of region 1 to Group A or B */
      bit_field_T INT1ENA             :1;   /* Interrupt enable of region 1 */
      bit_field_T REG1A               :2;   /* Region 1 access permission   */
      bit_field_T REG1ENA             :1;   /* Region 1 enable              */
      bit_field_T                     :3;   /* Reserved bit 7 - 5           */
      bit_field_T INT0AB              :1;   /* Interrupt assignment of region 0 to Group A or B */
      bit_field_T INT0ENA             :1;   /* Interrupt enable of region 0 */
      bit_field_T REG0A               :2;   /* Region 0 access permission   */
      bit_field_T REG0ENA             :1;   /* Region 0 enable              */
   } bits;
   unsigned32_T  word;
} DMAMPCTRL_T;

/* DMA Memory Protection Status Register (DMAMPST) */
typedef union
{
   struct
   {
      bit_field_T                     :7;   /* Reserved bit 31 - 25      */
      bit_field_T REG3FT              :1;   /* Region 3 fault            */
      bit_field_T                     :7;   /* Reserved bit 23 - 17      */
      bit_field_T REG2FT              :1;   /* Region 2 fault            */
      bit_field_T                     :7;   /* Reserved bit 15 - 9       */
      bit_field_T REG1FT              :1;   /* Region 1 fault            */
      bit_field_T                     :7;   /* Reserved bit 7 - 1        */
      bit_field_T REG0FT              :1;   /* Region 0 fault            */
   } bits;
   unsigned32_T  word;
} DMAMPST_T;

/* DMA Memory Protection Region 0 Start Address Register (DMAMPR0S) */
typedef union
{
   struct
   {
      bit_field_T STARTADDRESS        :32;   /* Region 0 Start Address   */
   } bits;
   unsigned32_T  word;
} DMAMPR0S_T;

/* DMA Memory Protection Region 0 End Address Register (DMAMPR0E) */
typedef union
{
   struct
   {
      bit_field_T ENDADDRESS        :32;    /* Region 0 End Address      */
   } bits;
   unsigned32_T  word;
} DMAMPR0E_T;

/* DMA Memory Protection Region 1 Start Address Register (DMAMPR1S) */
typedef union
{
   struct
   {
      bit_field_T STARTADDRESS      :32;   /* Region 1 Start Address     */
   } bits;
   unsigned32_T  word;
} DMAMPR1S_T;

/* DMA Memory Protection Region 1 End Address Register (DMAMPR1E) */
typedef union
{
   struct
   {
      bit_field_T ENDADDRESS        :32;   /* Region 1 End Address       */
   } bits;
   unsigned32_T  word;
} DMAMPR1E_T;

/* DMA Memory Protection Region 2 Start Address Register (DMAMPR2S) */
typedef union
{
   struct
   {
      bit_field_T STARTADDRESS      :32;   /* Region 2 Start Address     */
   } bits;
   unsigned32_T  word;
} DMAMPR2S_T;

/* DMA Memory Protection Region 2 End Address Register (DMAMPR2E) */
typedef union
{
   struct
   {
      bit_field_T ENDADDRESS        :32;   /* Region 2 End Address       */
   } bits;
   unsigned32_T  word;
} DMAMPR2E_T;

/* DMA Memory Protection Region 3 Start Address Register (DMAMPR3S) */
typedef union
{
   struct
   {
      bit_field_T STARTADDRESS      :32;   /* Region 3 Start Address     */
   } bits;
   unsigned32_T  word;
} DMAMPR3S_T;

/* DMA Memory Protection Region 3 End Address Register (DMAMPR3E) */
typedef union
{
   struct
   {
      bit_field_T ENDADDRESS        :32;   /* Region 3 End Address       */
   } bits;
   unsigned32_T  word;
} DMAMPR3E_T;

typedef struct
{
    DMAGCTRL_T   DMAGCTRL;
    DMAPEND_T    DMAPEND;
    bit_field_T   :32;
    DMASTAT_T    DMASTAT;
    bit_field_T   :32;
    DMAHWCHENAS_T DMAHWCHENAS;
    bit_field_T   :32;
    DMAHWCHENAR_T DMAHWCHENAR;
    bit_field_T   :32;
    DMASWCHENAS_T DMASWCHENAS;
    bit_field_T   :32;
    DMASWCHENAR_T DMASWCHENAR;
    bit_field_T   :32;
    DMACHPRIOS_T  DMACHPRIOS;
    bit_field_T   :32;
    DMACHPRIOR_T  DMACHPRIOR;
    bit_field_T   :32;
    DMAGCHIENAS_T DMAGCHIENAS;
    bit_field_T   :32;
    DMAGCHIENAR_T DMAGCHIENAR;
    bit_field_T   :32;
    DREQASIx_T    DMADREQASIX[8];
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;

    DMAPAR0_T    DMAPAR0;
    DMAPAR1_T    DMAPAR1;
    DMAPAR2_T    DMAPAR2;
    DMAPAR3_T    DMAPAR3;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;

    DMAFTCMAP_T  DMAFTCMAP;
    bit_field_T   :32;
    DMALFSMAP_T  DMALFSMAP;
    bit_field_T   :32;
    DMAHBCMAP_T  DMAHBCMAP;
    bit_field_T   :32;
    DMABTCMAP_T  DMABTCMAP;
    bit_field_T   :32;
    DMABERMAP_T  DMABERMAP;
    bit_field_T   :32;
    DMAFTCINTENAS_T DMAFTCINTENAS;
    bit_field_T   :32;
    DMAFTCINTENAR_T DMAFTCINTENAR;
    bit_field_T   :32;
    DMALFSINTENAS_T DMALFSINTENAS;
    bit_field_T   :32;
    DMALFSINTENAR_T DMALFSINTENAR;
    bit_field_T   :32;
    DMAHBCINTENAS_T DMAHBCINTENAS;
    bit_field_T   :32;
    DMAHBCINTENAR_T DMAHBCINTENAR;
    bit_field_T   :32;
    DMABTCINTENAS_T DMABTCINTENAS;
    bit_field_T   :32;
    DMABTCINTENAR_T DMABTCINTENAR;
    bit_field_T   :32;
    DMAGINTFLAG_T   DMAGINTFLAG;
    bit_field_T   :32;
    DMAFTCFLAG_T    DMAFTCFLAG;
    bit_field_T   :32;
    DMALFSFLAG_T    DMALFSFLAG;
    bit_field_T   :32;
    DMAHBCFLAG_T    DMAHBCFLAG;
    bit_field_T   :32;
    DMABTCFLAG_T    DMABTCFLAG;
    bit_field_T   :32;
    DMABERFLAG_T    DMABERFLAG;
    bit_field_T   :32;
    DMAFTCAOFFSET_T DMAFTCAOFFSET;

    DMALFSAOFFSET_T DMALFSAOFFSET;
    DMAHBCAOFFSET_T DMAHBCAOFFSET;
    DMABTCAOFFSET_T DMABTCAOFFSET;
    DMABERAOFFSET_T DMABERAOFFSET;
    DMAFTCBOFFSET_T DMAFTCBOFFSET;
    DMALFSBOFFSET_T DMALFSBOFFSET;
    DMAHBCBOFFSET_T DMAHBCBOFFSET;
    DMABTCBOFFSET_T DMABTCBOFFSET;
    DMABERBOFFSET_T DMABERBOFFSET;
    bit_field_T   :32;
    DMAPTCRL_T      DMAPTCRL;
    DMARTCRL_T      DMARTCRL;
    DMADCTRL_T      DMADCTRL;
    DMAWPR_T        DMAWPR;
    DMAWMR_T        DMAWMR;
    bit_field_T   :32;
    bit_field_T   :32;
    bit_field_T   :32;

    DMAPBACSADDR_T  DMAPBACSADDR;
    DMAPBACDADDR_T  DMAPBACDADDR;
    DMAPBACTC_T     DMAPBACTC;
    bit_field_T   :32;
    DMAPCR_T        DMAPCR;
    DMAPAR_T        DMAPAR;
    DMAMPCTRL_T     DMAMPCTRL;
    DMAMPST_T       DMAMPST;
    DMAMPR0S_T      DMAMPR0S;
    DMAMPR0E_T      DMAMPR0E;
    DMAMPR1S_T      DMAMPR1S;
    DMAMPR1E_T      DMAMPR1E;
    DMAMPR2S_T      DMAMPR2S;
    DMAMPR2E_T      DMAMPR2E;
    DMAMPR3S_T      DMAMPR3S;
    DMAMPR3E_T      DMAMPR3E;

} DMAREG_T;


/*********************************************************************************/

/* DMA Initial Source Address (ISADDR) */
typedef union
{
   struct
   {
      bit_field_T ISADDR        :32;   /* Initial source address    */
   } bits;
   volatile void const *word;
} DMAISADDR_T;

/* DMA Initial Destination Address Register (IDADDR) */
typedef union
{
   struct
   {
      bit_field_T IDADDR        :32;   /* Initial destination address    */
   } bits;
   volatile void  *word;
} DMAIDADDR_T;

/* DMA Initial Transfer Count Register (ITCOUNT) */
typedef union
{
   struct
   {
      bit_field_T                :3;    /* Reserved bit 31 - 29            */
      bit_field_T IFTCOUNT       :13;   /* Initial frame transfer count    */
      bit_field_T                :3;    /* Reserved bit 15 - 13            */
      bit_field_T IETCOUNT       :13;   /* Initial element transfer count  */
   } bits;
   unsigned32_T  word;
} DMAITCOUNT_T;

/* DMA Channel Control Register (CHCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :10;   /* Reserved bit 31 - 22            */
      bit_field_T CHAIN          :6;    /* Next channel to be triggered    */
      bit_field_T RES            :2;    /* Read element size               */
      bit_field_T WES            :2;    /* Write element size              */
      bit_field_T                :3;    /* Reserved bit 11 - 9             */
      bit_field_T TTYPE          :1;    /* Transfer type                   */
      bit_field_T                :3;    /* Reserved bit 7 - 5              */
      bit_field_T ADDMR          :2;    /* Addressing mode read            */
      bit_field_T ADDMW          :2;    /* Addressing mode write           */
      bit_field_T AIM            :1;    /* Auto-initiation mode            */
   } bits;
   unsigned32_T  word;
} DMACHCTRL_T;

/* DMA Element Index Offset Register (EIOFF) */
typedef union
{
   struct
   {
      bit_field_T                :3;    /* Reserved bit 31 - 29               */
      bit_field_T EIDXD          :13;   /* Destination address element index  */
      bit_field_T                :3;    /* Reserved bit 15 - 13               */
      bit_field_T EIDXS          :13;   /* Source address element index       */
   } bits;
   unsigned32_T  word;
} DMAEIOFF_T;

/* DMA Frame Index Offset Register (FIOFF) */
typedef union
{
   struct
   {
      bit_field_T                :3;    /* Reserved bit 31 - 29               */
      bit_field_T FIDXD          :13;   /* Destination address frame index    */
      bit_field_T                :3;    /* Reserved bit 15 - 13               */
      bit_field_T FIDXS          :13;   /* Source address frame index         */
   } bits;
   unsigned32_T  word;
} DMAFIOFF_T;

/* DMA Current Source Address Register (CSADDR) */
typedef union
{
   struct
   {
      bit_field_T CSADDR        :32;   /* Current source address     */
   } bits;
   unsigned32_T  word;
} DMACSADDR_T;

/* DMA Current Destination Address Register (CDADDR) */
typedef union
{
   struct
   {
      bit_field_T CDADDR        :32;   /* Current destination address    */
   } bits;
   unsigned32_T  word;
} DMACDADDR_T;

/* DMA Current Transfer Count Register (CTCOUNT) */
typedef union
{
   struct
   {
      bit_field_T                :3;    /* Reserved bit 31 - 29            */
      bit_field_T CFTCOUNT       :13;   /* Current frame transfer count    */
      bit_field_T                :3;    /* Reserved bit 15 - 13            */
      bit_field_T CETCOUNT       :13;   /* Current element transfer count  */
   } bits;
   unsigned32_T  word;
} DMACTCOUNT_T;

typedef struct
{
    DMAISADDR_T   DMAISADDR;
    DMAIDADDR_T   DMAIDADDR;
    DMAITCOUNT_T  DMAITCOUNT;
    bit_field_T   :32;
    DMACHCTRL_T   DMACHCTRL;
    DMAEIOFF_T    DMAEIOFF;
    DMAFIOFF_T    DMAFIOFF;
    bit_field_T   :32;
} DMAPRIMCHAN_T;

typedef struct
{
  DMAPRIMCHAN_T DMAPRIMCHAN[16];
} DMA_PRIMARY_CONTROL_PACKET_T;


typedef struct
{
    DMACSADDR_T   DMACSADDR;
    DMACDADDR_T   DMACDADDR;
    DMACTCOUNT_T  DMACTCOUNT;
    bit_field_T   :32;
} DMAWORKCHAN_T;

typedef struct
{
  DMAWORKCHAN_T DMAWORKCHAN[16];
} DMA_WORKING_CONTROL_PACKET_T;


/* RTP Global Control Register (RTPGLBCTRL) */
typedef union
{
   struct
   {
      bit_field_T                :7;    /* Reserved bit 15 - 12            */
      bit_field_T TEST           :1;    /* Reserved bit 15 - 12            */
      bit_field_T                :5;    /* Reserved bit 15 - 12            */
      bit_field_T PRESCALER      :3;    /* Reserved bit 15 - 12            */
      bit_field_T                :2;    /* Reserved bit 15 - 12            */
      bit_field_T DDM_WIDTH      :2;    /* Reserved bit 15 - 12            */
      bit_field_T DDM_RW         :1;    /* Reserved bit 15 - 12            */
      bit_field_T TM_DDM         :1;    /* Reserved bit 15 - 12            */
      bit_field_T PW             :2;    /* Reserved bit 15 - 12            */
      bit_field_T RESET          :1;    /* Reserved bit 15 - 12            */
      bit_field_T CONTCLK        :1;    /* Reserved bit 15 - 12            */
      bit_field_T HOVF           :1;    /* Reserved bit 15 - 12            */
      bit_field_T                :1;    /* Reserved bit 15 - 12            */
      bit_field_T ON_OFF         :4;    /* Reserved bit 15 - 12            */
   } bits;
   unsigned32_T  word;
} RTPGLBCTRL_T;

/* RTP Trace Enable Register (RTPTRENA) */
typedef union
{
   struct
   {
      bit_field_T                :7;    /* Reserved bit 31 - 25           */
      bit_field_T ENA4           :1;    /* Enable tracing for FIFO4       */
      bit_field_T                :7;    /* Reserved bit 23 - 17           */
      bit_field_T ENA3           :1;    /* Enable tracing for FIFO3       */
      bit_field_T                :7;    /* Reserved bit 15 - 9            */
      bit_field_T ENA2           :1;    /* Enable tracing for FIFO2       */
      bit_field_T                :7;    /* Reserved bit 7 - 1             */
      bit_field_T ENA1           :1;    /* Enable tracing for FIFO1       */
   } bits;
   unsigned32_T  word;
} RTPTRENA_T;

/* RTP Global Status Register (RTPGSR) */
typedef union
{
   struct
   {
      bit_field_T                :19;   /* Reserved bit 31 - 13     */
      bit_field_T EMPTYSER       :1;    /* Serializer empty         */
      bit_field_T EMPTYPER       :1;    /* Peripheral FIFO empty    */
      bit_field_T EMPTY          :3;    /* FIFO [1] empty           */
      bit_field_T                :4;    /* Reserved bit 7 - 4       */
      bit_field_T OVFPER         :1;    /* Overflow peripheral FIFO */
      bit_field_T OVF            :3;    /* Overflow FIFO [1]        */
   } bits;
   unsigned32_T  word;
} RTPGSR_T;

/* RTP RAM 1 Trace Region [1:2] Register (RTPRAM1REG[1:2]) */
typedef union
{
   struct
   {
      bit_field_T                :1;    /* Reserved bit 31                    */
      bit_field_T CPU_DMA        :2;    /* CPU and/or DMA access              */
      bit_field_T RW             :1;    /* Read/write                         */
      bit_field_T BLOCKSIZE      :4;    /* length of the trace region         */
      bit_field_T                :6;    /* Reserved bit 23 - 18               */
      bit_field_T STARTADDR      :18;   /* start addr of region to be traced  */
   } bits;
   unsigned32_T  word;
} RTPRAMREG_T;

/* RTP Direct Data Mode Write Register (RTPDDMW) */
typedef union
{
   struct
   {
      bit_field_T DATA           :32;   /* data to write in FIFO1*/
   } bits;
   unsigned32_T  word;
} RTPDDMW_T;

/* RTP Pin Control 0 Register (RTPPC0) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19                   */
      bit_field_T ENA_FUNC       :1;    /* Functional mode of RTPENA pin          */
      bit_field_T CLK_FUNC       :1;    /* Functional mode of RTPCLK pin          */
      bit_field_T SYNC_FUNC      :1;    /* Functional mode of RTPSYNC pin         */
      bit_field_T DATA_FUNC      :16;   /* Functional mode of RTPDATA[15:0] pins  */
   } bits;
   unsigned32_T  word;
} RTPPC0_T;

/* RTP Pin Control 1 Register (RTPPC1) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19             */
      bit_field_T RTP_ENA_DIR    :1;    /* Direction of RTPENA pin          */
      bit_field_T RTP_CLK_DIR    :1;    /* Direction of RTPCLK pin          */
      bit_field_T SYNC_DIR       :1;    /* Direction of RTPSYNC pin         */
      bit_field_T DATA_DIR       :16;   /* Direction of RTPDATA[15:0] pins  */
   } bits;
   unsigned32_T  word;
} RTPPC1_T;

/* RTP Pin Control 2 Register (RTPPC2) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19   */
      bit_field_T ENA_IN         :1;    /* RTPENA input           */
      bit_field_T CLK_IN         :1;    /* RTPCLK input           */
      bit_field_T SYNC_IN        :1;    /* RTPSYNC input          */
      bit_field_T DATA_IN        :16;   /* RTPDATA[15:0] input    */
   } bits;
   unsigned32_T  word;
} RTPPC2_T;

/* RTP Pin Control 3 Register (RTPPC3) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19   */
      bit_field_T ENA_OUT        :1;    /* RTPENA output          */
      bit_field_T CLK_OUT        :1;    /* RTPCLK output          */
      bit_field_T SYNC_OUT       :1;    /* RTPSYNC output         */
      bit_field_T DATA_OUT       :16;   /* RTPDATA[15:0] output   */
   } bits;
   unsigned32_T  word;
} RTPPC3_T;

/* RTP Pin Control 4 Register (RTPPC4) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19                    */
      bit_field_T ENA_SET        :1;    /* sets o/p state of RTPENA pin to high    */
      bit_field_T CLK_SET        :1;    /* sets o/p state of RTPCLK pin to high    */
      bit_field_T SYNC_SET       :1;    /* sets o/p state of RTPSYNC pin to high   */
      bit_field_T DATA_SET       :16;   /* sets o/p state of RTPDATA[15:0] pin to high  */
   } bits;
   unsigned32_T  word;
} RTPPC4_T;

/* RTP Pin Control 5 Register (RTPPC5) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19                  */
      bit_field_T ENA_CLR        :1;    /* sets o/p state of RTPENA pin to low   */
      bit_field_T CLK_CLR        :1;    /* sets o/p state of RTPCLK pin to low   */
      bit_field_T SYNC_CLR       :1;    /* sets o/p state of RTPSYNC pin to low  */
      bit_field_T DATA_CLR       :16;   /* sets o/p state of RTPDATA[15:0] pin to low */
   } bits;
   unsigned32_T  word;
} RTPPC5_T;

/* RTP Pin Control 6 Register (RTPPC6) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19            */
      bit_field_T ENA_PDR        :1;    /* RTPENA Open drain enable        */
      bit_field_T CLK_PDR        :1;    /* RTPCLK Open drain enable        */
      bit_field_T SYNC_PDR       :1;    /* RTPSYNC Open drain enable       */
      bit_field_T DATA_PDR       :16;   /* RTPDATA[15:0] Open drain enable */
   } bits;
   unsigned32_T  word;
} RTPPC6_T;

/* RTP Pin Control 7 Register (RTPPC7) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19       */
      bit_field_T ENA_PDIS       :1;    /* RTPENA Pull disable        */
      bit_field_T CLK_PDIS       :1;    /* RTPCLK Pull disable        */
      bit_field_T SYNC_PDIS      :1;    /* RTPSYNC Pull disable       */
      bit_field_T DATA_PDIS      :16;   /* RTPDATA[15:0Pull disable   */
   } bits;
   unsigned32_T  word;
} RTPPC7_T;

/* RTP Pin Control 8 Register (RTPPC8) */
typedef union
{
   struct
   {
      bit_field_T                :13;   /* Reserved bit 31 - 19      */
      bit_field_T ENA_PSEL       :1;    /* RTPENA Pull select        */
      bit_field_T CLK_PSEL       :1;    /* RTPCLK Pull select        */
      bit_field_T SYNC_PSEL      :1;    /* RTPSYNC Pull select       */
      bit_field_T DATA_PSEL      :16;   /* RTPDATA[15:0] Pull select */
   } bits;
   unsigned32_T  word;
} RTPPC8_T;

typedef struct
{
  RTPGLBCTRL_T RTPGLBCTRL;
  RTPTRENA_T   RTPTRENA;
  RTPGSR_T     RTPGSR;
  RTPRAMREG_T  RTPRAM1REG[2];
  RTPRAMREG_T  RTPRAM2REG[2];
  RTPRAMREG_T  RTPRAM3REG[2];
  RTPRAMREG_T  RTPPERREG[2];
  RTPDDMW_T    RTPDDMW;
  unsigned32_T :32;
  RTPPC0_T     RTPPC0;
  RTPPC1_T     RTPPC1;
  RTPPC2_T     RTPPC2;
  RTPPC3_T     RTPPC3;
  RTPPC4_T     RTPPC4;
  RTPPC5_T     RTPPC5;
  RTPPC6_T     RTPPC6;
  RTPPC7_T     RTPPC7;
  RTPPC8_T     RTPPC8;
} RTPREG_T;

/* ESM Influence Error Pin Set/Status Register 1 (ESMIEPSR1) */
typedef union
{
   struct
   {
      bit_field_T IEPSET31       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET30       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET29       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET28       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET27       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET26       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET25       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET24       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET23       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET22       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET21       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET20       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET19       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET18       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET17       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET16       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET15       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET14       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET13       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET12       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET11       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET10       :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET9        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET8        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET7        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET6        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET5        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET4        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET3        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET2        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET1        :1;    /* Set Influence on Error Pin   */
      bit_field_T IEPSET0        :1;    /* Set Influence on Error Pin   */
   } bits;
   unsigned32_T  word;
} ESMIEPSR1_T;

/* ESM Influence Error Pin Clear/Status Register 1 (ESMIEPCR1) */
typedef union
{
   struct
   {
      bit_field_T IEPCLR31       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR30       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR29       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR28       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR27       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR26       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR25       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR24       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR23       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR22       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR21       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR20       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR19       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR18       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR17       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR16       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR15       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR14       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR13       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR12       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR11       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR10       :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR9        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR8        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR7        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR6        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR5        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR4        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR3        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR2        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR1        :1;    /* Clear Influence on Error Pin  */
      bit_field_T IEPCLR0        :1;    /* Clear Influence on Error Pin  */
   } bits;
   unsigned32_T  word;
} ESMIEPCR1_T;

/* ESM Interrupt Enable Set/Status Register 1 (ESMIESR1) */
typedef union
{
   struct
   {
      bit_field_T INTENSET31       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET30       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET29       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET28       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET27       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET26       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET25       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET24       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET23       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET22       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET21       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET20       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET19       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET18       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET17       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET16       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET15       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET14       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET13       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET12       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET11       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET10       :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET9        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET8        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET7        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET6        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET5        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET4        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET3        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET2        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET1        :1;    /* Set interrupt Enable            */
      bit_field_T INTENSET0        :1;    /* Set interrupt Enable            */
   } bits;
   unsigned32_T  word;
} ESMIESR1_T;

/* ESM Interrupt Enable Clear/Status Register 1 (ESMIECR1) */
typedef union
{
   struct
   {
      bit_field_T INTENCLR31       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR30       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR29       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR28       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR27       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR26       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR25       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR24       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR23       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR22       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR21       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR20       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR19       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR18       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR17       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR16       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR15       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR14       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR13       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR12       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR11       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR10       :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR9        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR8        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR7        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR6        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR5        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR4        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR3        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR2        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR1        :1;    /* Clear interrupt Enable            */
      bit_field_T INTENCLR0        :1;    /* Clear interrupt Enable            */
   } bits;
   unsigned32_T  word;
} ESMIECR1_T;

/* ESM Interrupt Level Set/Status Register 1 (ESMILSR1) */
typedef union
{
   struct
   {
      bit_field_T INTLVLSET31       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET30       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET29       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET28       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET27       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET26       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET25       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET24       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET23       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET22       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET21       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET20       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET19       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET18       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET17       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET16       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET15       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET14       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET13       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET12       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET11       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET10       :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET9        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET8        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET7        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET6        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET5        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET4        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET3        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET2        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET1        :1;    /* Set Interrupt Level            */
      bit_field_T INTLVLSET0        :1;    /* Set Interrupt Level            */
   } bits;
   unsigned32_T  word;
} ESMILSR1_T;

/* ESM Interrupt Level Clear/Status Register 1 (ESMILCR1) */
typedef union
{
   struct
   {
      bit_field_T INTLVLCLR31       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR30       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR29       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR28       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR27       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR26       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR25       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR24       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR23       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR22       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR21       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR20       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR19       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR18       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR17       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR16       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR15       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR14       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR13       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR12       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR11       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR10       :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR9        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR8        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR7        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR6        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR5        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR4        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR3        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR2        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR1        :1;    /* Clear Interrupt Level            */
      bit_field_T INTLVLCLR0        :1;    /* Clear Interrupt Level            */
   } bits;
   unsigned32_T  word;
} ESMILCR1_T;

/* ESM Status Register 1 (ESMSR1) */
typedef union
{
   struct
   {
      bit_field_T ESF31       :1;    /* Error Status Flag            */
      bit_field_T ESF30       :1;    /* Error Status Flag            */
      bit_field_T ESF29       :1;    /* Error Status Flag            */
      bit_field_T ESF28       :1;    /* Error Status Flag            */
      bit_field_T ESF27       :1;    /* Error Status Flag            */
      bit_field_T ESF26       :1;    /* Error Status Flag            */
      bit_field_T ESF25       :1;    /* Error Status Flag            */
      bit_field_T ESF24       :1;    /* Error Status Flag            */
      bit_field_T ESF23       :1;    /* Error Status Flag            */
      bit_field_T ESF22       :1;    /* Error Status Flag            */
      bit_field_T ESF21       :1;    /* Error Status Flag            */
      bit_field_T ESF20       :1;    /* Error Status Flag            */
      bit_field_T ESF19       :1;    /* Error Status Flag            */
      bit_field_T ESF18       :1;    /* Error Status Flag            */
      bit_field_T ESF17       :1;    /* Error Status Flag            */
      bit_field_T ESF16       :1;    /* Error Status Flag            */
      bit_field_T ESF15       :1;    /* Error Status Flag            */
      bit_field_T ESF14       :1;    /* Error Status Flag            */
      bit_field_T ESF13       :1;    /* Error Status Flag            */
      bit_field_T ESF12       :1;    /* Error Status Flag            */
      bit_field_T ESF11       :1;    /* Error Status Flag            */
      bit_field_T ESF10       :1;    /* Error Status Flag            */
      bit_field_T ESF9        :1;    /* Error Status Flag            */
      bit_field_T ESF8        :1;    /* Error Status Flag            */
      bit_field_T ESF7        :1;    /* Error Status Flag            */
      bit_field_T ESF6        :1;    /* Error Status Flag            */
      bit_field_T ESF5        :1;    /* Error Status Flag            */
      bit_field_T ESF4        :1;    /* Error Status Flag            */
      bit_field_T ESF3        :1;    /* Error Status Flag            */
      bit_field_T ESF2        :1;    /* Error Status Flag            */
      bit_field_T ESF1        :1;    /* Error Status Flag            */
      bit_field_T ESF0        :1;    /* Error Status Flag            */
   } bits;
   unsigned32_T  word;
} ESMSRx_T;

/* ESM Error Pin Status Register (ESMEPSR) */
typedef union
{
   struct
   {
      bit_field_T             :31;   /* Reserved bit 31 - 1      */
      bit_field_T EPSF        :1;    /* Error Pin Status Flag    */
   } bits;
   unsigned32_T  word;
} ESMEPSR_T;

/* ESM Interrupt Offset High Register (ESMIOFFHR) */
typedef union
{
   struct
   {
      bit_field_T             :25;   /* Reserved bit 31 - 7           */
      bit_field_T INTOFFH     :7;    /* Offset High Level Interrupt   */
   } bits;
   unsigned32_T  word;
} ESMIOFFHR_T;

/* ESM Interrupt Offset Low Register (ESMIOFFLR) */
typedef union
{
   struct
   {
      bit_field_T             :25;   /* Reserved bit 31 - 7         */
      bit_field_T INTOFFL     :7;    /* Offset Low Level Interrupt  */
   } bits;
   unsigned32_T  word;
} ESMIOFFLR_T;

/* ESM Low-Time Counter Register (ESMLTCR) */
typedef union
{
   struct
   {
      bit_field_T             :16;   /* Reserved bit 31 - 6            */
      bit_field_T TLTC        :16;   /* Error Pin Low-Time Counter     */
   } bits;
   unsigned32_T  word;
} ESMLTCR_T;

/* ESM Low-Time Counter Preload Register (ESMLTCPR) */
typedef union
{
   struct
   {
      bit_field_T             :16;   /* Reserved bit 31 - 16                        */
      bit_field_T TLCP        :2;    /* Error pin Low-Time Counter Pre-load Value   */
      bit_field_T TLCP_FIXED  :14;   /* Error pin Low-Time Counter Pre-load Value   */
   } bits;
   unsigned32_T  word;
} ESMLTCPR_T;

/* ESM Error Key Register (ESMEKR) */
typedef union
{
   struct
   {
      bit_field_T             :28;   /* Reserved bit 31 - 4  */
      bit_field_T EKEY        :4;    /* Error Key            */
   } bits;
   unsigned32_T  word;
} ESMEKR_T;

typedef struct
{
  ESMIEPSR1_T  ESMIEPSR1;
  ESMIEPCR1_T  ESMIEPCR1;
  ESMIESR1_T   ESMIESR1;
  ESMIECR1_T   ESMIECR1;
  ESMILSR1_T   ESMILSR1;
  ESMILCR1_T   ESMILCR1;
  ESMSRx_T     ESMSRx[3];
  ESMEPSR_T    ESMEPSR;
  ESMIOFFHR_T  ESMIOFFHR;
  ESMIOFFLR_T  ESMIOFFLR;
  ESMLTCR_T    ESMLTCR;
  ESMLTCPR_T   ESMLTCPR;
  ESMEKR_T     ESMEKR;
  ESMSRx_T     ESMSSR2;
} ESMREG_T;

COMPILER_ASSERT(0x40==sizeof(ESMREG_T), ESMREG_T_Must_be_0x40_bytes);

/*********************************************************************************/

/*********************************************************************************/

typedef enum
{
/***************************************************************************
 **
 **  CIM interrupt channels
 **
 ***************************************************************************/
        CIM_SPI1      , /* SPI1 end-transfer/overrun          */
        CIM_COMP2     , /* COMP2 interrupt                    */
        CIM_COMP1     , /* COMP1 interrupt                    */
        CIM_TAP       , /* TAP interrupt                      */
        CIM_SPI2      , /* SPI2 end-transfer/overrun          */
        CIM_GIOA      , /* GIO interrupt A                    */
        CIM_RES1      , /* --- Reserved ---                   */
        CIM_HET1      , /* HET interrupt 1                    */
        CIM_I2C1      , /* I2C1 interrupt                     */
        CIM_SCIRXERR  , /* SCI1 or SCI2 error interrupt       */
        CIM_SCI1RX    , /* SCI1 receive interrupt             */
        CIM_RES2      , /* --- Reserved ---                   */
        CIM_I2C2      , /* I2C2 interrupt                     */
        CIM_HECC1A    , /* HECC1 interrupt A                  */
        CIM_SCCA      , /* SCC interrupt A                    */
        CIM_RES3      , /* --- Reserved ---                   */
        CIM_MIBADCEE  , /* MibADC end event conversion        */
        CIM_SCI2RX    , /* SCI2 receive interrupt             */
        CIM_DMA0      , /* DMA interrupt 0                    */
        CIM_I2C3      , /* I2C3 interrupt                     */
        CIM_SCI1TX    , /* SCI1 transmit interrupt            */
        CIM_SSI       , /* SW interrupt (SSI)                 */
        CIM_RES4      , /* --- Reserved ---                   */
        CIM_HET2      , /* HET interrupt 2                    */
        CIM_HECC1B    , /* HECC1 interrupt B                  */
        CIM_SCCB      , /* SCC interrupt B                    */
        CIM_SCI2TX    , /* SCI2 transmit interrupt            */
        CIM_MIBADCE1  , /* MibADC end Group 1 conversion      */
        CIM_DMA1      , /* DMA Interrupt 1                    */
        CIM_GIOB      , /* GIO interrupt B                    */
        CIM_MIBADCE2  , /* MibADC end Group 2 conversion      */
        CIM_SCI3        /* SCI3 error interrupt               */
} CIM_REQMASK_T;


/* CLKDIV bit field of PCR Register */
typedef enum
{
   SYS_CLOCK_DIV_1,     /* 0000 = SYSCLK/1 = ICLK */
   SYS_CLOCK_DIV_2,     /* 0001 = SYSCLK/2 = ICLK */
   SYS_CLOCK_DIV_3,     /* 0002 = SYSCLK/3 = ICLK */
   SYS_CLOCK_DIV_4,     /* 0003 = SYSCLK/4 = ICLK */
   SYS_CLOCK_DIV_5,     /* 0004 = SYSCLK/5 = ICLK */
   SYS_CLOCK_DIV_6,     /* 0005 = SYSCLK/6 = ICLK */
   SYS_CLOCK_DIV_7,     /* 0006 = SYSCLK/7 = ICLK */
   SYS_CLOCK_DIV_8,     /* 0007 = SYSCLK/8 = ICLK */
   SYS_CLOCK_DIV_9,     /* 0008 = SYSCLK/9 = ICLK */
   SYS_CLOCK_DIV_10,    /* 0009 = SYSCLK/10 = ICLK */
   SYS_CLOCK_DIV_11,    /* 0010 = SYSCLK/11 = ICLK */
   SYS_CLOCK_DIV_12,    /* 0011 = SYSCLK/12 = ICLK */
   SYS_CLOCK_DIV_13,    /* 0012 = SYSCLK/13 = ICLK */
   SYS_CLOCK_DIV_14,    /* 0013 = SYSCLK/14 = ICLK */
   SYS_CLOCK_DIV_15,    /* 0014 = SYSCLK/15 = ICLK */
   SYS_CLOCK_DIV_16     /* 0015 = SYSCLK/16 = ICLK */
} SYSCLK_ICLK_ratio_T;

/******************************
* Calibrations
*******************************/


/******************************
* Global Variables
*******************************/

/*   None                     */

/*************************
* Interface Declarations
*************************/

/*   None                     */



/******************************
* Function Prototypes
*******************************/

/*   None                     */

/*===========================================================================*\
 * File Revision History (top to bottom: first revision to last revision)
 *============================================================================
 *   Date       By     (Description on following lines)
 * --------  --------
 * 11/08/10  P. Roy SCR tcb_css#XXXX for release DBC_Layer2/78_12KC13
 *  Initial version with History Log
 *
 * 10/19/10 Prajakta M SCR tcb_css#4494 for release DBC_Layer2/78_12KC13.
 *  Updated parentheses of macros used for SPI FMT setting for SCR 4494 .
 *
 * 10/29/10 Prajakta M SCR tcb_css#4495 for release DBC_Layer2/78_12KC13.
 *  Added macros and updated bitfields with comments. Deleted macros and bitfield/structures related to VF338.
 *
 * 01/20/11 P. Roy   SCR tcb_css#XXXX for release TMS470_Registers/12V_1.48.
 *  Updated for compiling with Boot
 *
 * 04/09/11 Nandu   Cradle scr.221  for release TMS470_Registers/12V_1.50.
 *  Added macros
 *
 * 05/11/11 P. Roy  SCR tcb_css#XXXX for release DBC7p8_Boot/10_BOOT_12C
 *   Corrected for PBIST and CCM related macros
 *
 * 06/10/11 P. Roy  SCR tcb_css#XXXX for release DBC7p8_Boot/10_BOOT_12C
 *   Merged files from different branches
 *
 * 06/01/11  Dipti Cradle SCR #293 for release DBC7p8_Boot/10_BOOT_12C
 *  Added macros for Flash test.
 *
 * 06/17/11  Dipti  Cradle SCR 267 for Release DBC7p8_Boot/10_BOOT_12C
 *   Added macros for 12V BOOT DMA Initialization
 *
 * 07/07/11  P. Roy Cradle SCR #233 for release DBC7p8_Boot/10_BOOT_12C
 *  Added macros for PBIST Test Result
 *
 * 11/03/11  P. Roy Cradle SCR #XXX for release DBC7p8_Boot/10_BOOT_12C
 *  Added CCM7 related Macros as well as PLL Control Macros
 *
 * 12/09/11  P. Roy Cradle SCR 522 for release TMS470_Registers/12V_1.55
 *  Added RTI2 related macros
 *
 * 01/17/12  P. Roy Cradle SCR XXX for release TMS470_Registers/12V_1.55
 *  Added DMA Parity check related macros
 *
\*===========================================================================*/

#endif  /* SYSREG_H */
