/**
 * @memXfer.js - DSS Generic TI Loader include file that contains functions
 * used by main.js for host to target memory data transfer.
 */

/**
 * Load data from file(s) on host to target memory.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {loadRaw} Array of strings, where each string describes a raw file to
 * memory transfer.
 * @param {loadDat} Array of strings, where each string describes a text-based
 * file to memory transfer.
 *@param{maxLength]} nb of data to transfer, in case the value given by script is not known at launch time
 *@param{NbXfers} : number of time the transfer alreay occcurred
 */
 
function memLoad(dssScriptEnv, debugSession, loadRaw, loadDat,NbXfers)
{
    var errCode = 0;
    var xferNum = 0;

    if (loadRaw.length > 0)
    {
		if (NbXfers == -1) // synchronous mode, load multiple data set
		{
			for (xferNum = 0; xferNum < loadRaw.length; xferNum++)
			{
				errCode = loadRawData(dssScriptEnv, debugSession, loadRaw[xferNum]);	
			}
		}
		else // async mode, load 1 set of data upon specific function call
		{		
			errCode = loadRawData(dssScriptEnv, debugSession, loadRaw[NbXfers]);	
		}
    }

    if (loadDat.length > 0)
    {
		if (NbXfers == -1) // synchronous mode, load multiple data set
		{
			for (xferNum = 0; xferNum < loadDat.length; xferNum++)
			{
				errCode = loadDatData(dssScriptEnv, debugSession, loadDat[NbXfers]);
 			}
		}
		else // async mode, load 1 set of data upon specific function call
		{		
			errCode = loadDatData(dssScriptEnv, debugSession, loadDat[NbXfers]);	
		}
   }

    return errCode;
}

/**
 * Save memory on target to a file(s) on the host.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {saveRaw} Array of strings, where each string describes a memory to
 * raw file transfer.
 * @param {saveDat} Array of strings, where each string describes a memory to
 * text-based file transfer.
*@param{maxLength]} nb of data to transfer, in case the value given by script is not known at launch time
 *@param{NbXfers} : number of time the transfer alreay occcurred
 */
function memSave(dssScriptEnv, debugSession, saveRaw, saveDat,NbXfers)
{
    var errCode = 0;
    var xferNum = 0;

    if (saveRaw.length > 0)
    {
		if (NbXfers == -1) // synchronous mode, load multiple data set
		{
			for (xferNum = 0; xferNum < saveRaw.length; xferNum++)
			{
				errCode = saveRawData(dssScriptEnv, debugSession, saveRaw[xferNum]);
			}
		}
		else // async mode, load 1 set of data upon specific function call
		{
			errCode = saveRawData(dssScriptEnv, debugSession, saveRaw[NbXfers]);
	    }
	}
    if (saveDat.length > 0)
    {
		if (NbXfers == -1) // synchronous mode, load multiple data set
		{
			for (xferNum = 0; xferNum < saveDat.length; xferNum++)
			{
				errCode = saveDatData(dssScriptEnv, debugSession, saveDat[xferNum]);
			}
		}
		else // async mode, load 1 set of data upon specific function call
		{
 			errCode = saveRawData(dssScriptEnv, debugSession, saveDat[NbXfers]);
	    }
   }

    return errCode;
}

/**
 * Load binary data from file on host to target memory.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {parameters} String containing parameters to the actual load, with
 * format "[page],[address],[filename],[type_size],[byte_swap]".
 */
function loadRawData(dssScriptEnv, debugSession, parameters)
{
    var errCode = 0;

    matchResult = parameters.match(/([0-4]|65535),(.+),(.+),(8|16|32),(true|false)/);
    if ((matchResult == null))
    {
        dssScriptEnv.traceWrite("--mem_load_raw parameters: " + parameters +
				" option are invalid! see loadti --help for parameter specification");
        errCode = 1;
    }
	else
	{
        var nPage = parseInt(RegExp.$1);
		var nAddress ;
		var str = RegExp.$2;
		var sFilename = RegExp.$3;
        var nTypeSize = parseInt(RegExp.$4);
        var bByteSwap = false;

        if (RegExp.$5 == "true")
        {
            bByteSwap = true;
        }

		var patt1=/^0x/g;
        if (patt1.test(str) ) {
			nAddress = parseInt(str);
			dssScriptEnv.traceWrite("Address given as a value");
		}
		else  
		{  // need to evaluate varaiable value
			dssScriptEnv.traceWrite("Address given as a variable");
			try
			{
				nAddress = debugSession.expression.evaluate(str);
			}
			catch (ex)
			{
				errCode = getErrorCode(ex);
				dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate input address!");
				return errCode;
			}
		}

		dssScriptEnv.traceWrite("Load Data file " + sFilename + " at address " +  nAddress);
        try
		{
            debugSession.memory.loadRaw(nPage, nAddress, sFilename, nTypeSize, bByteSwap);
        }
		catch (ex)
		{
			errCode = getErrorCode(ex);
            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not load file " + sFilename +
					" to target memory address " + nAddress);
        }
    }

    return errCode;
}

/**
 * Load text-based data from file on host to target memory.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {parameters} String containing parameters to the actual load, with
 * format "[page],[address],[filename],[len]".
 */
function loadDatData(dssScriptEnv, debugSession, parameters)
{
    var errCode = 0;

    matchResult = parameters.match(/([0-4]|65535),(.+),(.+),(.+)/);

    if ((matchResult == null))
    {
        dssScriptEnv.traceWrite("--mem_load_dat parameters: " + parameters +
				" option are invalid! see loadti --help for parameter specification");
        errCode = 1;
    }
	else
	{
        var nPage = parseInt(RegExp.$1);
 		var nAddress ;
		var str = RegExp.$2;
		var sFilename = RegExp.$3;

        var nLength = parseInt(RegExp.$4);

		var patt1=/^0x/g;
        if (patt1.test(str) )
			nAddress = parseInt(str);
		else // need to evaluate varaiable value
		{
			try
			{
				nAddress = debugSession.expression.evaluate(str);
			}
			catch (ex)
			{
				errCode = getErrorCode(ex);
				dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate input address!");
				return errCode;
			}
		}
		dssScriptEnv.traceWrite("Load " + nLength + " bytes from Data file " + sFilename + " at address " +  nAddress);
        try
		{
            debugSession.memory.loadData(nPage, nAddress, sFilename, nLength);
        }
		catch (ex)
		{
			errCode = getErrorCode(ex);
            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not load file " + sFilename +
					" to target memory!");
        }
    }

    return errCode;
}

/**
 * Save memory on target to a binary file on the host.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {parameters} String containing parameters to the actual save, with
 * format "[page],[address],[filename],[len],[type_size],[byte_swap]".
 */
function saveRawData(dssScriptEnv, debugSession, parameters)
{
    var errCode = 0;

    matchResult = parameters.match(/([0-4]|65535),(.+),(.+),(.+),(8|16|32),(true|false)/);

    if ((matchResult == null))
    {
        dssScriptEnv.traceWrite("--mem_save_raw parameters: " + parameters +
				" option are invalid! see loadti --help for parameter specification");
        errCode = 1;
    }
	else
	{
        var nPage = parseInt(RegExp.$1);
 		// check if addres sis hexa decimal or  variable name
		var nAddress ;
		var str = RegExp.$2;
        var sFilename = RegExp.$3;
		var str2 = RegExp.$4;
        var nLength ;
        var nTypeSize = parseInt(RegExp.$5);
        var bByteSwap = false;
        if (RegExp.$6 == "true")
        {
            bByteSwap = true;
        }
		dssScriptEnv.traceWrite("Input var name:" + str);
		if (str == '')
		{
			errCode = 1;
			dssScriptEnv.traceWrite("Input parameter address / name incorrect - Abort!");
		}else
		{
			var patt1=/^0x/g;
	        if (patt1.test(str) )
				nAddress = parseInt(str);
			else // need to evaluate varaiable value
			{
				try
				{
					nAddress = debugSession.expression.evaluate(str);
				}
				catch (ex)
				{
					errCode = getErrorCode(ex);
		            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate input address!");
					return errCode;
		        }
				
			}
			patt1=/^[0-9]/g;
	        if (patt1.test(str2) )
				nLength = parseInt(str2);
			else 
			{ // need to evaluate varaiable value
				try
				{
					nLength = debugSession.expression.evaluate(str2);
				}
				catch (ex)
				{
					errCode = getErrorCode(ex);
		            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate length!");
					return errCode;
		        }
				 if (nLength == 0)
				 {
					dssScriptEnv.traceWrite("Number of data to save is 0 ");
					return 0;
				 }
			}

	        try
			{
				dssScriptEnv.traceWrite("Save " + nLength + " bytes to Data file " + sFilename + " from address " +  nAddress);        
	            debugSession.memory.saveRaw(nPage, nAddress, sFilename, nLength, nTypeSize, bByteSwap);
			}
			catch (ex)
			{
				errCode = getErrorCode(ex);
	            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not save memory to host!");
	        }
		}
    }

    return errCode;
}

/**
 * Save memory on target to a text-based file on the host.
 *
 * @param {dssScriptEnv} DSS Scripting Environment object.
 * @param {debugSession} Debug Session object.
 * @param {parameters} String containing parameters to the actual save, with
 * format "[page],[address],[filename],[len],[io_format],[append]".
 */
function saveDatData(dssScriptEnv, debugSession, parameters)
{
    var errCode = 0;

    matchResult = parameters.match(/([0-4]|65535),(.+),(.+),(.+),([1-5]),(true|false)/);

    if ((matchResult == null))
    {
        dssScriptEnv.traceWrite("--mem_save_dat parameters: " + parameters +
				" option are invalid! see loadti --help for parameter specification");
        errCode = 1;
    }
	else
	{
        var nPage = parseInt(RegExp.$1);
		var nAddress ;
		var str = RegExp.$2;
		var sFilename = RegExp.$3;
         var nLength ;
		var str2 = RegExp.$4;
		var nIOFormat = parseInt(RegExp.$5);
        var bAppend = false;
		// modification to addan max size to be transferred

        if (RegExp.$6 == "true")
        {
            bAppend = true;
        }

		dssScriptEnv.traceWrite("Input var name:" + str);
		if (str == '')
		{
			errCode = 1;
			dssScriptEnv.traceWrite("Input parameter address / name incorrect - Abort!");
		}else
		{
			var patt1=/^0x/g;
	        if (patt1.test(str) )
				nAddress = parseInt(str);
			else // need to evaluate varaiable value
			{
				try
				{
					nAddress = debugSession.expression.evaluate(str);
				}
				catch (ex)
				{
					errCode = getErrorCode(ex);
		            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate input address!");
					return errCode;
		        }
			}
			
			patt1=/^[0-9]/g;
	        if (patt1.test(str2) )
				nLength = parseInt(str2);
			else // need to evaluate varaiable value
			{
				try
				{
					nLength = debugSession.expression.evaluate(str2);
				}
				catch (ex)
				{
					errCode = getErrorCode(ex);
		            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not evaluate length!");
					return errCode;
		        }
				 if (nLength == 0)
				 {
					dssScriptEnv.traceWrite("Number of data to save is 0 ");
					return 0;
				 }
			}

	        try
			{
				dssScriptEnv.traceWrite("Save " + nLength + " bytes to Data file " + sFilename + " from address " +  nAddress);        
	            debugSession.memory.saveData(nPage, nAddress, sFilename, nLength, nIOFormat, bAppend);
	        }
			catch (ex)
			{
				errCode = getErrorCode(ex);
	            dssScriptEnv.traceWrite("Error code #" + errCode + ", could not save memory to host!");
	        }
		}
    }

    return errCode;
}
