/*
 *  Copyright (c) Texas Instruments Incorporated 2018
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \file ipc_testsetup.c
 *
 *  \brief IPC  example code
 *  
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <pthread.h>
#include <semaphore.h>
#include <unix.h>

#include <stdio.h>
#include <sys/neutrino.h>
#include <stdlib.h>

#include <ti/drv/ipc/ipc.h>
#include <ti/osal/TaskP.h>
#include "ipc_setup.h"

#define MSGSIZE  256
#define SERVICE  "ti.ipc4.ping-pong"
#define ENDPT1   13
#define NUMMSGS  10 /* number of message sent per task */

extern uint8_t  *pCntrlBuf;
extern uint8_t  *pTaskBuf;
extern uint8_t  *pSendTaskBuf;
extern uint8_t  *pRecvTaskBuf;
extern uint8_t  *pSysVqBuf;

extern uint32_t  selfProcId;
extern uint32_t *pRemoteProcArray;
extern uint32_t  gNumRemoteProc;

uint32_t rpmsgDataSize = RPMSG_DATA_SIZE;

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */


/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */


/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
#if !defined(BUILD_MPU1_0) && defined(A72_LINUX_OS) && defined(A72_LINUX_OS_IPC_ATTACH)
static uint32_t		RecvEndPt = 0;
#endif

//#define DEBUG_PRINT

bool g_exitRespTsk = 0;

void* rpmsg_exit_responseTask()
{
    g_exitRespTsk = 1;
    return(0);
}

/*
 * This "Task" waits for a "ping" message from any processor
 * then replies with a "pong" message.
 */
void* rpmsg_responderFxn(void* arg)
{
    RPMessage_Handle    handle;
    RPMessage_Params    params;
    uint32_t		myEndPt = 0;
    uint32_t		remoteEndPt;
    uint32_t		remoteProcId;
    uint16_t		len;
    int32_t		n;
    int32_t		status = 0;
    void		*buf;
    
    uint32_t            bufSize = rpmsgDataSize;
    char                str[MSGSIZE];

    uint8_t     recvBuf[RPMSG_DATA_SIZE]  __attribute__ ((aligned (8)));

    buf = recvBuf;
    if(buf == NULL) 
    {
        printf("RecvTask: buffer allocation failed\n");
	    return(0);
    }

    RPMessageParams_init(&params);
    params.requestedEndpt = ENDPT1;
    params.buf = buf;
    params.bufSize = bufSize;

    handle = RPMessage_create(&params, &myEndPt);
    if(!handle) 
    {
        printf("RecvTask: Failed to create endpoint\n");
        return(0);
    }

#if !defined(BUILD_MPU1_0) && defined(A72_LINUX_OS) && defined(A72_LINUX_OS_IPC_ATTACH)
    RecvEndPt = myEndPt;
#endif

    status = RPMessage_announce(RPMESSAGE_ALL, myEndPt, SERVICE);
    if(status != IPC_SOK) 
    {
	    printf("RecvTask: RPMessage_announce() failed\n");
        return(0);
    }

    while(!g_exitRespTsk)
    {
        //printf("RecvTask: Waiting for message...\n");
	    status = RPMessage_recv(handle, (Ptr)str, &len, &remoteEndPt, &remoteProcId,
                IPC_RPMESSAGE_TIMEOUT_FOREVER);
	    if(status != IPC_SOK)
        {
	    }
        else
        {
            /* NULL terminated string */
            str[len] = '\0';
#ifdef DEBUG_PRINT
	        printf("RecvTask: Revcvd msg \"%s\" len %d from %s\n",
                     str, len, Ipc_mpGetName(remoteProcId));
#endif
        }

    /* do not do anything */
    if(selfProcId == remoteProcId)
    {
        delay(10000);
        continue;
    }

	status = sscanf(str, "ping %d", &n);
	if(status == 1)
        {
	    len = snprintf(str, MSGSIZE-1, "pong %d", n);
	}
        else
        {
            /* If this is not ping/pong message, just print the message */
            printf("%s <--> %s : %s recvd\n",
                Ipc_mpGetSelfName(),
                Ipc_mpGetName(remoteProcId),
                str);
        }
#ifdef DEBUG_PRINT
	printf("RecvTask: Sending msg \"%s\" len %d from %s to %s\n",
                     str, len, Ipc_mpGetSelfName(),
                     Ipc_mpGetName(remoteProcId));
#endif
	status = RPMessage_send(handle, remoteProcId, remoteEndPt, myEndPt, str, len);
	if (status != IPC_SOK)
        {
	    printf("RecvTask: RPMessage_send "
			" failed status %d procId %d\n", status, remoteProcId);
	}
    }
    return(0);
}

void* rpmsg_senderFxn(void* arg_array)
{
    RPMessage_Handle    handle;
    RPMessage_Params    params;
    uint32_t            myEndPt = 0;
    uint32_t            remoteEndPt = ENDPT1;
    uint32_t            remoteProcId;
    uint16_t            dstProc;
    uint16_t            len;
    uint32_t            i;
    int32_t             status = 0;
    char                buf[256];   
    uint8_t            *buf1;

    uint32_t            cntPing = 0;
    uint32_t            cntPong = 0;
    uint32_t           *arg0;
    uint32_t           *arg1;

    arg0 = (uint32_t *) *(uint64_t *)arg_array;
    arg1 = (uint32_t *) *(uint64_t *)(arg_array + sizeof(void*));

    buf1 = &pSendTaskBuf[rpmsgDataSize * (*arg1)];
    dstProc = (uint16_t) *arg0;

    remoteProcId = dstProc;
    /* Create the endpoint for receiving. */
    RPMessageParams_init(&params);
    params.numBufs = 2;
    params.buf = buf1;
    params.bufSize = rpmsgDataSize;
    handle = RPMessage_create(&params, &myEndPt);
    if(!handle) 
    {
        printf("SendTask %d: Failed to create message endpoint\n",
             dstProc);
        return(0);
    }

    status = RPMessage_getRemoteEndPt(dstProc, SERVICE, &remoteProcId,
                             &remoteEndPt, WAIT_FOREVER);
    if(dstProc != remoteProcId) 
    {
        printf("SendTask%d: RPMessage_getRemoteEndPt() malfunctioned, status %d\n",
             dstProc, status);
        return(0);
    }

    for (i = 0; i < NUMMSGS; i++)
    {
        /* Send data to remote endPt: */
	    len = snprintf(buf, MSGSIZE-1, "ping %d", i);
	    buf[len++] = '\0';

#ifdef DEBUG_PRINT
        printf("SendTask%d: Sending \"%s\" from %s to %s...\n", dstProc,
            buf, Ipc_mpGetSelfName(),
            Ipc_mpGetName(dstProc));
#endif
        /* Increase the Ping Counter */
        cntPing++;

	    status = RPMessage_send(handle, dstProc, ENDPT1, myEndPt, (Ptr)buf, len);
        if (status != IPC_SOK)
        {
            printf("SendTask%d: rpmsg_senderFxn: RPMessage_send "
	        " failed status %d\n", dstProc, status);
	    }

        /* wait a for a response message: */
        status = RPMessage_recv(handle, (Ptr)buf, &len, &remoteEndPt,
                         &remoteProcId, IPC_RPMESSAGE_TIMEOUT_FOREVER);

        if(status != IPC_SOK)
        {
            printf("SendTask%d: RPMessage_recv failed with code %d\n",
                    dstProc, status);
        }

        /* Make it NULL terminated string */
        if(len >= MSGSIZE)
        {
            buf[MSGSIZE-1] = '\0';
        }
        else
        {
            buf[len] = '\0';
        }
#ifdef DEBUG_PRINT
	    printf("SendTask%d: Received \"%s\" len %d from %s endPt %d \n",
	           dstProc, buf, len, Ipc_mpGetName(remoteProcId),
                   remoteEndPt);
#endif
        cntPong++;
        if((i+1)%50 == 0)
        {
            printf("%s <--> %s, ping/pong iteration %d ...\n",
                    Ipc_mpGetSelfName(), Ipc_mpGetName(dstProc), i);
        }
    }

    printf("SendTask%d: %s <--> %s, Ping- %d, pong - %d completed\n",
             dstProc, Ipc_mpGetSelfName(),
             Ipc_mpGetName(dstProc),
             cntPing, cntPong);

    /* Delete the RPMesg object now */
    RPMessage_delete(&handle);
    return(0);
}


uint32_t remoteProcNum[IPC_MAX_PROCS];

int32_t Ipc_echo_test(void)
{
    uint32_t          t; 
    TaskP_Params      params;
    uint32_t          numProc = gNumRemoteProc;
    Ipc_VirtIoParams  vqParam;
    uint32_t          index = 0;


    /* Step1 : Initialize the multiproc */
    Ipc_mpSetConfig(selfProcId, numProc, pRemoteProcArray);

    printf("IPC_echo_test (core : %s) .....\r\n",
         Ipc_mpGetSelfName());

    Ipc_init(NULL);

    printf("IPC init done\n");


    /* Step2 : Initialize Virtio */
    vqParam.vqObjBaseAddr = (void*)pSysVqBuf;
    vqParam.vqBufSize     = numProc * Ipc_getVqObjMemoryRequiredPerCore();
    vqParam.vringBaseAddr = (void*)VRING_BASE_ADDRESS;
    vqParam.vringBufSize  = IPC_VRING_BUFFER_SIZE;
    vqParam.timeoutCnt    = 100;  /* Wait for counts */
    Ipc_initVirtIO(&vqParam);

    /* Step 3: Initialize RPMessage */
    RPMessage_Params cntrlParam;

    /* Initialize the param */
    RPMessageParams_init(&cntrlParam);

    /* Set memory for HeapMemory for control task */
    cntrlParam.buf         = pCntrlBuf;
    cntrlParam.bufSize     = rpmsgDataSize;
    cntrlParam.stackBuffer = &pTaskBuf[index++ * IPC_TASK_STACKSIZE];
    cntrlParam.stackSize   = IPC_TASK_STACKSIZE;
    RPMessage_init(&cntrlParam);

    /* Respond to messages coming in to endPt ENDPT1 */
    TaskP_Params_init(&params);
    params.priority   = 3;
    params.stack      = &pTaskBuf[index++ * IPC_TASK_STACKSIZE];
    params.stacksize  = IPC_TASK_STACKSIZE;
    params.arg0       = (void *) NULL;
    TaskP_create(&rpmsg_responderFxn, &params);

    for(t = 0; t < numProc; t++, index++)
    {
       void *            arg_array[2];

       arg_array[0] = &pRemoteProcArray[t];
       arg_array[1] = &remoteProcNum[t];

       remoteProcNum[t] = t;
       /* send messages to peer(s) on ENDPT1 */
       TaskP_Params_init(&params);
       params.priority  = 3;
       params.stack     = &pTaskBuf[index * IPC_TASK_STACKSIZE];
       params.stacksize = IPC_TASK_STACKSIZE;
       params.arg0        = (void *) arg_array;

       TaskP_create(&rpmsg_senderFxn, &params);

    }

    while(1)
    {
        printf("This example application will not exit\n");
    	sleep(10);
    }

    return 1;
}
